import 'dart:convert';

import 'package:collection/collection.dart';
import '/src/osm_elements/osm_element_type.dart';

/**
 * A base class for the three basic OSM elements [OSMNode], [OSMWay] and [OSMRelation].
 */
abstract class OSMElement {

  /**
   * A [Map] containing all OSM Tags of this element.
   *
   * Each OSM Tag contains and represents one key value pair.
   */
  final Map<String, String> tags;

  /**
   * The unique identifier of this element.
   *
   * This id is generated by the OSM Server.
   * You shouldn't set or alter the [id] on your own.
   * An [id] <= [0] is invalid. For this implementation a value of [0] will be assigned to all elements by default.
   * This indicates that the element hasn't been uploaded to the server yet.
   */
  int id;

  /**
   * The version number of this element.
   *
   * This number is generated by the OSM Server and indicates the number of changes/iterations this element has been undergone.
   * You shouldn't set or alter the [version] number on your own.
   * A [version] <= [0] is invalid. For this implementation a value of [0] will be assigned to all elements by default.
   * This indicates that the element hasn't been uploaded to the server yet.
   */
  int version;


  OSMElement({
    Map<String, String>? tags,
    int? id,
    int? version,
  }) : tags = tags ?? <String, String>{},
       id = id ?? 0,
       version = version ?? 0;


  /**
   * A getter for the type of this element.
   *
   * This returns one of the three basic OSM element types defined in [OSMElementType].
   */
  OSMElementType get type;


  /**
   * Whether this element has any children like tags.
   * For ways this additionally includes node references while relations include members.
   */
  bool get hasBody => tags.isNotEmpty;


  /**
   * A function to construct the XML [StringBuffer] of this element.
   *
   * This will serialize the entire element to XML.
   * Optionally an already existing [StringBuffer] can be passed to write the changes to.
   * Additional XML attributes can be set via the [additionalAttributes] parameter.
   * For example the `changeset` attribute can be set by passing `{'changeset': 1}`.
   */
  StringBuffer toXML({
    StringBuffer? buffer,
    Map<String, dynamic> additionalAttributes = const {},
    bool includeBody = true
  }) {
    final attributes = {
      if (id != 0) 'id': id,
      if (version != 0) 'version': version,
      ...additionalAttributes
    };

    final elementName = type.name;

    final sanitizer = const HtmlEscape(HtmlEscapeMode.attribute);
    final stringBuffer = buffer ?? StringBuffer()
    ..write('<')..write(elementName);
    for (final attributeEntry in attributes.entries) {
      // escape special XML characters
      final key = sanitizer.convert(attributeEntry.key);
      final value = sanitizer.convert(attributeEntry.value.toString());
      stringBuffer
      ..write(' ')..write(key)..write('="')..write(value)..write('"');
    }
    if (!includeBody || !hasBody) {
      stringBuffer.writeln('/>');
    }
    else {
      stringBuffer.writeln('>');
      bodyToXML(stringBuffer);
      stringBuffer..write('</')..write(elementName)..writeln('>');
    }

    return stringBuffer;
  }


  /**
   * A function to construct the XML body [StringBuffer] of this element.
   *
   * This will serialize tags and child elements of this element to XML.
   * Optionally an already existing [StringBuffer] can be passed to write the changes to.
   */
  StringBuffer bodyToXML([ StringBuffer? buffer ]) {
    final sanitizer = const HtmlEscape(HtmlEscapeMode.attribute);
    final stringBuffer = buffer ?? StringBuffer();
    tags.forEach((key, value) {
      // escape special XML characters
      key = sanitizer.convert(key);
      value = sanitizer.convert(value);

      stringBuffer
      ..write('<tag')
      ..write(' k="')..write(key)..write('"')
      ..write(' v="')..write(value)..write('"')
      ..writeln('/>');
    });
    return stringBuffer;
  }


  OSMElement copyWith({
    Map<String, String>? tags,
    int? id,
    int? version
  });


  @override
  int get hashCode =>
    id.hashCode ^
    version.hashCode ^
    // do not use nodeIds.hashCode since the hasCodes may differ even if the values are equal.
    // see https://api.flutter.dev/flutter/dart-core/Object/hashCode.html
    // "The default hash code implemented by Object represents only the identity of the object,"
    Object.hashAll(tags.keys) ^
    Object.hashAll(tags.values);


  /**
   * Elements are considered equal if their properties match.
   */
  @override
  bool operator == (o) =>
    o is OSMElement &&
    id == o.id &&
    version == o.version &&
    MapEquality().equals(tags, o.tags);
}