import 'package:xml/xml.dart';
import 'package:collection/collection.dart';
import '/src/commons/date_time_utils.dart';
import '/src/commons/bounding_box.dart';
import '/src/osm_user/osm_user.dart';
import 'osm_changeset_comment.dart';


/**
 * An immutable container class for an OSM changeset.
 */
class OSMChangeset {

  /**
   * The unique identifier of this changeset.
   *
   * This id is generated by the OSM Server.
   * You shouldn't set the [id] on your own.
   */
  final int id;

  /**
   * A [Map] containing all OSM Tags of this changeset.
   *
   * Each OSM Tag contains and represents one key value pair.
   */
  final Map<String, String> tags;

  /**
   * The date and time the changeset was created.
   */
  final DateTime createdAt;

  /**
   * The date and time the changeset was closed.
   * This will be null if the changeset is still open.
   */
  final DateTime? closedAt;

  /**
   * The user who created this changeset.
   */
  final OSMUser user;

  /**
   * The bounding box containing all the changes of this changeset.
   * If the changeset doesn't have any changes yet in other words [changesCount] is [0] this property will be null.
   */
  final BoundingBox? bbox;

  /**
   * The number of changes this changeset contains.
   */
  final int changesCount;

  /**
   * The number of comments in this changeset.
   */
  final int commentsCount;

  /**
   * An optional [List] of [OSMChangesetComment]s containing the discussion of this changeset.
   *
   * The [discussion] property is null if the discussion wasn't requested from the server, otherwise it's a [List] of zero ore more items.
   */
  final List<OSMChangesetComment>? discussion;

  /**
   * A short version for checking whether this chageset is still open.
   */
  bool get isOpen => closedAt == null;


  /**
   * A short version for checking whether this chageset has been closed.
   */
  bool get isClosed => closedAt != null;


  const OSMChangeset({
    required this.id,
    required this.tags,
    required this.createdAt,
    this.closedAt,
    required this.user,
    this.bbox,
    required this.changesCount,
    required this.commentsCount,
    this.discussion
  });


  /**
   * A factory method for constructing an [OSMChangeset] from an XML [String].
   */
  factory OSMChangeset.fromXMLString(String xmlString) {
    final xmlDoc = XmlDocument.parse(xmlString);
    final changesetElement = xmlDoc.findAllElements('changeset').first;
    return OSMChangeset.fromXMLElement(changesetElement);
  }


  /**
   * A factory method for constructing an [OSMChangeset] object from an [XmlElement].
   */
  factory OSMChangeset.fromXMLElement(XmlElement changesetElement) {
    final int id;
    final tags = <String, String>{};
    final DateTime createdAt;
    DateTime? closedAt;
    final OSMUser user;
    final int changesCount;
    final int commentsCount;
    BoundingBox? bbox;
    List<OSMChangesetComment>? comments;

    // try parsing the xml attributes
    try {
      final idValue = changesetElement.getAttribute('id');
      id = int.parse(idValue!);

      final createdAtValue = changesetElement.getAttribute('created_at');
      createdAt = parseUTCDate(createdAtValue!);

      final closedAtValue = changesetElement.getAttribute('closed_at');
      if (closedAtValue != null) {
        closedAt = parseUTCDate(closedAtValue);
      }

      final userName = changesetElement.getAttribute('user')!;
      final uidValue = changesetElement.getAttribute('uid');
      final uid = int.parse(uidValue!);
      user = OSMUser(
        id: uid,
        name: userName,
      );

      final changesCountValue = changesetElement.getAttribute('changes_count');
      changesCount = int.parse(changesCountValue!);

      final commentsCountValue = changesetElement.getAttribute('comments_count');
      commentsCount = int.parse(commentsCountValue!);

      final minLatValue = changesetElement.getAttribute('min_lat');
      final maxLatValue = changesetElement.getAttribute('max_lat');
      final minLonValue = changesetElement.getAttribute('min_lon');
      final maxLonValue = changesetElement.getAttribute('max_lon');

      if (minLatValue != null && maxLatValue != null && minLonValue != null && maxLonValue != null) {
        bbox = BoundingBox(
          double.parse(minLonValue),
          double.parse(minLatValue),
          double.parse(maxLonValue),
          double.parse(maxLatValue)
        );
      }
    }
    catch (e) {
      throw('Could not parse the given changeset XML string.');
    }

    changesetElement.findElements('tag').forEach((tag) {
      final key = tag.getAttribute('k');
      final value = tag.getAttribute('v');
      if (key != null && value != null) {
        tags[key] = value;
      }
    });

    final discussionElement = changesetElement.getElement('discussion');
    if (discussionElement != null) {
      comments = [];

      discussionElement.childElements.forEach((comment) {
        final date = comment.getAttribute('date');
        final uid = comment.getAttribute('uid');
        final userName = comment.getAttribute('user');
        final textElement = comment.getElement('text');

        if (date != null && uid != null && userName != null && textElement != null) {
          comments!.add(OSMChangesetComment(
            user: OSMUser(
              id: int.parse(uid),
              name: userName,
            ),
            date: parseUTCDate(date),
            text: textElement.text
          ));
        }
      });
    }

    return OSMChangeset(
      id: id,
      tags: tags,
      createdAt: createdAt,
      closedAt: closedAt,
      user: user,
      changesCount: changesCount,
      commentsCount: commentsCount,
      bbox: bbox,
      discussion: comments
    );
  }


  @override
  String toString() =>
    '$runtimeType - id: $id; '
    'tags: $tags; '
    'createdAt: $createdAt; '
    'closedAt: $closedAt; '
    'user: $user; '
    'bbox: $bbox; '
    'changesCount: $changesCount; '
    'commentsCount: $commentsCount; '
    'discussion: $discussion';


  @override
  int get hashCode =>
    id.hashCode ^
    createdAt.hashCode ^
    closedAt.hashCode ^
    user.hashCode ^
    bbox.hashCode ^
    changesCount.hashCode ^
    commentsCount.hashCode ^
    // do not use COLLECTION.hashCode since the hasCodes may differ even if the values are equal.
    // see https://api.flutter.dev/flutter/dart-core/Object/hashCode.html
    // "The default hash code implemented by Object represents only the identity of the object,"
    Object.hashAll(tags.keys) ^
    Object.hashAll(tags.values) ^
    (discussion != null ? Object.hashAll(discussion!) : discussion.hashCode);


  @override
  bool operator == (o) =>
    identical(this, o) ||
    o is OSMChangeset &&
    runtimeType == o.runtimeType &&
    id == o.id &&
    createdAt == o.createdAt &&
    closedAt == o.closedAt &&
    user == o.user &&
    bbox == o.bbox &&
    changesCount == o.changesCount &&
    commentsCount == o.commentsCount &&
    MapEquality().equals(tags, o.tags) &&
    ListEquality().equals(discussion, o.discussion);
}