import 'dart:io';

import 'package:analyzer/dart/analysis/utilities.dart';
import 'package:analyzer/dart/ast/ast.dart' hide Declaration;
import 'package:dart_style/dart_style.dart';
import 'package:ffigen/ffigen.dart';
import 'package:ffigen/src/config_provider/config_impl.dart';
import 'package:ffigen/src/config_provider/config_types.dart';
import 'package:ffigen/src/config_provider/spec_utils.dart';
import 'package:ffigen/src/code_generator.dart';
import 'package:pub_semver/pub_semver.dart';

void main() {
  final generator = FfiGen();
  generator.run(_GenerationMode.shared.resolveConfig());
  generator.run(_GenerationMode.native.resolveConfig());
  generator.run(_GenerationMode.dynamicLibrary.resolveConfig());

  postProcess();
}

void postProcess() {
  final formatter = DartFormatter(languageVersion: Version(3, 6, 0));

  // Generate a shared interface for the `@Native` and `DynamicLibrary.lookup`
  // bindings.
  final sharedLibrary = StringBuffer()
    ..writeln()
    ..writeln('abstract interface class SqliteLibrary {');
  final nativeImplementation = StringBuffer()
    ..writeln('// Generated by tool/generate_bindings.dart')
    ..writeln('// ignore_for_file: type=lint')
    ..writeln("import 'dart:ffi' as ffi;")
    ..writeln("import 'native.dart' as native;")
    ..writeln("import 'shared.dart';")
    ..writeln('final class NativeAssetsLibrary implements SqliteLibrary {');

  final dynamicBindingsFile =
      File('lib/src/ffi/generated/dynamic_library.dart');
  final dynamicBindingsContents = dynamicBindingsFile.readAsStringSync();

  String patchSource(String source) {
    // The generated bindings use `imp1.` as a prefix to reference members in
    // the shared Dart file. We're appending to that file though, so no need to
    // include that.
    return source.replaceAll('imp1.', '');
  }

  final generatedBindings = parseString(content: dynamicBindingsContents).unit;
  // Generated `class nativeLibrary
  final nativeLibrary =
      generatedBindings.childEntities.whereType<ClassDeclaration>().first;
  for (final entry in nativeLibrary.members) {
    if (entry is MethodDeclaration) {
      void writeDefinition(StringBuffer buffer) {
        if (entry.isGetter) {
          buffer
            ..write(patchSource(entry.returnType!.toSource()))
            ..write(' get ')
            ..write(entry.name.lexeme);
        } else if (entry.isSetter) {
          buffer
            ..write('set ')
            ..write(entry.name.lexeme)
            ..write(patchSource(entry.parameters!.toSource()));
        } else {
          buffer
            ..write(patchSource(entry.returnType!.toSource()))
            ..write(' ')
            ..write(entry.name.lexeme)
            ..write(patchSource(entry.parameters!.toSource()));
        }
      }

      writeDefinition(sharedLibrary);
      sharedLibrary.writeln(';');

      nativeImplementation.writeln('@override');
      writeDefinition(nativeImplementation);
      nativeImplementation.writeln('{');
      if (!entry.isSetter) {
        nativeImplementation.write('return ');
      }
      nativeImplementation
        ..write('native.')
        ..write(entry.name.lexeme);

      if (entry.isSetter) {
        nativeImplementation
            .write(' = ${entry.parameters!.parameters.single.name!.lexeme}');
      } else if (!entry.isGetter) {
        final forwardedArgs =
            entry.parameters!.parameters.map((e) => e.name!.lexeme).join(', ');
        nativeImplementation.write('($forwardedArgs)');
      }

      nativeImplementation.writeln(';}');
    }
  }

  sharedLibrary.writeln('}');
  nativeImplementation.writeln('}');

  final sharedFile = File('lib/src/ffi/generated/shared.dart');
  sharedFile.writeAsStringSync(
      formatter.format('${sharedFile.readAsStringSync()}$sharedLibrary'));

  dynamicBindingsFile.writeAsStringSync(dynamicBindingsContents.replaceFirst(
      'class NativeLibrary',
      'final class NativeLibrary implements imp1.SqliteLibrary'));

  File('lib/src/ffi/generated/native_library.dart')
      .writeAsStringSync(formatter.format(nativeImplementation.toString()));
}

enum _GenerationMode {
  dynamicLibrary('dynamic_library.dart'),
  native('native.dart'),
  shared('shared.dart');

  final String output;

  const _GenerationMode(this.output);

  Uri get outputDartUri => Uri.parse('lib/src/ffi/generated/').resolve(output);

  Config resolveConfig() {
    final libraryImports = <String, LibraryImport>{};
    Map<String, ImportedType> typeMappings = <String, ImportedType>{};

    if (this != shared) {
      typeMappings = symbolFileImportExtractor(
          [symbolUri.toString()], libraryImports, null, null);
    }

    final config = Config(
      preamble: '// ignore_for_file: type=lint',
      ffiNativeConfig: FfiNativeConfig(
          enabled: this == native,
          assetId: 'package:sqlite3_native_assets/sqlite3_native_assets.dart'),
      output: outputDartUri,
      entryPoints: [Uri.parse('assets/sqlite3.h')],
      symbolFile: this == shared
          ? SymbolFile(Uri.parse('shared.dart'), symbolUri)
          : null,
      structDecl: _includeSqlite3Only,
      functionDecl: this == shared ? _includeNothing : _includeSqlite3Only,
      globals: this == shared ? _includeNothing : _includeSqlite3Only,
      varArgFunctions: makeVarArgFunctionsMapping({
        'sqlite3_db_config': [
          RawVarArgFunction('', ['int', 'int*'])
        ],
      }, libraryImports),
    ) as ConfigImpl;

    // Adding these throug the constructor unmangles their keys, which causes
    // the header parser to no longer recognize them.
    config.libraryImports.addAll(libraryImports);
    config.usrTypeMappings.addAll(typeMappings);
    return config;
  }

  static Uri symbolUri = Uri.parse('lib/src/ffi/generated/shared_symbols.yml');

  static DeclarationFilters _includeSqlite3Only =
      DeclarationFilters(shouldInclude: (d) => d.isSqlite3Symbol);

  static DeclarationFilters _includeNothing =
      DeclarationFilters(shouldInclude: (d) => false);
}

extension on Declaration {
  bool get isSqlite3Symbol => originalName.startsWith('sqlite3');
}
