import 'dart:io';

import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:integration_test/integration_test.dart';
import 'package:ntodotxt/database/controller/database.dart';
import 'package:ntodotxt/drawer/state/drawer_cubit.dart';
import 'package:ntodotxt/filter/controller/filter_controller.dart'
    show FilterController;
import 'package:ntodotxt/filter/model/filter_model.dart'
    show Filter, ListFilter, ListGroup, ListOrder;
import 'package:ntodotxt/filter/repository/filter_repository.dart';
import 'package:ntodotxt/filter/state/filter_cubit.dart';
import 'package:ntodotxt/filter/state/filter_list_bloc.dart';
import 'package:ntodotxt/filter/state/filter_list_event.dart';
import 'package:ntodotxt/login/state/login_cubit.dart';
import 'package:ntodotxt/login/state/login_state.dart'
    show LoginLocal, LoginState, LoginWebDAV;
import 'package:ntodotxt/main.dart';
import 'package:ntodotxt/setting/controller/setting_controller.dart';
import 'package:ntodotxt/setting/repository/setting_repository.dart';
import 'package:ntodotxt/todo/model/todo_model.dart' show Priority, Todo;
import 'package:ntodotxt/todo_file/state/todo_file_cubit.dart';
import 'package:ntodotxt/webdav/client/webdav_client.dart';
import 'package:path_provider/path_provider.dart';
import 'package:sqflite_common_ffi/sqflite_ffi.dart';

// https://developer.android.com/studio/run/emulator-networking#networkaddresses
// Special alias to your host loopback interface (127.0.0.1 on your development machine)
const String server = 'https://10.0.2.2:8443';
const String path = '/remote.php/dav/files';
const String username = 'test';
const String password = 'test';

class FakeController extends Fake implements FilterController {
  List<Filter> items = [
    const Filter(
      id: 1,
      name: 'Agenda',
      order: ListOrder.ascending,
      filter: ListFilter.incompletedOnly,
      group: ListGroup.upcoming,
    ),
    const Filter(
      id: 2,
      name: 'Highly prioritized',
      order: ListOrder.ascending,
      filter: ListFilter.incompletedOnly,
      group: ListGroup.project,
      priorities: {Priority.A},
    ),
    const Filter(
      id: 3,
      name: 'Projectideas',
      order: ListOrder.ascending,
      filter: ListFilter.incompletedOnly,
      group: ListGroup.none,
      projects: {'projectideas'},
    ),
    const Filter(
      id: 4,
      name: 'Completed only',
      order: ListOrder.ascending,
      filter: ListFilter.completedOnly,
      group: ListGroup.none,
    ),
  ];

  @override
  Future<List<Filter>> list() async {
    return Future.value(items);
  }
}

class AppTester extends StatelessWidget {
  final DatabaseController dbController =
      const DatabaseController(inMemoryDatabasePath);
  final ThemeMode? themeMode;
  final String appCacheDir;

  const AppTester({
    this.themeMode,
    required this.appCacheDir,
    super.key,
  });

  @override
  Widget build(BuildContext context) {
    return MultiRepositoryProvider(
      providers: [
        RepositoryProvider<SettingRepository>(
          create: (BuildContext context) => SettingRepository(
            SettingController(dbController),
          ),
        ),
        RepositoryProvider<FilterRepository>(
          create: (BuildContext context) => FilterRepository(
            FakeController(),
          ),
        ),
      ],
      child: MultiBlocProvider(
        providers: [
          BlocProvider<LoginCubit>(
            create: (BuildContext context) => LoginCubit(
              state: const LoginWebDAV(
                server: server,
                path: path,
                username: username,
                password: password,
                acceptUntrustedCert: true,
              ),
            ),
          ),
          BlocProvider<TodoFileCubit>(
            create: (BuildContext context) => TodoFileCubit(
              repository: context.read<SettingRepository>(),
              localPath: appCacheDir,
            )..load(),
          ),
          BlocProvider<DrawerCubit>(
            create: (BuildContext context) => DrawerCubit(),
          ),
          // Default filter
          BlocProvider<FilterCubit>(
            create: (BuildContext context) => FilterCubit(
              settingRepository: context.read<SettingRepository>(),
              filterRepository: context.read<FilterRepository>(),
            )..load(),
          ),
          BlocProvider<FilterListBloc>(
            create: (BuildContext context) => FilterListBloc(
              repository: context.read<FilterRepository>(),
            )
              ..add(const FilterListSubscriped())
              ..add(const FilterListSynchronizationRequested()),
          ),
        ],
        child: Builder(
          builder: (BuildContext context) {
            return BlocBuilder<LoginCubit, LoginState>(
              builder: (BuildContext context, LoginState state) {
                if (state is LoginLocal || state is LoginWebDAV) {
                  return CoreApp(loginState: state);
                } else {
                  return const InitialApp();
                }
              },
            );
          },
        ),
      ),
    );
  }
}

void main() async {
  final IntegrationTestWidgetsFlutterBinding binding =
      IntegrationTestWidgetsFlutterBinding.ensureInitialized();
  SystemChrome.setEnabledSystemUIMode(SystemUiMode.edgeToEdge);

  final DateTime today = DateTime.now();
  final List<Todo> todoList = [
    Todo(
      creationDate: today.subtract(const Duration(days: 7)),
      priority: Priority.A,
      description:
          'Automate the generation of +app screenshots +learnflutter @development @automation @productivity due:${Todo.date2Str(today.add(const Duration(days: 3)))!}',
    ),
    Todo(
      creationDate: today.subtract(const Duration(days: 14)),
      priority: Priority.B,
      description:
          'Publish this +app +learnflutter @development due:${Todo.date2Str(today.add(const Duration(days: 7)))!}',
    ),
    Todo(
      creationDate: today.subtract(const Duration(days: 2)),
      description:
          'Increase test +coverage for this +app +learnflutter @development @testing @productivity',
    ),
    Todo(
      creationDate: today.subtract(const Duration(days: 2)),
      completion: true,
      completionDate: today.subtract(const Duration(days: 1)),
      description:
          'Write some tests for this +app +learnflutter @development @testing @productivity',
    ),
    Todo(
      creationDate: today.subtract(const Duration(days: 21)),
      priority: Priority.C,
      description:
          'Setup a good project management tool @development @productivity',
    )
  ];

  setUp(() async {
    // Setup todos.
    WebDAVClient client = WebDAVClient(
      server: server,
      path: path,
      username: username,
      password: password,
    );
    try {
      await client.upload(
          content: todoList.join(Platform.lineTerminator),
          filename: 'todo.txt');
    } catch (e) {
      fail('An exception was thrown: $e');
    }
  });

  group('dark mode', () {
    group('take screenshots', () {
      testWidgets('of todo list (default)', (tester) async {
        await tester.pumpWidget(
          AppTester(
            themeMode: ThemeMode.dark,
            appCacheDir: (await getApplicationCacheDirectory()).path,
          ),
        );
        await tester.pumpAndSettle(const Duration(milliseconds: 5000));

        await binding.convertFlutterSurfaceToImage();
        await tester.pumpAndSettle();
        await binding.takeScreenshot('phone/1');
      });
      testWidgets('of todo list (with open drawer)', (tester) async {
        await tester.pumpWidget(
          AppTester(
            themeMode: ThemeMode.dark,
            appCacheDir: (await getApplicationCacheDirectory()).path,
          ),
        );
        await tester.pumpAndSettle(const Duration(milliseconds: 5000));

        await tester.tap(find.byTooltip('Open drawer'));
        await tester.pumpAndSettle();

        await tester.drag(
            find.byType(DraggableScrollableSheet), const Offset(0, -500));
        await tester.pumpAndSettle();

        await binding.convertFlutterSurfaceToImage();
        await tester.pumpAndSettle();
        await binding.takeScreenshot('phone/2');
      });
      testWidgets('of todo edit page', (tester) async {
        await tester.pumpWidget(
          AppTester(
            themeMode: ThemeMode.dark,
            appCacheDir: (await getApplicationCacheDirectory()).path,
          ),
        );
        await tester.pumpAndSettle(const Duration(milliseconds: 5000));

        await tester.tap(find.text(
          'Publish this +app +learnflutter @development',
          findRichText: true,
        ));
        await tester.pumpAndSettle();

        await binding.convertFlutterSurfaceToImage();
        await tester.pumpAndSettle();
        await binding.takeScreenshot('phone/3');
      });
      testWidgets('of filter list (default)', (tester) async {
        await tester.pumpWidget(
          AppTester(
            themeMode: ThemeMode.dark,
            appCacheDir: (await getApplicationCacheDirectory()).path,
          ),
        );
        await tester.pumpAndSettle(const Duration(milliseconds: 5000));

        await tester.tap(find.byTooltip('Open drawer'));
        await tester.pumpAndSettle();

        await tester.drag(
            find.byType(DraggableScrollableSheet), const Offset(0, -500));
        await tester.pumpAndSettle();

        await tester.tap(find.text('Filters'));
        await tester.pumpAndSettle();

        await binding.convertFlutterSurfaceToImage();
        await tester.pumpAndSettle();
        await binding.takeScreenshot('phone/4');
      });
      testWidgets('of filter edit page', (tester) async {
        await tester.pumpWidget(
          AppTester(
            themeMode: ThemeMode.dark,
            appCacheDir: (await getApplicationCacheDirectory()).path,
          ),
        );
        await tester.pumpAndSettle(const Duration(milliseconds: 5000));

        await tester.tap(find.byTooltip('Open drawer'));
        await tester.pumpAndSettle();

        await tester.drag(
            find.byType(DraggableScrollableSheet), const Offset(0, -500));
        await tester.pumpAndSettle();

        await tester.tap(find.text('Filters'));
        await tester.pumpAndSettle();

        await tester.tap(find.text('Projectideas'));
        await tester.pumpAndSettle();

        await binding.convertFlutterSurfaceToImage();
        await tester.pumpAndSettle();
        await binding.takeScreenshot('phone/5');
      });
    });
  });
}
