package de.taz.app.android.api

import android.content.Context
import androidx.annotation.VisibleForTesting
import de.taz.app.android.BuildConfig
import de.taz.app.android.MAX_SIMULTANEOUS_QUERIES
import de.taz.app.android.TAZ_AUTH_HEADER
import de.taz.app.android.api.dto.DataDto
import de.taz.app.android.api.dto.WrapperDto
import de.taz.app.android.api.mappers.AuthInfoMapper
import de.taz.app.android.api.variables.Variables
import de.taz.app.android.data.HTTP_CLIENT_ENGINE
import de.taz.app.android.singletons.AuthHelper
import de.taz.app.android.util.Json
import de.taz.app.android.util.SingletonHolder
import de.taz.app.android.util.reportAndRethrowExceptions
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.http.ContentType
import io.ktor.http.Url
import io.ktor.http.contentType
import io.ktor.serialization.JsonConvertException
import io.ktor.serialization.kotlinx.json.json
import io.ktor.serialization.kotlinx.serialization
import kotlinx.coroutines.sync.Semaphore
import kotlinx.coroutines.sync.withPermit

/**
 * class to get DTOs from the [BuildConfig.GRAPHQL_ENDPOINT]
 */
class GraphQlClient @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE) constructor(
    private val httpClient: HttpClient = HttpClient(HTTP_CLIENT_ENGINE) {
        install(ContentNegotiation) {
            json(Json)
            serialization(ContentType.Any, Json)
        }
    },
    private val url: String,
    private val queryService: QueryService,
    private val authHelper: AuthHelper
) {
    private constructor(applicationContext: Context) : this(
        url = BuildConfig.GRAPHQL_ENDPOINT_PREFIX + BuildConfig.GRAPHQL_ENDPOINT,
        queryService = QueryService.getInstance(applicationContext),
        authHelper = AuthHelper.getInstance(applicationContext)
    )

    private val unrecoverableGraphQlErrorCategories = listOf("businessLogic", "graphql ")

    companion object : SingletonHolder<GraphQlClient, Context>(::GraphQlClient)

    private val maxSimultaneousRequestSemaphore = Semaphore(MAX_SIMULTANEOUS_QUERIES)

    /**
     * function to get DTO from query
     * @param queryType - the type of the query to execute
     * @param variables - the variables to set on query
     * @return the [DataDto] generated by parsing the returned json
     */
    @Throws(
        MalformedServerResponseException::class,
        GraphQlImplementationException::class,
        GraphQlRecoverableServerException::class
    )
    suspend fun query(queryType: QueryType, variables: Variables? = null): WrapperDto {
        val query = queryService.get(queryType)
        variables?.let { query.variables = variables }

        val wrapper: WrapperDto = try {
            maxSimultaneousRequestSemaphore.withPermit {
                httpClient.post(Url(url)) {
                    contentType(ContentType.Application.Json)
                    setBody(query)
                    val token = authHelper.token.get()
                    if (token.isNotEmpty()
                        && (authHelper.isLoggedIn())
                    ) {
                        header(TAZ_AUTH_HEADER, token)
                    }
                }.body()
            }
        } catch (e: NullPointerException) {
            reportAndRethrowExceptions {
                throw MalformedServerResponseException(e)
            }
        } catch (e: JsonConvertException) {
            reportAndRethrowExceptions {
                throw MalformedServerResponseException(e)
            }
        }

        if (wrapper.errors.isNotEmpty()) {
            if (wrapper.errors.findLast { item ->
                    unrecoverableGraphQlErrorCategories.contains(item.extensions?.category)
                } != null) {
                log.error("A faulty response from graphQL received ${wrapper.errors}")
                throw GraphQlImplementationException(wrapper)
            } else {
                throw GraphQlRecoverableServerException(wrapper)
            }
        }

        // if response carries authinfo we save it
        wrapper.data?.product?.authInfo?.let {
            val authInfo = AuthInfoMapper.from(it)
            // only update if it changes
            if (authHelper.status.get() != authInfo.status) {
                authHelper.status.set(authInfo.status)
                authHelper.elapsedDateMessage.set(authInfo.message ?: "")
            }
        }
        return wrapper
    }

    class MalformedServerResponseException(cause: Throwable? = null) :
        Exception("GraphQL server returned unexpected response", cause)

    class GraphQlImplementationException(wrapperDto: WrapperDto?) :
        Exception("An unrecoverable GraphQL exception occurred: $wrapperDto")

    class GraphQlRecoverableServerException(wrapperDto: WrapperDto?) :
        Exception("A probably recoverable error occurred: $wrapperDto")
}
