package de.tadris.flang_lib.script

import java.io.File

/**
 * Sealed class hierarchy for different parameter types in the codebase.
 */
sealed class Parameter {
    abstract val name: String
    abstract val filePath: String
    abstract fun getCurrentValue(): Double
    abstract fun setValue(newValue: Double)

    /**
     * Parameter defined as #define macro in a header file.
     */
    data class DefineParameter(
        override val name: String,
        override val filePath: String,
        val defineName: String
    ) : Parameter() {
        override fun getCurrentValue(): Double {
            return readDefineValue(File(filePath), defineName)
        }

        override fun setValue(newValue: Double) {
            writeDefineValue(File(filePath), defineName, newValue)
        }
    }
}

/**
 * Reads a #define value from a header file.
 */
fun readDefineValue(file: File, defineName: String): Double {
    val pattern = Regex("""^\s*#define\s+$defineName\s+([\d.eE+-]+)""")
    file.readLines().forEach { line ->
        val match = pattern.find(line)
        if (match != null) {
            return match.groupValues[1].toDouble()
        }
    }
    error("Could not find #define $defineName in ${file.path}")
}

/**
 * Writes a #define value to a header file.
 * Uses atomic write operation (write to temp, then rename).
 */
fun writeDefineValue(file: File, defineName: String, newValue: Double) {
    if (!file.exists()) {
        error("File does not exist: ${file.path}")
    }

    // Match the #define line more flexibly - match any non-comment content after the define name
    val pattern = Regex("""^(\s*#define\s+$defineName\s+)(.*)$""")
    val lines = file.readLines()

    // Format the value properly - ensure at least one decimal place
    val formattedValue = if (newValue == newValue.toInt().toDouble()) {
        "${newValue.toInt()}.0"
    } else {
        newValue.toString()
    }

    var modified = false
    val newLines = lines.map { line ->
        val match = pattern.find(line)
        if (match != null) {
            modified = true
            // Keep the prefix, replace everything after it with the new value
            "${match.groupValues[1]}$formattedValue"
        } else {
            line
        }
    }

    // Verify that we actually modified something
    if (!modified) {
        // Debug: print lines that contain the define name
        val debugLines = lines.filter { it.contains(defineName) }
        val debugMsg = if (debugLines.isEmpty()) {
            "No lines found containing '$defineName'"
        } else {
            "Found ${debugLines.size} line(s) containing '$defineName':\n" +
            debugLines.joinToString("\n") { "  > $it" }
        }
        error("Could not find #define $defineName in ${file.path}\n$debugMsg")
    }

    // Atomic write
    val tempFile = File(file.parentFile, file.name + ".tmp")
    tempFile.writeText(newLines.joinToString("\n") + "\n")
    tempFile.renameTo(file)
}

/**
 * Creates a backup of a file with .bak extension.
 */
fun backupFile(filePath: String): File {
    val file = File(filePath)
    val backupFile = File(filePath + ".bak")
    file.copyTo(backupFile, overwrite = true)
    return backupFile
}

/**
 * Restores a file from its backup.
 */
fun restoreFromBackup(filePath: String, backupFile: File) {
    backupFile.copyTo(File(filePath), overwrite = true)
    backupFile.delete()
}
