#ifndef FAST_EVALUATION_H
#define FAST_EVALUATION_H

#include "fast_board.h"
#include "fast_move_generator.h"
#include "bitboard_utils.h"
#include <math.h>

// Evaluation constants
#define MATE_EVAL 10000.0
#define MATE_STEP_LOSS 1.0

// Piece values for evaluation
#define PIECE_VALUE_PAWN 110
#define PIECE_VALUE_HORSE 200
#define PIECE_VALUE_ROOK 400
#define PIECE_VALUE_FLANGER 400
#define PIECE_VALUE_UNI 900
#define PIECE_VALUE_KING 400

// Evaluation hyper parameters
#define EVAL_WEIGHT_MATRIX 2.0
#define EVAL_WEIGHT_MOVEMENT 10.0
#define EVAL_WEIGHT_PIECE_VALUE 140.0
#define EVAL_WEIGHT_KINGS_EVAL 2.0
#define EVAL_KINGS_WEIGHT_MULTIPLIER 0.6

// Location evaluation structure
typedef struct {
    int16_t occupied_by;
    uint16_t white_control;
    uint16_t black_control;
    uint16_t weight;
} LocationEvaluation;

// Overall statistics for each color
typedef struct {
    int movements;
    int piece_value;
} OverallStats;

// Fast board evaluation structure
typedef struct {
    FastBoard* board;
    FastMoveGenerator move_generator;  // Own move generator for evaluation
    LocationEvaluation evaluation_matrix[ARRAY_SIZE];
    OverallStats white_stats;
    OverallStats black_stats;
} FastBoardEvaluation;

// Evaluation functions
void fast_evaluation_init(FastBoardEvaluation* eval, FastBoard* board);
double fast_evaluation_evaluate(FastBoardEvaluation* eval);
void fast_evaluation_prepare(FastBoardEvaluation* eval);
void fast_evaluation_evaluate_location(FastBoardEvaluation* eval, FastBoardIndex index);
double fast_evaluation_get_kings_eval(FastBoardEvaluation* eval);

// Location evaluation functions
void location_evaluation_add_threat(LocationEvaluation* loc_eval, FastColor color, int threat);
double location_evaluation_evaluate_field(const LocationEvaluation* loc_eval);

// Statistics functions
void overall_stats_reset(OverallStats* stats);
OverallStats* get_stats(FastBoardEvaluation* eval, FastColor color);

#endif // FAST_EVALUATION_H
