#ifndef FAST_BOT_H
#define FAST_BOT_H

#include "fast_board.h"
#include "fast_evaluation.h"
#include "fast_move_generator.h"
#include "fast_transposition.h"
#include <stdbool.h>
#include <stddef.h>
#include <pthread.h>
#include <stdatomic.h>
#include <stdbool.h>

// Search constants
#define QUIESCENCE_DEPTH 5
#define NULL_WINDOW_SIZE 0.01  // Size of null window for PVS (tune for performance)

// Move evaluation structure
typedef struct {
    FastMove move;
    double evaluation;
    int depth;
} MoveEvaluation;

// Bot result structure
typedef struct {
    MoveEvaluation best_move;
    MoveEvaluation* all_evaluations;
    int evaluation_count;
    long total_evaluations;
} BotResult;

// Transposition table types moved to fast_transposition.h

// Fast Flang Bot structure
typedef struct {
    int min_depth;
    int max_depth;
    bool use_opening_database;
    int threads;
    bool use_lme;
    int lme_max_extension;
    bool only_best_move;  // If true, only search for best move (skip all move evaluations)

    TranspositionTable tt;
    FastBoardEvaluation evaluator;
    FastMoveGenerator move_generator;
    long total_evaluations;
} FastFlangBot;

typedef struct {
    FastFlangBot* bot;
    const FastBoard* root_board;
    FastMove* root_moves;
    int root_count;
    int root_depth;      // current iterative depth
    FastColor side_to_move;

    atomic_int next_idx; // work queue
    atomic_bool stop;    // time cutoff

    pthread_mutex_t best_lock;
    FastMove best_move;
    double   best_eval;
    
    // Storage for all move evaluations
    MoveEvaluation* all_evaluations;
} RootSearchCtx;

// Bot functions
void fast_bot_init(FastFlangBot* bot, int min_depth, int max_depth, int threads, int ttsize_mb, bool use_lme, int lme_max_extension, bool only_best_move);
void fast_bot_destroy(FastFlangBot* bot);
BotResult fast_bot_find_best_move(FastFlangBot* bot, FastBoard* board, bool print_time);
BotResult fast_bot_find_best_move_iterative(FastFlangBot* bot, FastBoard* board, bool print_time, long max_time_ms);

// Search functions
double fast_bot_evaluate_move(FastFlangBot* bot, FastBoard* board, uint8_t depth, double alpha, double beta, int ply);
double fast_bot_quiescence(FastFlangBot* bot, FastBoard* board, double alpha, double beta, int depth);

// Transposition table functions moved to fast_transposition.h


// Helper functions
bool is_mate_score(double value);
FastColor get_color_at_move(const FastBoard* board);

#endif // FAST_BOT_H