package de.tadris.flang.network_api.script

import de.tadris.flang.network_api.FlangAPI
import de.tadris.flang.network_api.util.Sha256
import de.tadris.flang_lib.Board
import de.tadris.flang_lib.analysis.AnalysisListener
import de.tadris.flang_lib.analysis.ComputerAnalysis
import de.tadris.flang_lib.analysis.MoveInfo
import de.tadris.flang_lib.bot.CFlangEngine
import de.tadris.flang_lib.bot.CFlangSocketEngine
import de.tadris.flang_lib.bot.Engine
import de.tadris.flang_lib.bot.fast.FastFlangBot
import de.tadris.flang_lib.puzzle.PuzzleGenerator
import java.io.File
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit
import kotlin.math.min
import kotlin.math.roundToInt

fun main(){
    val fmns = File("doc/games2.txt").readText()
    val credentials = File("doc/auth.txt").readText().split(":")

    val searchDepth = 6
    val reviewDepth = 9

    val games = fmns.lines().subList(12646, 13000).map { it.trim() }.filter { it.isNotEmpty() }.map { Board.fromFMN(it).getFMN2() }
    val totalStart = System.currentTimeMillis()
    var totalPuzzles = 0

    val api = FlangAPI("www.tadris.de", 443, "api/flang", useSSL = true, loggingEnabled = true)
    api.login(credentials[0], credentials[1])

    val executor = Executors.newFixedThreadPool(12)

    games.forEachIndexed { index, fmn ->
        executor.submit {
            val start = System.currentTimeMillis()
            val engine = CFlangEngine(minDepth = searchDepth, maxDepth = reviewDepth, ttSizeMB = 512, threads = 1)
            val searcher = object : Engine{
                override fun findBestMove(
                    board: Board,
                    printTime: Boolean
                ) = engine.findBestMoveIterative(board, printTime, 3000)
            }
            val depths = mutableListOf<Int>()
            val analysis = ComputerAnalysis(fmn, searcher, object : AnalysisListener {
                override fun onMoveAnalyzed(currentMove: Int, totalMoves: Int, moveInfo: MoveInfo?) {
                    moveInfo?.depth?.let { depths += it.coerceAtMost(reviewDepth) }
                }
            })
            analysis.analyze()
            val puzzleGenerator = PuzzleGenerator(analysis.getFullEvaluationDataAfterAnalysis(), engine)
            val puzzles = puzzleGenerator.searchPuzzles()

            puzzles.forEach { puzzle ->
                api.injectPuzzle(puzzle.startFMN, puzzle.puzzleFMN)
                totalPuzzles++
            }

            val avgDepth = depths.sum() / depths.size.toDouble()
            val time = System.currentTimeMillis() - start
            val moves = analysis.getFullEvaluationDataAfterAnalysis().evaluations.size
            val timePerMove = time / moves
            println("Processed game ${index + 1}/${games.size}. Time: ${time/1000}s | Moves: $moves | Time per move: $timePerMove ms | Avg. depth: $avgDepth")
        }
    }

    executor.shutdown()
    executor.awaitTermination(1, TimeUnit.DAYS)

    val totalTime = System.currentTimeMillis() - totalStart
    println("-------------------")
    println("TOTAL TIME: " + (totalTime.toDouble() / 1000 / 60).roundToInt() + " min")
    println("TOTAL PUZZLES: $totalPuzzles")
    println("-------------------")
}

private fun generateRandomGames(count: Int, depth: Int): List<String> {
    val engine = FastFlangBot(min(5, depth), depth, ttSizeMB = 1024)
    return buildList {
        repeat(count){
            val board = Board()
            repeat(5){
                val randomMove = board.findAllMoves(board.atMove).random()
                board.executeOnBoard(randomMove)
            }
            while (!board.gameIsComplete()){
                val move = engine.findBestMove(board, printTime = false)
                board.executeOnBoard(move.bestMove.move)
            }
            val fmn = board.getFMN2()
            println("Generated $fmn")
            add(fmn)
        }
    }
}