package de.tadris.flang

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.Location
import de.tadris.flang_lib.Piece
import de.tadris.flang_lib.PieceState
import de.tadris.flang_lib.Type
import de.tadris.flang_lib.Vector
import de.tadris.flang_lib.action.Move
import de.tadris.flang_lib.fast.*
import org.junit.Assert.*
import org.junit.Test

class TestFastBoardExtensions {

    @Test
    fun testTypeConversion() {
        // Test Type to FastType conversion
        assertEquals(FAST_KING, Type.KING.fast())
        assertEquals(FAST_PAWN, Type.PAWN.fast())
        assertEquals(FAST_HORSE, Type.HORSE.fast())
        assertEquals(FAST_ROOK, Type.ROOK.fast())
        assertEquals(FAST_UNI, Type.UNI.fast())
        assertEquals(FAST_FLANGER, Type.FLANGER.fast())
        
        // Test FastType to Type conversion
        assertEquals(Type.KING, FAST_KING.decodeType())
        assertEquals(Type.PAWN, FAST_PAWN.decodeType())
        assertEquals(Type.HORSE, FAST_HORSE.decodeType())
        assertEquals(Type.ROOK, FAST_ROOK.decodeType())
        assertEquals(Type.UNI, FAST_UNI.decodeType())
        assertEquals(Type.FLANGER, FAST_FLANGER.decodeType())
    }

    @Test
    fun testInvalidTypeDecoding() {
        try {
            val invalidType: FastType = 99
            invalidType.decodeType()
            fail("Should throw exception for invalid type")
        } catch (e: Exception) {
            assertTrue(e.message!!.contains("Unknown type"))
        }
    }

    @Test
    fun testColorConversion() {
        // Test Color to FastColor conversion
        assertEquals(FAST_WHITE, Color.WHITE.fast())
        assertEquals(FAST_BLACK, Color.BLACK.fast())
        
        // Test FastColor to Color conversion
        assertEquals(Color.WHITE, FAST_WHITE.decodeColor())
        assertEquals(Color.BLACK, FAST_BLACK.decodeColor())
    }

    @Test
    fun testPieceStateConversion() {
        // Test PieceState to FastFrozenState conversion
        assertEquals(FAST_NORMAL, PieceState.NORMAL.fast())
        assertEquals(FAST_FROZEN, PieceState.FROZEN.fast())
    }

    @Test
    fun testBoardConversion() {
        val originalBoard = Board()
        
        // Place some pieces on the original board
        originalBoard.setOnBoard(Location(originalBoard, 0, 0), 
            Piece(Location(originalBoard, 0, 0), Type.KING, Color.WHITE, PieceState.NORMAL), Board.FreezeMethod.UNFREEZE)
        originalBoard.setOnBoard(Location(originalBoard, 0, 1), 
            Piece(Location(originalBoard, 0, 1), Type.PAWN, Color.BLACK, PieceState.NORMAL), Board.FreezeMethod.UNFREEZE)
        originalBoard.setOnBoard(Location(originalBoard, 7, 7), 
            Piece(Location(originalBoard, 7, 7), Type.ROOK, Color.WHITE, PieceState.NORMAL), Board.FreezeMethod.UNFREEZE)
        
        // Convert to FastBoard
        val fastBoard = originalBoard.fast()
        
        // Verify conversion
        assertEquals(originalBoard.atMove.fast(), fastBoard.atMove)
        
        // Check specific pieces
        val whiteKing = fastBoard.getAt(0, 0)
        assertEquals(FAST_KING, whiteKing.getType())
        assertEquals(FAST_WHITE, whiteKing.getColor())
        assertEquals(FAST_NORMAL, whiteKing.getFrozen())
        
        val blackPawn = fastBoard.getAt(0, 1)
        assertEquals(FAST_PAWN, blackPawn.getType())
        assertEquals(FAST_BLACK, blackPawn.getColor())
        assertEquals(FAST_NORMAL, blackPawn.getFrozen())
        
        val whiteRook = fastBoard.getAt(7, 7)
        assertEquals(FAST_ROOK, whiteRook.getType())
        assertEquals(FAST_WHITE, whiteRook.getColor())
        assertEquals(FAST_NORMAL, whiteRook.getFrozen())
        
        // Check empty squares
        val emptySquare = fastBoard.getAt(3, 3)
        assertEquals(FAST_NONE, emptySquare.getType())
    }

    @Test
    fun testBoardConversionWithFrozenPieces() {
        val originalBoard = Board()
        
        // Place and freeze a piece
        originalBoard.setOnBoard(Location(originalBoard, 4, 4), 
            Piece(Location(originalBoard, 4, 4), Type.KING, Color.WHITE, PieceState.FROZEN), Board.FreezeMethod.FORCE)
        
        // Convert to FastBoard
        val fastBoard = originalBoard.fast()
        
        // Verify frozen piece
        val frozenKing = fastBoard.getAt(4, 4)
        assertEquals(FAST_KING, frozenKing.getType())
        assertEquals(FAST_WHITE, frozenKing.getColor())
        assertEquals(FAST_FROZEN, frozenKing.getFrozen())
        
        // Verify frozen indices are set correctly
        assertEquals(indexOf(4, 4), fastBoard.frozenWhiteIndex)
        assertEquals(-1, fastBoard.frozenBlackIndex)
    }

    @Test
    fun testMoveConversion() {
        val board = Board()
        
        // Create a piece and place it
        val fromLocation = Location(board, 4, 1)
        val toLocation = Location(board, 4, 2)
        val piece = Piece(fromLocation, Type.PAWN, Color.WHITE, PieceState.NORMAL)
        board.setOnBoard(fromLocation, piece, Board.FreezeMethod.UNFREEZE)
        
        // Create a move
        val move = Move(piece, toLocation)
        
        // Convert to FastMove
        val fastMove = move.fast()
        
        // Verify conversion
        assertEquals(fromLocation.fast(), fastMove.getFromIndex())
        assertEquals(toLocation.fast(), fastMove.getToIndex())
        assertEquals(piece.fast(), fastMove.getFromPieceState())
        assertEquals(0.toByte(), fastMove.getToPieceState()) // empty destination
    }

    @Test
    fun testVectorConversion() {
        val vector = Vector(3, 5)
        val fastIndex = vector.fast()
        
        assertEquals(indexOf(3, 5), fastIndex)
    }

    @Test
    fun testNullPieceConversion() {
        val nullPiece: de.tadris.flang_lib.Piece? = null
        val fastPieceState = nullPiece.fast()
        
        assertEquals(0.toByte(), fastPieceState)
    }

    @Test
    fun testNonNullPieceConversion() {
        val board = Board()
        val piece = Piece(Location(board, 0, 0), Type.KING, Color.WHITE, PieceState.FROZEN)
        
        val fastPieceState = piece.fast()
        
        assertEquals(FAST_KING, fastPieceState.getType())
        assertEquals(FAST_WHITE, fastPieceState.getColor())
        assertEquals(FAST_FROZEN, fastPieceState.getFrozen())
    }

    @Test
    fun testBoardWithDifferentAtMove() {
        val board = Board()
        board.atMove = Color.BLACK
        
        val fastBoard = board.fast()
        
        assertEquals(FAST_BLACK, fastBoard.atMove)
    }

    @Test
    fun testComplexBoardConversion() {
        val board = Board()
        
        // Set up a complex board state
        board.setOnBoard(Location(board, 0, 0), 
            Piece(Location(board, 0, 0), Type.KING, Color.WHITE, PieceState.NORMAL), Board.FreezeMethod.UNFREEZE)
        board.setOnBoard(Location(board, 7, 7), 
            Piece(Location(board, 7, 7), Type.KING, Color.BLACK, PieceState.NORMAL), Board.FreezeMethod.UNFREEZE)
        board.setOnBoard(Location(board, 0, 1), 
            Piece(Location(board, 0, 1), Type.PAWN, Color.WHITE, PieceState.NORMAL), Board.FreezeMethod.UNFREEZE)
        board.setOnBoard(Location(board, 7, 6), 
            Piece(Location(board, 7, 6), Type.PAWN, Color.BLACK, PieceState.NORMAL), Board.FreezeMethod.UNFREEZE)
        
        // Freeze a piece
        board.setOnBoard(Location(board, 3, 3), 
            Piece(Location(board, 3, 3), Type.ROOK, Color.WHITE, PieceState.FROZEN), Board.FreezeMethod.FORCE)
        
        board.atMove = Color.BLACK
        
        // Convert to FastBoard
        val fastBoard = board.fast()
        
        // Verify complex state
        assertEquals(FAST_BLACK, fastBoard.atMove)
        
        // Check white king
        val whiteKing = fastBoard.getAt(0, 0)
        assertEquals(FAST_KING, whiteKing.getType())
        assertEquals(FAST_WHITE, whiteKing.getColor())
        assertEquals(FAST_NORMAL, whiteKing.getFrozen())
        
        // Check black king
        val blackKing = fastBoard.getAt(7, 7)
        assertEquals(FAST_KING, blackKing.getType())
        assertEquals(FAST_BLACK, blackKing.getColor())
        assertEquals(FAST_NORMAL, blackKing.getFrozen())
        
        // Check frozen rook
        val frozenRook = fastBoard.getAt(3, 3)
        assertEquals(FAST_ROOK, frozenRook.getType())
        assertEquals(FAST_WHITE, frozenRook.getColor())
        assertEquals(FAST_FROZEN, frozenRook.getFrozen())
        
        // Check frozen index
        assertEquals(indexOf(3, 3), fastBoard.frozenWhiteIndex)
        assertEquals(-1, fastBoard.frozenBlackIndex)
    }

    @Test
    fun testMoveWithCapture() {
        val board = Board()
        
        // Set up a capture scenario
        val fromLocation = Location(board, 0, 0)
        val toLocation = Location(board, 0, 7)
        
        val attacker = Piece(fromLocation, Type.ROOK, Color.WHITE, PieceState.NORMAL)
        val victim = Piece(toLocation, Type.PAWN, Color.BLACK, PieceState.NORMAL)
        
        board.setOnBoard(fromLocation, attacker, Board.FreezeMethod.UNFREEZE)
        board.setOnBoard(toLocation, victim, Board.FreezeMethod.UNFREEZE)
        
        // Create capture move
        val move = Move(attacker, toLocation)
        
        // Convert to FastMove
        val fastMove = move.fast()
        
        // Verify capture move
        assertEquals(fromLocation.fast(), fastMove.getFromIndex())
        assertEquals(toLocation.fast(), fastMove.getToIndex())
        assertEquals(attacker.fast(), fastMove.getFromPieceState())
        assertEquals(victim.fast(), fastMove.getToPieceState())
    }
}