package de.tadris.flang_lib.bot

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.action.Move
import java.io.BufferedReader
import java.io.File
import java.io.InputStreamReader
import kotlin.math.roundToInt

class CFlangEngine(
    private val minDepth: Int = 1,
    private val maxDepth: Int = 6,
    private val cflangPath: String = "./cflang/cflang",
    private val threads: Int = Runtime.getRuntime().availableProcessors() - 1,
    private val ttSizeMB: Int = 64,
    private val maxTimeMs: Long = -1,
    private val useLME: Boolean = false,
    private val lmeMaxExtension: Int = 1
) : Engine {

    init {
        // Verify that the cflang executable exists and is executable
        val cflangFile = File(cflangPath)
        if (!cflangFile.exists()) {
            throw IllegalArgumentException("CFlang executable not found at: $cflangPath")
        }
        if (!cflangFile.canExecute()) {
            throw IllegalArgumentException("CFlang executable is not executable: $cflangPath")
        }
    }

    override fun findBestMove(board: Board, printTime: Boolean): BotResult {
        return runCFlangSearch(board, printTime, null)
    }

    override fun findBestMoveWithFixedDepth(board: Board, printTime: Boolean, depth: Int): BotResult {
        return runCFlangSearch(board, printTime, null, overrideMaxDepth = depth)
    }

    override fun findBestMoveIterative(board: Board, printTime: Boolean, maxTimeMs: Long): BotResult {
        return runCFlangSearch(board, printTime, maxTimeMs)
    }

    private fun runCFlangSearch(
        board: Board, 
        printTime: Boolean, 
        timeLimit: Long? = null,
        overrideMaxDepth: Int? = null
    ): BotResult {
        val startTime = System.currentTimeMillis()
        val fbn2 = board.getFBN2()
        val actualMaxDepth = overrideMaxDepth ?: maxDepth
        val actualTimeLimit = timeLimit ?: maxTimeMs
        
        // Build command arguments
        val command = mutableListOf(
            cflangPath,
            fbn2,
            "--min-depth", minDepth.toString(),
            "--max-depth", actualMaxDepth.toString(),
            "--threads", threads.toString(),
            "--ttsize", ttSizeMB.toString(),
            "--machine-readable"
        )
        
        if (useLME) {
            command.add("--use-lme")
            command.addAll(listOf("--lme-max-ext", lmeMaxExtension.toString()))
        }
        
        if (actualTimeLimit > 0) {
            command.addAll(listOf("--max-time", actualTimeLimit.toString()))
        }

        if (printTime) {
            println("Running CFlang: ${command.joinToString(" ")}")
        }

        try {
            val processBuilder = ProcessBuilder(command)
            val process = processBuilder.start()
            
            val reader = BufferedReader(InputStreamReader(process.inputStream))
            val errorReader = BufferedReader(InputStreamReader(process.errorStream))
            
            // Read all output lines
            val outputLines = reader.readLines()
            val errorLines = errorReader.readLines()
            
            val exitCode = process.waitFor()
            
            if (exitCode != 0) {
                val errorMsg = errorLines.joinToString("\n")
                throw RuntimeException("CFlang process failed with exit code $exitCode: $errorMsg")
            }
            
            // Parse the machine-readable output
            val result = parseCFlangOutput(outputLines, board)
            
            // Print timing information if requested
            if (printTime) {
                val endTime = System.currentTimeMillis()
                val totalTime = endTime - startTime
                val evalsK = result.count / 1000
                val evalsPerMs = if (totalTime > 0) result.count / totalTime else 0
                
                println("Moves: ${result.evaluations.size}, Evals: ${evalsK}K, Depth: ${result.bestMove.depth}, Time: ${totalTime}ms, EPms: $evalsPerMs")
            }
            
            return result
            
        } catch (e: Exception) {
            throw RuntimeException("Failed to execute CFlang: ${e.message}", e)
        }
    }

    private fun parseCFlangOutput(lines: List<String>, board: Board): BotResult {
        var bestMove: MoveEvaluation? = null
        val allMoves = mutableListOf<MoveEvaluation>()
        var totalEvaluations = 0L

        for (line in lines) {
            when {
                line.startsWith("BEST ") -> {
                    val parts = line.substring(5).split(" ")
                    if (parts.size >= 3) {
                        val moveStr = parts[0]
                        val eval = parts[1].toDouble()
                        val depth = parts[2].toInt()
                        
                        try {
                            val move = parseMove(board, moveStr)
                            bestMove = MoveEvaluation(move, eval, depth)
                        } catch (e: Exception) {
                            throw RuntimeException("Failed to parse best move '$moveStr': ${e.message}", e)
                        }
                    }
                }
                line.startsWith("MOVE ") -> {
                    val parts = line.substring(5).split(" ")
                    if (parts.size >= 3) {
                        val moveStr = parts[0]
                        val eval = parts[1].toDouble()
                        val depth = parts[2].toInt()
                        
                        try {
                            val move = parseMove(board, moveStr)
                            allMoves.add(MoveEvaluation(move, eval, depth))
                        } catch (e: Exception) {
                            // Skip moves that can't be parsed rather than failing
                            println("Warning: Failed to parse move '$moveStr': ${e.message}")
                        }
                    }
                }
                line.startsWith("EVALUATIONS ") -> {
                    totalEvaluations = line.substring(12).toLong()
                }
            }
        }

        if (bestMove == null) {
            throw RuntimeException("No best move found in CFlang output")
        }

        val sortedMoves = allMoves
            .shuffled()
            .sortedBy { -((it.evaluation * 100).roundToInt() / 100.0) * board.atMove.evaluationNumber }
        return BotResult(bestMove, sortedMoves, totalEvaluations)
    }

    private fun parseMove(board: Board, moveStr: String): Move {
        // CFlang outputs moves in format like "f1f3" (from-to notation)
        // Convert to the format expected by Move.parse()
        if (moveStr.length == 4) {
            // This is likely a from-to move like "f1f3"
            return Move.parse(board, moveStr)
        } else if (moveStr.length >= 5 && moveStr[4] == '-') {
            // This might be piece-from-to format like "Ff1-f3", extract the from-to part
            val fromTo = moveStr.substring(1, 3) + moveStr.substring(5, 7)
            return Move.parse(board, fromTo)
        } else {
            // Try parsing as-is first
            return Move.parse(board, moveStr)
        }
    }
}