package de.tadris.flang.ui.board

import android.content.Context
import android.util.Log
import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.Location
import de.tadris.flang_lib.action.Move
import de.tadris.flang_lib.Piece
import de.tadris.flang_lib.Vector

class BoardMoveDetector(
    private val context: Context,
    private val boardView: BoardView,
    private val myColor: Color?,
    var listener: MoveListener?,
) : BoardView.FieldClickListener {

    private var movesAllowed = true
    private var premovesAllowed = true

    var selected: MiscView? = null
    var options: MutableList<MiscView> = mutableListOf()

    override fun onFieldTouchBegin(location: Location) {
        Log.d("MoveDetector", "Touch begin: $location. movesAllowed: $movesAllowed premovesAllowed: $premovesAllowed")
        val piece = location.piece
        if(!movesAllowed && !premovesAllowed){
            return
        }
        if(selected == null){
            if(piece != null && !location.board.gameIsComplete()){
                if(piece.color == location.board.atMove){
                    if(myColor == null || (location.board.atMove == myColor && myColor == piece.color)){
                        Log.d("MoveDetector", "Selecting $piece")
                        selectPiece(piece)
                    }
                }else if(premovesAllowed && myColor != location.board.atMove && myColor == piece.color){ // opponent is at move and I clicked on my piece
                    Log.d("MoveDetector", "Selecting $piece (premove)")
                    selectPiece(piece)
                }
            }
        }else{
            val clickedOnSelected = selected!!.getLocation() == location
            if(options.find { it.getLocation() == location } == null){
                deselect()
                if(piece != null && !clickedOnSelected){
                    onFieldTouchBegin(location)
                }
            }
        }
    }

    private fun selectPiece(piece: Piece){
        listener?.onPremoveClearRequested()
        selected = MiscView(context, piece.location, MiscView.MiscType.SELECTED)
        boardView.attach(selected!!)
        options = mutableListOf()

        piece.getPossibleMoves().forEach { move ->
            addOption(move.target)
        }
    }

    private fun addOption(target: Vector){
        val view = MiscView(context, target, MiscView.MiscType.OPTION)
        options.add(view)
        boardView.attach(view)
    }

    private fun deselect(){
        if(selected == null){ return }
        boardView.detach(selected!!)
        options.forEach {
            boardView.detach(it)
        }
        selected = null
        options.clear()
    }

    override fun onFieldRelease(location: Location) {
        options.toList().forEach {
            if(it.getLocation() == location){
                requestMove(location.board, Move(selected!!.getLocation().toLocation(boardView.board).piece!!, location))
            }
        }
    }

    private fun requestMove(board: Board, move: Move){
        deselect()
        if(board.atMove == move.piece.color){
            listener?.onMoveRequested(move)
        }else if(premovesAllowed){
            listener?.onPremoveRequested(move)
        }
    }

    fun setAllowed(allowed: Boolean, premoves: Boolean){
        this.movesAllowed = allowed
        this.premovesAllowed = premoves
        if(!allowed){
            deselect()
        }
    }

    interface MoveListener {

        fun onMoveRequested(move: Move)

        fun onPremoveRequested(move: Move)

        fun onPremoveClearRequested()

    }

}