#ifndef FAST_EVALUATION_H
#define FAST_EVALUATION_H

#include "fast_board.h"
#include "fast_move_generator.h"
#include "bitboard_utils.h"
#include <math.h>

// Evaluation constants
#define MATE_EVAL 10000.0
#define MATE_STEP_LOSS 100.0

// Evaluation hyper parameters
#define EVAL_WEIGHT_MATRIX 5.0
#define EVAL_WEIGHT_MOVEMENT 10.0
#define EVAL_WEIGHT_PIECE_VALUE 60.0
#define EVAL_WEIGHT_KINGS_EVAL 1.0
#define EVAL_KINGS_WEIGHT_MULTIPLIER 0.3125

// Location evaluation structure
typedef struct {
    int16_t occupied_by;
    uint16_t white_control;
    uint16_t black_control;
    uint16_t weight;
} LocationEvaluation;

// Overall statistics for each color
typedef struct {
    int movements;
    int piece_value;
} OverallStats;

// Fast board evaluation structure
typedef struct {
    FastBoard* board;
    FastMoveGenerator move_generator;  // Own move generator for evaluation
    LocationEvaluation evaluation_matrix[ARRAY_SIZE];
    OverallStats white_stats;
    OverallStats black_stats;
} FastBoardEvaluation;

// Evaluation functions
void fast_evaluation_init(FastBoardEvaluation* eval, FastBoard* board);
double fast_evaluation_evaluate(FastBoardEvaluation* eval);
void fast_evaluation_prepare(FastBoardEvaluation* eval);
void fast_evaluation_evaluate_location(FastBoardEvaluation* eval, FastBoardIndex index);
double fast_evaluation_get_kings_eval(FastBoardEvaluation* eval);

// Location evaluation functions
void location_evaluation_reset(LocationEvaluation* loc_eval);
void location_evaluation_add_threat(LocationEvaluation* loc_eval, FastColor color, int threat);
double location_evaluation_evaluate_field(const LocationEvaluation* loc_eval);

// Statistics functions
void overall_stats_reset(OverallStats* stats);
OverallStats* get_stats(FastBoardEvaluation* eval, FastColor color);

#endif // FAST_EVALUATION_H