package de.tadris.flang

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.fast.*
import org.junit.Assert.*
import org.junit.Test

class TestFastBoardEquivalence {

    @Test
    fun testBoardStateEquivalence() {
        // Test various board states to ensure FastBoard and Board represent the same state
        val testBoards = listOf(
            Board(),
            Board.fromFMN("Uf3"),
            Board.fromFMN("Uf3 uc6"),
            Board.fromFMN("Uf3 uc6 Hd3 a7b6 Kg1 ka7"),
            Board.fromFMN("Uf3 uc6 Hd3 a7b6 Kg1 ka7 Pb2 c2 Rc1 pa6 e2e3 a4 g2g3 d7e6 Kg2 d3")
        )
        
        for ((index, board) in testBoards.withIndex()) {
            val fastBoard = board.fast()
            val decodedBoard = fastBoard.decode()
            
            println("Testing board state $index:")
            println("Original: ${board.getFBN()}")
            println("Fast:     ${fastBoard.getFBN()}")
            println("Decoded:  ${decodedBoard.getFBN()}")
            
            assertEquals("Board state $index: FBN should match", board.getFBN(), fastBoard.getFBN())
            assertEquals("Board state $index: atMove should match", board.atMove, fastBoard.atMove.decodeColor())
            assertEquals("Board state $index: moveNumber should match", board.moveNumber, fastBoard.moveNumber)
            
            // Test specific pieces
            for (i in 0 until Board.ARRAY_SIZE) {
                val originalPiece = board.getAt(i * 2)
                val fastPiece = fastBoard.getAt(i)
                
                if (originalPiece != null) {
                    assertEquals("Board $index, index $i: piece type mismatch", 
                        originalPiece.type.fast(), fastPiece.getType())
                    assertEquals("Board $index, index $i: piece color mismatch", 
                        originalPiece.color.fast(), fastPiece.getColor())
                    assertEquals("Board $index, index $i: piece state mismatch", 
                        originalPiece.state.fast(), fastPiece.getFrozen())
                } else {
                    assertEquals("Board $index, index $i: empty square should be FAST_NONE", 
                        FAST_NONE, fastPiece.getType())
                }
            }
            
            // Test frozen piece indices
            assertEquals("Board $index: frozen white index should match", 
                board.findFrozenFigure(Color.WHITE), fastBoard.frozenWhiteIndex)
            assertEquals("Board $index: frozen black index should match", 
                board.findFrozenFigure(Color.BLACK), fastBoard.frozenBlackIndex)
        }
    }

    @Test
    fun testWinConditions() {
        // Test win condition detection
        val testBoards = listOf(
            Board(),
            Board.fromFMN("Uf3 uc6 Hd3 a7b6 Kg1 ka7 Pb2 c2 Rc1 pa6 e2e3 a4 g2g3 d7e6 Kg2 d3")
        )
        
        for ((index, board) in testBoards.withIndex()) {
            val fastBoard = board.fast()
            
            assertEquals("Board $index: white win condition should match", 
                board.hasWon(Color.WHITE), fastBoard.hasWon(FAST_WHITE))
            assertEquals("Board $index: black win condition should match", 
                board.hasWon(Color.BLACK), fastBoard.hasWon(FAST_BLACK))
            assertEquals("Board $index: game completion should match", 
                board.gameIsComplete(), fastBoard.gameIsComplete())
        }
    }

    @Test
    fun testMoveExecutionEquivalence() {
        val board = Board()
        val fastBoard = board.fast()
        
        // Execute the same move on both boards
        val firstMove = board.findAllMoves(Color.WHITE).first()
        val fastFirstMove = firstMove.fast()
        
        println("Original move: ${firstMove}")
        println("Fast move: ${fastFirstMove.toDebugString()}")
        
        val boardCopy = Board(board.pieces.copyOf(), board.atMove, board.moveNumber)
        boardCopy.executeOnBoard(firstMove)
        
        val fastBoardCopy = FastBoard(fastBoard.pieces.copyOf(), fastBoard.atMove, fastBoard.frozenWhiteIndex, fastBoard.frozenBlackIndex)
        fastBoardCopy.moveNumber = fastBoard.moveNumber
        fastBoardCopy.executeOnBoard(fastFirstMove)
        
        println("After move:")
        println("Original: ${boardCopy.getFBN()}")
        println("Fast:     ${fastBoardCopy.getFBN()}")
        
        assertEquals("After move execution: FBN should match", boardCopy.getFBN(), fastBoardCopy.getFBN())
        assertEquals("After move execution: atMove should match", boardCopy.atMove, fastBoardCopy.atMove.decodeColor())
        assertEquals("After move execution: moveNumber should match", boardCopy.moveNumber, fastBoardCopy.moveNumber)
    }

    @Test
    fun testFrozenPieceHandling() {
        // Test boards with frozen pieces
        val board = Board()
        val king = board.findKing(Color.WHITE)!!
        board.setOnBoard(king.location, king, Board.FreezeMethod.FORCE)
        
        val fastBoard = board.fast()
        
        assertEquals("Frozen piece FBN should match", board.getFBN(), fastBoard.getFBN())
        assertEquals("Frozen white index should match", board.findFrozenFigure(Color.WHITE), fastBoard.frozenWhiteIndex)
        
        // Test unfreezing
        board.unfreezeAllOnBoard(Color.WHITE)
        fastBoard.unfreezeOnBoard(FAST_WHITE)
        
        assertEquals("After unfreezing: FBN should match", board.getFBN(), fastBoard.getFBN())
        assertEquals("After unfreezing: frozen white index should match", board.findFrozenFigure(Color.WHITE), fastBoard.frozenWhiteIndex)
    }

    @Test
    fun testPieceIterationEquivalence() {
        val board = Board.fromFMN("Uf3 uc6 Hd3 a7b6 Kg1 ka7 Pb2 c2")
        val fastBoard = board.fast()
        
        // Test piece iteration
        val regularPieces = mutableListOf<String>()
        val fastPieces = mutableListOf<String>()
        
        board.eachPiece(null) { piece ->
            regularPieces.add("${piece.type.name}:${piece.color.name}:${piece.state.name}@${piece.location.x},${piece.location.y}")
        }
        
        fastBoard.eachPiece(null) { index, piece ->
            val type = piece.getType().decodeType()?.name ?: "NULL"
            val color = piece.getColor().decodeColor().name
            val state = if (piece.getFrozen()) "FROZEN" else "NORMAL"
            fastPieces.add("$type:$color:$state@${index.x},${index.y}")
        }
        
        regularPieces.sort()
        fastPieces.sort()
        
        assertEquals("Piece iteration should produce same results", regularPieces, fastPieces)
    }

    @Test
    fun testDebugBoardDifferences() {
        // This test is specifically to debug why bot evaluations differ
        val board = Board.fromFMN("Uf3 uc6 Hd3 a7b6 Kg1 ka7 Pb2 c2 Rc1 pa6 e2e3 a4 g2g3 d7e6 Kg2 d3")
        val fastBoard = board.fast()
        
        println("=== DEBUGGING BOARD DIFFERENCES ===")
        println("Original FBN: ${board.getFBN()}")
        println("Fast FBN:     ${fastBoard.getFBN()}")
        println("AtMove - Original: ${board.atMove}, Fast: ${fastBoard.atMove.decodeColor()}")
        println("MoveNumber - Original: ${board.moveNumber}, Fast: ${fastBoard.moveNumber}")
        
        // Check every position
        for (i in 0 until Board.ARRAY_SIZE) {
            val originalPiece = board.getAt(i * 2)
            val fastPiece = fastBoard.getAt(i)
            val x = i % Board.BOARD_SIZE
            val y = i / Board.BOARD_SIZE
            
            val originalStr = if (originalPiece != null) {
                "${originalPiece.type.name}:${originalPiece.color.name}:${originalPiece.state.name}"
            } else {
                "EMPTY"
            }
            
            val fastStr = if (fastPiece.getType() != FAST_NONE) {
                "${fastPiece.getType().decodeType()?.name}:${fastPiece.getColor().decodeColor().name}:${if (fastPiece.getFrozen()) "FROZEN" else "NORMAL"}"
            } else {
                "EMPTY"
            }
            
            if (originalStr != fastStr) {
                println("DIFFERENCE at ($x,$y): Original=$originalStr, Fast=$fastStr")
            }
        }
        
        // Check frozen pieces
        println("Frozen white - Original: ${board.findFrozenFigure(Color.WHITE)}, Fast: ${fastBoard.frozenWhiteIndex}")
        println("Frozen black - Original: ${board.findFrozenFigure(Color.BLACK)}, Fast: ${fastBoard.frozenBlackIndex}")
        
        // Check win conditions
        println("White won - Original: ${board.hasWon(Color.WHITE)}, Fast: ${fastBoard.hasWon(FAST_WHITE)}")
        println("Black won - Original: ${board.hasWon(Color.BLACK)}, Fast: ${fastBoard.hasWon(FAST_BLACK)}")
        println("Game complete - Original: ${board.gameIsComplete()}, Fast: ${fastBoard.gameIsComplete()}")
    }
}