package de.tadris.flang

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.Piece
import de.tadris.flang_lib.PieceState
import de.tadris.flang_lib.Type
import de.tadris.flang_lib.fast.*
import org.junit.Assert.*
import org.junit.Test

class TestFastBoardBugs {

    @Test
    fun testEmptySquareColorMapping() {
        val board = Board()
        val fastBoard = board.fast()
        
        println("=== TESTING EMPTY SQUARE COLOR MAPPING ===")
        
        // Check empty squares
        for (i in 0 until Board.ARRAY_SIZE) {
            val originalPiece = board.getAt(i * 2)
            val fastPiece = fastBoard.getAt(i)
            val x = i % Board.BOARD_SIZE
            val y = i / Board.BOARD_SIZE
            
            if (originalPiece == null) {
                // Empty square - should be FAST_NONE
                assertEquals("Empty square at ($x,$y) should be FAST_NONE", FAST_NONE, fastPiece.getType())
                
                // Check what color it thinks it is
                val fastColor = fastPiece.getColor()
                println("Empty square at ($x,$y): color=${fastColor} (${fastColor.decodeColor().name})")
                
                // Empty squares should not have a meaningful color, but if they do, 
                // it should be consistent
                if (fastColor != FAST_BLACK) {
                    println("WARNING: Empty square at ($x,$y) has color $fastColor instead of expected FAST_BLACK")
                }
            }
        }
    }

    @Test
    fun testPawnPromotionDifferences() {
        val board = Board()
        val fastBoard = board.fast()
        
        println("=== TESTING PAWN PROMOTION DIFFERENCES ===")
        
        // Test white pawn promotion
        val whitePawnIndex = indexOf(0, 6) // Put white pawn near promotion
        board.setOnBoard(board.getLocationForIndex(whitePawnIndex * 2), 
            Piece(board.getLocationForIndex(whitePawnIndex * 2), Type.PAWN, Color.WHITE, PieceState.NORMAL),
            Board.FreezeMethod.UNFREEZE)
        
        fastBoard.setAt(whitePawnIndex, packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL))
        
        // Promote to winning Y
        val promotionIndex = indexOf(0, 7) // Y=7 is white winning Y
        board.setOnBoard(board.getLocationForIndex(promotionIndex * 2), Piece(board.getLocationForIndex(promotionIndex * 2), de.tadris.flang_lib.Type.PAWN, Color.WHITE, PieceState.NORMAL))
        fastBoard.updateAt(promotionIndex, FAST_PAWN, FAST_WHITE)
        
        val regularPromotedPiece = board.getAt(promotionIndex * 2)
        val fastPromotedPiece = fastBoard.getAt(promotionIndex)
        
        println("Regular promoted piece: ${regularPromotedPiece?.type?.name} ${regularPromotedPiece?.state?.name}")
        println("Fast promoted piece: ${fastPromotedPiece.getType().decodeType()?.name} ${if (fastPromotedPiece.getFrozen()) "FROZEN" else "NORMAL"}")
        
        // Check if they match
        assertEquals("Promoted piece type should match", regularPromotedPiece?.type?.fast(), fastPromotedPiece.getType())
        assertEquals("Promoted piece state should match", regularPromotedPiece?.state?.fast(), fastPromotedPiece.getFrozen())
    }

    @Test
    fun testWinConditionDifferences() {
        val board = Board()
        val fastBoard = board.fast()
        
        println("=== TESTING WIN CONDITION DIFFERENCES ===")
        
        // Place white king in winning position
        val kingIndex = indexOf(4, 7) // Y=7 is white winning Y
        board.setOnBoard(board.getLocationForIndex(kingIndex * 2), 
            Piece(board.getLocationForIndex(kingIndex * 2), de.tadris.flang_lib.Type.KING, Color.WHITE, PieceState.NORMAL),
            Board.FreezeMethod.UNFREEZE)
        
        fastBoard.setAt(kingIndex, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        
        // Place black king somewhere else
        val blackKingIndex = indexOf(4, 3)
        board.setOnBoard(board.getLocationForIndex(blackKingIndex * 2), 
            Piece(board.getLocationForIndex(blackKingIndex * 2), de.tadris.flang_lib.Type.KING, Color.BLACK, PieceState.NORMAL),
            Board.FreezeMethod.UNFREEZE)
        
        fastBoard.setAt(blackKingIndex, packPieceState(FAST_KING, FAST_BLACK, FAST_NORMAL))
        
        val regularWhiteWon = board.hasWon(Color.WHITE)
        val fastWhiteWon = fastBoard.hasWon(FAST_WHITE)
        
        println("Regular white won: $regularWhiteWon")
        println("Fast white won: $fastWhiteWon")
        
        assertEquals("White win condition should match", regularWhiteWon, fastWhiteWon)
        
        // Check the Y coordinate calculation
        val regularKingIndex = board.findKingIndex(Color.WHITE)
        val fastKingIndexFound = fastBoard.findKingIndex(FAST_WHITE)
        
        println("Regular king index: $regularKingIndex")
        println("Fast king index: $fastKingIndexFound")
        
        if (regularKingIndex != -1 && fastKingIndexFound != -1) {
            val regularY = Board.getYForIndex(regularKingIndex)
            val fastY = fastKingIndexFound.y
            
            println("Regular king Y: $regularY")
            println("Fast king Y: $fastY")
            
            assertEquals("King Y coordinates should match", regularY, fastY)
        }
    }

    @Test
    fun testFBNConsistency() {
        val boards = listOf(
            Board(),
            Board.fromFMN("Uf3"),
            Board.fromFMN("Uf3 uc6 Hd3 a7b6 Kg1 ka7 Pb2 c2 Rc1 pa6 e2e3 a4 g2g3 d7e6 Kg2 d3")
        )
        
        for ((index, board) in boards.withIndex()) {
            val fastBoard = board.fast()
            val decodedBoard = fastBoard.decode()
            
            println("=== BOARD $index ===")
            println("Original: ${board.getFBN()}")
            println("Fast:     ${fastBoard.getFBN()}")
            println("Decoded:  ${decodedBoard.getFBN()}")
            
            if (board.getFBN() != fastBoard.getFBN()) {
                println("MISMATCH DETECTED!")
                
                // Find the exact differences
                val original = board.getFBN()
                val fast = fastBoard.getFBN()
                
                for (i in 0 until minOf(original.length, fast.length)) {
                    if (original[i] != fast[i]) {
                        println("Difference at position $i: original='${original[i]}' fast='${fast[i]}'")
                        val squareIndex = i / 2
                        val x = squareIndex % Board.BOARD_SIZE
                        val y = squareIndex / Board.BOARD_SIZE
                        println("This corresponds to square ($x,$y)")
                    }
                }
            }
            
            assertEquals("Board $index FBN should match", board.getFBN(), fastBoard.getFBN())
        }
    }

    @Test
    fun debugSpecificBoardState() {
        // Debug a specific board state that might cause issues
        val board = Board.fromFMN("Uf3 uc6 Hd3 a7b6 Kg1 ka7 Pb2 c2 Rc1 pa6 e2e3 a4 g2g3 d7e6 Kg2 d3")
        val fastBoard = board.fast()
        
        println("=== DEBUGGING SPECIFIC BOARD STATE ===")
        println("Original FBN: ${board.getFBN()}")
        println("Fast FBN:     ${fastBoard.getFBN()}")
        
        // Check every detail
        assertEquals("FBN should match exactly", board.getFBN(), fastBoard.getFBN())
        assertEquals("AtMove should match", board.atMove, fastBoard.atMove.decodeColor())
        assertEquals("MoveNumber should match", board.moveNumber, fastBoard.moveNumber)
        assertEquals("Game completion should match", board.gameIsComplete(), fastBoard.gameIsComplete())
        assertEquals("White win should match", board.hasWon(Color.WHITE), fastBoard.hasWon(FAST_WHITE))
        assertEquals("Black win should match", board.hasWon(Color.BLACK), fastBoard.hasWon(FAST_BLACK))
        
        println("All checks passed - board states are equivalent")
    }
}