package de.tadris.flang_lib.bot.fast

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.action.Move
import de.tadris.flang_lib.bot.BotResult
import de.tadris.flang_lib.bot.Engine
import de.tadris.flang_lib.bot.FastBoardEvaluation
import de.tadris.flang_lib.bot.FastNeoBoardEvaluation
import de.tadris.flang_lib.bot.MoveEvaluation
import de.tadris.flang_lib.bot.NeoBoardEvaluation
import de.tadris.flang_lib.fast.FAST_KING
import de.tadris.flang_lib.fast.FAST_NONE
import de.tadris.flang_lib.fast.FAST_PAWN
import de.tadris.flang_lib.fast.FastBoard
import de.tadris.flang_lib.fast.FastMove
import de.tadris.flang_lib.fast.FastMoveGenerator
import de.tadris.flang_lib.fast.MoveBuffer
import de.tadris.flang_lib.fast.evaluationNumber
import de.tadris.flang_lib.fast.fast
import de.tadris.flang_lib.fast.getColor
import de.tadris.flang_lib.fast.getFromIndex
import de.tadris.flang_lib.fast.getFromPieceState
import de.tadris.flang_lib.fast.getOpponent
import de.tadris.flang_lib.fast.getToIndex
import de.tadris.flang_lib.fast.getToPieceState
import de.tadris.flang_lib.fast.getType
import de.tadris.flang_lib.fast.winningY
import de.tadris.flang_lib.fast.y
import de.tadris.flang_lib.opening.DefaultOpeningDatabase
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit
import kotlin.math.absoluteValue
import kotlin.math.ln
import kotlin.math.max
import kotlin.math.min
import kotlin.math.roundToInt

/**
 * Enhanced FlangBot based on ParallelFlangBot with transposition tables for improved performance.
 * Features:
 * - Zobrist hashing for position identification
 * - Transposition table for caching evaluations
 * - Parallel search with shared transposition table
 * - Enhanced move ordering
 */
class FastFlangBot(
    val minDepth: Int,
    val maxDepth: Int,
    val useOpeningDatabase: Boolean = true,
    val evaluationFactory: () -> FastBoardEvaluation = { FastNeoBoardEvaluation() },
    private val threads: Int = Runtime.getRuntime().availableProcessors(),
    ttSizeMB: Int = 64,
    val useNullMovePruning: Boolean = false,
    val nullMoveReduction: Int = 3,
    val nullMoveMinDepth: Int = 2,
    val useLMR: Boolean = false,
    val lmrMinDepth: Int = 2,
    val lmrMinMoveIndex: Int = 3,
    val useLME: Boolean = false,
    val lmeMaxExtension: Int = 1,
): Engine {

    val lock = Object()
    var totalEvaluations = 0L
    val moveEvaluations = mutableListOf<MoveEvaluation>()

    // Transposition table and Zobrist hashing
    private val transpositionTable = TranspositionTable(ttSizeMB)
    private val zobristHash = ZobristHash()
    private val moveOrderer = MoveOrderer()
    
    // Mate detection constants
    private companion object {
        const val MATE_TT_DEPTH = 99 // Special depth for mate positions in TT
    }
    
    /**
     * Check if a value represents a mate score
     */
    private fun isMateScore(value: Double): Boolean {
        return value.absoluteValue > 5000.0
    }

    /**
     * Find best move to exact depth (no iterative deepening)
     */
    override fun findBestMove(board: Board, printTime: Boolean) =
        findBestMoveIterative(board, printTime)

    fun findBestMoveWithFixedDepth(board: Board, printTime: Boolean, depth: Int): BotResult {
        if(useOpeningDatabase){
            DefaultOpeningDatabase.db.query(board)?.let { return it }
        }

        val start = System.currentTimeMillis()

        // Start new search in transposition table#
        totalEvaluations = 0
        transpositionTable.newSearch()
        moveOrderer.clearKillerMoves()

        val eval = findBestMove(board, depth)
        val end = System.currentTimeMillis()

        if (printTime) {
            val time = end - start
            println("Moves: ${eval?.evaluations?.size}, Evals: ${totalEvaluations/1000}k, Depth: $depth, Time: ${end - start}ms, EPms: ${totalEvaluations / max(1, time)}")
            println(transpositionTable.getUsageStats())
        }
        return eval ?: throw Exception("No moves available")
    }

    fun findBestMoveIterative(board: Board, printTime: Boolean = true, maxTimeMs: Long = Long.MAX_VALUE): BotResult {
        if(useOpeningDatabase){
            DefaultOpeningDatabase.db.query(board)?.let { return it }
        }

        val start = System.currentTimeMillis()

        // Start new search in transposition table
        totalEvaluations = 0
        transpositionTable.newSearch()
        moveOrderer.clearKillerMoves()

        val eval = findBestMoveIterativeDeepening(board, if(maxTimeMs <= 0) Long.MAX_VALUE else maxTimeMs, start)
        val end = System.currentTimeMillis()

        if (printTime) {
            val time = end - start
            println("Moves: ${eval?.evaluations?.size}, Evals: ${totalEvaluations/1000}k, Depth: ${eval?.bestMove?.depth ?: maxDepth}, Time: ${end - start}ms, EPms: ${totalEvaluations / max(1, time)}")
            println(transpositionTable.getUsageStats())
        }
        return eval!!
    }

    /**
     * Iterative deepening search - searches progressively deeper until time limit or max depth
     */
    private fun findBestMoveIterativeDeepening(board: Board, maxTimeMs: Long, startTime: Long): BotResult? {
        var bestResult: BotResult? = null
        var depth = min(5, minDepth)
        
        while (depth <= maxDepth) {
            val iterationStart = System.currentTimeMillis()
            
            // Check if we have enough time for this iteration
            val elapsed = iterationStart - startTime
            if (depth > minDepth && bestResult != null && elapsed >= maxTimeMs) break
            
            // Estimate time needed for this depth (roughly 3x previous depth)
            if (depth > minDepth && bestResult != null) {
                val lastIterationTime = iterationStart - startTime
                val estimatedTime = lastIterationTime * 3
                if (elapsed + estimatedTime > maxTimeMs) break
            }
            
            val result = findBestMove(board, depth)
            if (result != null) {
                bestResult = result
                
                // Update the depth in the best move for reporting
                bestResult = BotResult(
                    MoveEvaluation(bestResult.bestMove.move, bestResult.bestMove.evaluation, depth),
                    bestResult.evaluations.toList(),
                    bestResult.count
                )
            }
            
            depth++
        }
        
        return bestResult
    }

    private fun findBestMove(board: Board, depth: Int): BotResult? {
        moveEvaluations.clear()

        val allMoves = board.findAllMoves(board.atMove).toMutableList()

        if(allMoves.isEmpty()) return null

        // Enhanced move ordering: check for hash move from previous search
        val boardHash = zobristHash.computeHash(board.fast())

        // Sort moves: hash move first, then by quick evaluation
        val eval = NeoBoardEvaluation()
        allMoves.sortBy { eval.evaluate(board.executeOnNewBoard(it)) }

        // Create a new executor for this search
        val executors = Executors.newFixedThreadPool(threads)
        
        allMoves.forEach {
            executors.submit(FastFlangBotThread(board, it, depth - 1, evaluationFactory()))
        }

        executors.shutdown()
        executors.awaitTermination(1, TimeUnit.HOURS)

        moveEvaluations.shuffle()
        moveEvaluations.sortBy { -((it.evaluation * 100).roundToInt() / 100.0) * board.atMove.evaluationNumber }
        
        if (moveEvaluations.size == 0) {
            return null
        }
        
        val bestMove = moveEvaluations[0]

        // Store the best move in transposition table
        // Use special depth for mate positions to prevent eviction
        val storeDepth = if (isMateScore(bestMove.evaluation)) MATE_TT_DEPTH else depth
        transpositionTable.store(
            zobristHash = boardHash,
            depth = storeDepth,
            value = bestMove.evaluation,
            nodeType = TranspositionTable.NodeType.EXACT,
            bestMove = bestMove.move.fast()
        )

        return BotResult(MoveEvaluation(bestMove.move, bestMove.evaluation, depth), moveEvaluations.toList(), totalEvaluations)
    }

    inner class FastFlangBotThread(
        val board: Board, 
        val move: Move, 
        val depth: Int, 
        val evaluation: FastBoardEvaluation
    ) : Runnable {

        private val maxDepth get() = if(useLME) depth + lmeMaxExtension else depth
        private val moveBuffers = Array(maxDepth) { MoveBuffer() }
        private var bufferStackPointer = 0
        private val moveGenerator = FastMoveGenerator(FastBoard(), false, 1)
        private var evaluationCount = 0
        private var nullMoveLevel = 0
        private var lmeCounter = 0

        /**
         * Calculate Late Move Reduction amount (aggressive)
         */
        private fun calculateLMRReduction(depth: Int, moveIndex: Int): Int {
            if (!useLMR || depth < lmrMinDepth || moveIndex < lmrMinMoveIndex) {
                return 0
            }
            
            // More aggressive formula: reduction = 0.75 + ln(depth) * ln(moveIndex) / 2.5
            val reduction = (0.75 + ln(depth.toDouble()) * ln(moveIndex.toDouble()) / 2.5).roundToInt()
            
            // Allow higher reductions, cap at maximum of 4, and ensure it doesn't exceed depth - 1
            return min(min(reduction, 4), depth - 1)
        }

        /**
         * Check if a move should be extended by LME
         */
        private fun shouldExtendMove(move: FastMove, depth: Int): Boolean {
            if (!useLME || depth <= 1) {
                return false
            }
            if(lmeCounter >= lmeMaxExtension){
                return false
            }
            
            val fromPiece = move.getFromPieceState()

            if(fromPiece.getType() == FAST_KING){
                val fromIndex = move.getFromIndex()
                val toIndex = move.getToIndex()
                val color = fromPiece.getColor()

                return (toIndex.y - fromIndex.y) * color.evaluationNumber > 0 // check if king moves forward
                        && (toIndex.y - color.winningY).absoluteValue <= 3 // and distance to win is low
            }
            
            return false
        }

        /**
         * Check if a move should be reduced by LMR (more selective)
         */
        private fun shouldReduceMove(move: FastMove, moveIndex: Int, depth: Int, inCheck: Boolean, alpha: Double, beta: Double): Boolean {
            if (!useLMR || depth < lmrMinDepth || moveIndex < lmrMinMoveIndex) {
                return false
            }
            
            // Don't reduce tactical moves
            if (move.getToPieceState().getType() != FAST_NONE) {
                return false // Capture
            }
            
            // Don't reduce if in check or gives check
            if (inCheck) {
                return false
            }

            // Don't reduce killer moves (they're likely good)
            if (moveOrderer.isKillerMove(move, depth)) {
                return false
            }
            
            // Don't reduce if we're in the search window (PV node)
            if (beta - alpha > 1.0) {
                return false
            }
            
            // Don't reduce king moves
            if (move.getFromPieceState().getType() == FAST_KING) {
                return false
            }

            // Don't reduce pawn moves
            if (move.getFromPieceState().getType() == FAST_PAWN) {
                return false
            }
            
            return true
        }

        fun pushMoveBuffer(): MoveBuffer {
            if (bufferStackPointer >= moveBuffers.size) {
                throw IllegalStateException("Move buffer stack overflow - recursion depth $bufferStackPointer exceeds limit")
            }
            return moveBuffers[bufferStackPointer++]
        }
        
        fun popMoveBuffer() {
            if (bufferStackPointer <= 0) {
                throw IllegalStateException("Move buffer stack underflow")
            }
            bufferStackPointer--
        }

        override fun run() {
            try {
                val evaluation = MoveEvaluation(
                    move, 
                    evaluateMove(board.executeOnNewBoard(move).fast(), depth, -NeoBoardEvaluation.MATE_EVAL, NeoBoardEvaluation.MATE_EVAL),
                    depth + 1
                )
                synchronized(lock) {
                    moveEvaluations += evaluation
                    totalEvaluations += evaluationCount
                }
            } catch (e: Throwable) {
                e.printStackTrace()
            }
        }

        private fun evaluateMove(board: FastBoard, depth: Int, alphaParam: Double, betaParam: Double): Double {
            var alpha = alphaParam
            var beta = betaParam

            // Check transposition table first
            val boardHash = zobristHash.computeHash(board)
            val ttResult = transpositionTable.probe(boardHash, depth, alpha, beta)

            if (ttResult != null && ttResult.hit) {
                return ttResult.value
            }

            if (depth <= 0) {
                evaluationCount++
                val value = evaluation.evaluate(board)

                // Store leaf evaluation in transposition table
                // Use special depth for mate positions to prevent eviction
                val storeDepth = if (isMateScore(value)) MATE_TT_DEPTH else 0
                transpositionTable.store(
                    zobristHash = boardHash,
                    depth = storeDepth,
                    value = value,
                    nodeType = TranspositionTable.NodeType.EXACT
                )

                return value
            }

            // Null move pruning - unfreeze current player's frozen piece
            if (useNullMovePruning && 
                nullMoveLevel == 0 &&  // Prevent consecutive null moves
                depth >= nullMoveMinDepth &&
                !board.isInCheck(board.atMove, moveGenerator) &&
                board.getFrozenPieceIndex(board.atMove) != -1) {  // Only if current player has a frozen piece
                
                nullMoveLevel++
                
                // Make null move (unfreeze piece and pass turn)
                val previouslyFrozenIndex = board.makeNullMove()
                
                // Search with reduced depth
                val nullMoveValue = evaluateMove(board, depth - nullMoveReduction, alpha, beta)
                
                // Undo null move (re-freeze piece if needed)
                board.undoNullMove(previouslyFrozenIndex)
                
                nullMoveLevel--
                
                // If null move search fails high, we can prune
                if (nullMoveValue >= beta) {
                    return beta
                }
            }

            var bestEvaluation = -NeoBoardEvaluation.MATE_EVAL * board.atMove.evaluationNumber
            var bestMove: FastMove? = null
            var nodeType = TranspositionTable.NodeType.UPPER_BOUND

            moveGenerator.board = board
            val moveBuffer = pushMoveBuffer()
            moveGenerator.loadMovesToBuffer(board.atMove, moveBuffer)

            // Enhanced move ordering using MoveOrderer
            val orderedMoves = moveOrderer.orderMoves(moveBuffer, board, ttResult?.bestMove, depth)
            val inCheck = board.isInCheck(board.atMove, moveGenerator)

            for(moveIndex in orderedMoves.indices){
                val move = orderedMoves[moveIndex]

                board.executeOnBoard(move) // make move
                
                var rawMoveEvaluation: Double

                if(shouldExtendMove(move, depth)){
                    // Late Move Extensions (LME)
                    // Search at depth + 1
                    lmeCounter++
                    rawMoveEvaluation = evaluateMove(board, depth, alpha, beta)
                    lmeCounter--
                }else if (shouldReduceMove(move, moveIndex, depth, inCheck, alpha, beta)) {
                    // Late Move Reduction (LMR)

                    val reduction = calculateLMRReduction(depth, moveIndex)
                    val reducedDepth = depth - 1 - reduction
                    
                    // Search with reduced depth first
                    rawMoveEvaluation = evaluateMove(board, reducedDepth, alpha, beta)
                    
                    // If the reduced search suggests this move might be good, re-search at full depth
                    val needsFullSearch = if (board.atMove.getOpponent()) { // Original player was white
                        rawMoveEvaluation > alpha
                    } else { // Original player was black  
                        rawMoveEvaluation < beta
                    }
                    
                    if (needsFullSearch) {
                        // Re-search at full depth
                        rawMoveEvaluation = evaluateMove(board, depth - 1, alpha, beta)
                    }
                } else {
                    // Search at full depth (no reduction)
                    rawMoveEvaluation = evaluateMove(board, depth - 1, alpha, beta)
                }
                
                board.revertMove(move) // revert move

                val finalMoveEvaluation =
                    when {
                        rawMoveEvaluation > 1000 -> rawMoveEvaluation - NeoBoardEvaluation.MATE_STEP_LOSS
                        rawMoveEvaluation < -1000 -> rawMoveEvaluation + NeoBoardEvaluation.MATE_STEP_LOSS
                        else -> rawMoveEvaluation
                    }

                if (board.atMove) { // check if white
                    if (finalMoveEvaluation > bestEvaluation) {
                        bestEvaluation = finalMoveEvaluation
                        bestMove = move
                        nodeType = TranspositionTable.NodeType.EXACT
                    }
                    alpha = max(alpha, bestEvaluation)
                    if (alpha >= beta) {
                        nodeType = TranspositionTable.NodeType.LOWER_BOUND
                        bestMove = move
                        // Update move ordering heuristics for cutoff
                        moveOrderer.updateKillerMove(move, depth)
                        moveOrderer.updateHistory(move, depth)
                        break // Beta cutoff
                    }
                } else {
                    if (finalMoveEvaluation < bestEvaluation) {
                        bestEvaluation = finalMoveEvaluation
                        bestMove = move
                        nodeType = TranspositionTable.NodeType.EXACT
                    }
                    beta = min(beta, bestEvaluation)
                    if (beta <= alpha) {
                        nodeType = TranspositionTable.NodeType.UPPER_BOUND
                        bestMove = move
                        // Update move ordering heuristics for cutoff
                        moveOrderer.updateKillerMove(move, depth)
                        moveOrderer.updateHistory(move, depth)
                        break // Alpha cutoff
                    }
                }
            }

            // Store result in transposition table
            // Use special depth for mate positions to prevent eviction
            val storeDepth = if (isMateScore(bestEvaluation)) MATE_TT_DEPTH else depth
            transpositionTable.store(
                zobristHash = boardHash,
                depth = storeDepth,
                value = bestEvaluation,
                nodeType = nodeType,
                bestMove = bestMove
            )

            popMoveBuffer()
            return bestEvaluation
        }

    }

    /**
     * Get transposition table statistics
     */
    fun getTranspositionTableStats(): String {
        return transpositionTable.getUsageStats()
    }

    /**
     * Clear transposition table (useful between games)
     */
    fun clearTranspositionTable() {
        transpositionTable.clear()
    }
}