package de.tadris.flang.network_api.script

import de.tadris.flang.network_api.FlangAPI
import de.tadris.flang.network_api.util.Sha256
import de.tadris.flang_lib.Board
import de.tadris.flang_lib.analysis.AnalysisListener
import de.tadris.flang_lib.analysis.ComputerAnalysis
import de.tadris.flang_lib.analysis.MoveInfo
import de.tadris.flang_lib.bot.Engine
import de.tadris.flang_lib.bot.fast.FastFlangBot
import de.tadris.flang_lib.puzzle.PuzzleGenerator
import java.io.File
import kotlin.math.min

fun main(){
    val fmns = File("doc/games2.txt").readText()
    val credentials = File("doc/auth.txt").readText().split(":")

    val games = fmns.lines().subList(3000, 3200).map { it.trim() }.filter { it.isNotEmpty() }.map { Board.fromFMN(it).getFMN2() }

    val engine = FastFlangBot(6, 8, ttSizeMB = 1024)
    val searcher = object : Engine{
        override fun findBestMove(
            board: Board,
            printTime: Boolean
        ) = engine.findBestMoveIterative(board, printTime, 500)
    }


    val api = FlangAPI("www.tadris.de", 443, "api/flang", useSSL = true, loggingEnabled = true)
    api.login(credentials[0], credentials[1])

    val puzzles = games.flatMapIndexed { index, fmn ->
        println("Processing game ${index + 1}/${games.size}")
        val analysis = ComputerAnalysis(fmn, searcher, object : AnalysisListener {
            override fun onMoveAnalyzed(currentMove: Int, totalMoves: Int, moveInfo: MoveInfo?) {
                println("Move $currentMove/$totalMoves -> ${moveInfo?.depth}")
            }
        })
        analysis.analyze()
        val puzzleGenerator = PuzzleGenerator(analysis.getFullEvaluationDataAfterAnalysis(), engine)
        val puzzles = puzzleGenerator.searchPuzzles()

        puzzles.forEach { puzzle ->
            api.injectPuzzle(puzzle.startFMN, puzzle.puzzleFMN)
        }

        puzzles
    }
    println("Found ${puzzles.size} puzzles:")
    puzzles.forEach {
        println("- $it")
    }
}

private fun generateRandomGames(count: Int, depth: Int): List<String> {
    val engine = FastFlangBot(min(5, depth), depth, ttSizeMB = 1024)
    return buildList {
        repeat(count){
            val board = Board()
            repeat(5){
                val randomMove = board.findAllMoves(board.atMove).random()
                board.executeOnBoard(randomMove)
            }
            while (!board.gameIsComplete()){
                val move = engine.findBestMove(board, printTime = false)
                board.executeOnBoard(move.bestMove.move)
            }
            val fmn = board.getFMN2()
            println("Generated $fmn")
            add(fmn)
        }
    }
}