package de.tadris.flang

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.fast.*
import org.junit.Assert.*
import org.junit.Test

class TestFastBoard {

    @Test
    fun testFastBoardCreation() {
        val board = FastBoard()
        assertEquals(Board.ARRAY_SIZE, board.pieces.size)
        assertEquals(FAST_WHITE, board.atMove)
        assertEquals(-1, board.frozenWhiteIndex)
        assertEquals(-1, board.frozenBlackIndex)
        assertEquals(0, board.moveNumber)
    }

    @Test
    fun testFastBoardCreationWithInvalidSize() {
        try {
            FastBoard(ByteArray(50))
            fail("Should throw IllegalArgumentException for wrong size")
        } catch (e: IllegalArgumentException) {
            assertTrue(e.message!!.contains("Cannot create board with size"))
        }
    }

    @Test
    fun testBoardConversion() {
        val originalBoard = Board()
        val fastBoard = originalBoard.fast()
        
        assertEquals(originalBoard.atMove.fast(), fastBoard.atMove)
        assertEquals(Board.ARRAY_SIZE, fastBoard.pieces.size)
        
        // Check that piece states are properly converted
        for (i in 0 until Board.ARRAY_SIZE) {
            val piece = originalBoard.getAt(i * 2)
            val fastPieceState = fastBoard.getAt(i)
            
            if (piece != null) {
                assertEquals(piece.type.fast(), fastPieceState.getType())
                assertEquals(piece.color.fast(), fastPieceState.getColor())
                assertEquals(piece.state.fast(), fastPieceState.getFrozen())
            } else {
                assertEquals(FAST_NONE, fastPieceState.getType())
            }
        }
    }

    @Test
    fun testGetAndSetAt() {
        val board = FastBoard()
        
        // Test setting and getting by index
        val pieceState = packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL)
        board.setAt(0, pieceState)
        assertEquals(pieceState, board.getAt(0))
        
        // Test clearing
        board.clearAt(0)
        assertEquals(0.toByte(), board.getAt(0))
        
        // Test getting by coordinates
        board.setAt(indexOf(3, 4), pieceState)
        assertEquals(pieceState, board.getAt(3, 4))
    }

    @Test
    fun testUpdateAt() {
        val board = FastBoard()
        
        // Test normal piece update
        board.updateAt(0, FAST_KING, FAST_WHITE)
        val piece = board.getAt(0)
        assertEquals(FAST_KING, piece.getType())
        assertEquals(FAST_WHITE, piece.getColor())
        assertEquals(FAST_NORMAL, piece.getFrozen())
        
        // Test pawn promotion
        board.updateAt(indexOf(0, 7), FAST_PAWN, FAST_WHITE) // white winning Y is 7
        val promotedPiece = board.getAt(indexOf(0, 7))
        assertEquals(FAST_UNI, promotedPiece.getType())
        assertEquals(FAST_WHITE, promotedPiece.getColor())
        assertEquals(FAST_FROZEN, promotedPiece.getFrozen())
        
        // Test clearing with FAST_NONE
        board.updateAt(0, FAST_NONE, FAST_WHITE)
        assertEquals(0.toByte(), board.getAt(0))
    }

    @Test
    fun testFindIndex() {
        val board = FastBoard()
        
        // Place a king
        board.setAt(10, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        assertEquals(10, board.findIndex(FAST_KING, FAST_WHITE))
        assertEquals(-1, board.findIndex(FAST_KING, FAST_BLACK))
        
        // Test findKingIndex
        assertEquals(10, board.findKingIndex(FAST_WHITE))
        assertEquals(-1, board.findKingIndex(FAST_BLACK))
    }

    @Test
    fun testFrozenPieceHandling() {
        val board = FastBoard()
        
        // Test setting frozen piece indices
        board.setFrozenPieceIndex(FAST_WHITE, 10)
        board.setFrozenPieceIndex(FAST_BLACK, 20)
        
        assertEquals(10, board.getFrozenPieceIndex(FAST_WHITE))
        assertEquals(20, board.getFrozenPieceIndex(FAST_BLACK))
        
        // Test unfreezing
        board.setAt(10, packPieceState(FAST_KING, FAST_WHITE, FAST_FROZEN))
        board.unfreezeOnBoard(FAST_WHITE)
        
        val piece = board.getAt(10)
        assertEquals(FAST_NORMAL, piece.getFrozen())
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE))
    }

    @Test
    fun testHasWon() {
        val board = FastBoard()
        
        // Place white king in winning position (y=7)
        board.setAt(indexOf(4, 7), packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        // Place black king not in winning position
        board.setAt(indexOf(3, 3), packPieceState(FAST_KING, FAST_BLACK, FAST_NORMAL))
        
        assertTrue(board.hasWon(FAST_WHITE))
        assertFalse(board.hasWon(FAST_BLACK))
        
        // Test win by elimination
        board.clearAt(indexOf(3, 3)) // Remove black king
        assertTrue(board.hasWon(FAST_WHITE))
        
        // Test no win when no king
        val emptyBoard = FastBoard()
        assertFalse(emptyBoard.hasWon(FAST_WHITE))
        assertFalse(emptyBoard.hasWon(FAST_BLACK))
    }

    @Test
    fun testGameIsComplete() {
        val board = FastBoard()
        
        // Game not complete initially
        assertFalse(board.gameIsComplete())
        
        // Place white king in winning position
        board.setAt(indexOf(4, 6), packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        
        assertTrue(board.gameIsComplete())
    }

    @Test
    fun testExecuteMove() {
        val board = FastBoard()
        
        // Set up initial position
        val fromIndex = indexOf(4, 1)
        val toIndex = indexOf(4, 2)
        val pieceState = packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL)
        board.setAt(fromIndex, pieceState)
        
        // Create move
        val move = packMove(fromIndex, toIndex, pieceState, 0, -1)
        
        // Execute move
        board.executeOnBoard(move)
        
        // Check results
        assertEquals(0.toByte(), board.getAt(fromIndex)) // From square should be empty
        assertEquals(FAST_PAWN, board.getAt(toIndex).getType()) // To square should have pawn
        assertEquals(FAST_WHITE, board.getAt(toIndex).getColor())
        assertEquals(FAST_BLACK, board.atMove) // Should switch to black
        assertEquals(1, board.moveNumber)
    }

    @Test
    fun testRevertMove() {
        val board = FastBoard()
        
        // Set up initial position
        val fromIndex = indexOf(4, 1)
        val toIndex = indexOf(4, 2)
        val pieceState = packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL)
        board.setAt(fromIndex, pieceState)
        
        // Create and execute move
        val move = packMove(fromIndex, toIndex, pieceState, 0, -1)
        board.executeOnBoard(move)
        
        // Revert move
        board.revertMove(move)
        
        // Check results
        assertEquals(pieceState, board.getAt(fromIndex)) // Piece should be back
        assertEquals(0.toByte(), board.getAt(toIndex)) // To square should be empty
        assertEquals(FAST_WHITE, board.atMove) // Should be back to white
        assertEquals(0, board.moveNumber)
    }

    @Test
    fun testGetFBN() {
        val board = FastBoard()
        
        // Place some pieces
        board.setAt(0, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        board.setAt(1, packPieceState(FAST_PAWN, FAST_BLACK, FAST_FROZEN))
        
        val fbn = board.getFBN()
        
        // Check that FBN contains expected characters
        assertTrue(fbn.startsWith("K+"))
        assertTrue(fbn.contains("p-"))
        assertEquals(Board.ARRAY_SIZE * 2, fbn.length) // Each piece has char + state
    }

    @Test
    fun testToString() {
        println(FastBoard())
        println(Board().fast())
    }

    @Test
    fun testEachPiece() {
        val board = FastBoard()
        
        // Place pieces
        board.setAt(0, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        board.setAt(1, packPieceState(FAST_PAWN, FAST_BLACK, FAST_NORMAL))
        board.setAt(2, packPieceState(FAST_ROOK, FAST_WHITE, FAST_NORMAL))
        
        // Count all pieces
        var allPieceCount = 0
        board.eachPiece(null) { _, piece ->
            if (piece.getType() != FAST_NONE) allPieceCount++
        }
        assertEquals(3, allPieceCount)
        
        // Count only white pieces
        var whitePieceCount = 0
        board.eachPiece(FAST_WHITE) { _, piece ->
            if (piece.getType() != FAST_NONE) whitePieceCount++
        }
        assertEquals(2, whitePieceCount)
        
        // Count only black pieces
        var blackPieceCount = 0
        board.eachPiece(FAST_BLACK) { _, piece ->
            if (piece.getType() != FAST_NONE) blackPieceCount++
        }
        assertEquals(1, blackPieceCount)
    }

    @Test
    fun testMoveNumberTracking() {
        val board = FastBoard()
        assertEquals(0, board.moveNumber)
        
        // Set up and execute several moves
        val move1 = packMove(indexOf(0, 1), indexOf(0, 2), 
            packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL), 0, -1)
        board.setAt(indexOf(0, 1), packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL))
        board.executeOnBoard(move1)
        assertEquals(1, board.moveNumber)
        
        val move2 = packMove(indexOf(1, 6), indexOf(1, 5), 
            packPieceState(FAST_PAWN, FAST_BLACK, FAST_NORMAL), 0, -1)
        board.setAt(indexOf(1, 6), packPieceState(FAST_PAWN, FAST_BLACK, FAST_NORMAL))
        board.executeOnBoard(move2)
        assertEquals(2, board.moveNumber)
        
        // Revert moves
        board.revertMove(move2)
        assertEquals(1, board.moveNumber)
        
        board.revertMove(move1)
        assertEquals(0, board.moveNumber)
    }

    @Test
    fun testIsInCheck() {
        val board = FastBoard()
        val moveGenerator = FastMoveGenerator(board, false, 1)
        
        // Set up a simple check scenario: white king at (4, 0), black rook at (4, 7)
        val whiteKingIndex = indexOf(4, 0)
        val blackRookIndex = indexOf(4, 7)
        
        board.setAt(whiteKingIndex, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        board.setAt(blackRookIndex, packPieceState(FAST_ROOK, FAST_BLACK, FAST_NORMAL))
        
        // White king should be in check from black rook
        assertTrue("White king should be in check", board.isInCheck(FAST_WHITE, moveGenerator))
        
        // Black king is not on board, so not in check
        assertFalse("Black should not be in check (no king)", board.isInCheck(FAST_BLACK, moveGenerator))
        
        // Clear the board and test no check scenario
        board.clearAt(blackRookIndex)
        assertFalse("White king should not be in check after removing attacker", 
            board.isInCheck(FAST_WHITE, moveGenerator))
    }

    @Test
    fun testIsInCheckWithMultiplePieces() {
        val board = FastBoard()
        val moveGenerator = FastMoveGenerator(board, false, 1)
        
        // Set up scenario: black king at (0, 7), white horse at (3, 5) - no check
        val blackKingIndex = indexOf(0, 7)
        val whiteHorseIndex = indexOf(3, 5)
        
        board.setAt(blackKingIndex, packPieceState(FAST_KING, FAST_BLACK, FAST_NORMAL))
        board.setAt(whiteHorseIndex, packPieceState(FAST_HORSE, FAST_WHITE, FAST_NORMAL))
        
        // Black king should not be in check from this horse position
        // Horse at (3,5) cannot reach (0,7) with L-shaped moves
        assertFalse("Black king should not be in check from this horse position", 
            board.isInCheck(FAST_BLACK, moveGenerator))
        
        // Move horse to (2, 6) where it can attack the king at (0, 7)
        // Horse move vector (-2, 1) from (2,6) to (0,7)
        board.clearAt(whiteHorseIndex)
        val attackingHorseIndex = indexOf(2, 6)
        board.setAt(attackingHorseIndex, packPieceState(FAST_HORSE, FAST_WHITE, FAST_NORMAL))
        
        // Now black king should be in check
        assertTrue("Black king should be in check from horse at (2, 6)", 
            board.isInCheck(FAST_BLACK, moveGenerator))
    }

    @Test
    fun testIsInCheckNoKing() {
        val board = FastBoard()
        val moveGenerator = FastMoveGenerator(board, false, 1)
        
        // Set up board with no kings
        board.setAt(indexOf(0, 0), packPieceState(FAST_ROOK, FAST_WHITE, FAST_NORMAL))
        board.setAt(indexOf(7, 7), packPieceState(FAST_ROOK, FAST_BLACK, FAST_NORMAL))
        
        // Neither side should be in check if there's no king
        assertFalse("White should not be in check (no king)", board.isInCheck(FAST_WHITE, moveGenerator))
        assertFalse("Black should not be in check (no king)", board.isInCheck(FAST_BLACK, moveGenerator))
    }

    @Test
    fun testNullMoveUnfreezePiece() {
        val board = FastBoard()
        
        // Set up initial state: white to move, no frozen pieces
        assertEquals(FAST_WHITE, board.atMove)
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE))
        assertEquals(0, board.moveNumber)
        
        // Place a white pawn and freeze it
        val pieceIndex = indexOf(2, 3)
        board.setAt(pieceIndex, packPieceState(FAST_PAWN, FAST_WHITE, FAST_FROZEN))
        board.setFrozenPieceIndex(FAST_WHITE, pieceIndex)
        
        // Verify setup
        assertEquals(pieceIndex, board.getFrozenPieceIndex(FAST_WHITE))
        assertTrue("Piece should be frozen", board.getAt(pieceIndex).getFrozen())
        
        // Make null move (should unfreeze the piece)
        val previouslyFrozenIndex = board.makeNullMove()
        
        // Verify null move effects
        assertEquals(pieceIndex, previouslyFrozenIndex)
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE)) // No frozen pieces now
        assertEquals(FAST_NORMAL, board.getAt(pieceIndex).getFrozen()) // Piece should be unfrozen
        assertEquals(FAST_BLACK, board.atMove) // Should switch to black
        assertEquals(1, board.moveNumber) // Move number should increment
        
        // Undo null move
        board.undoNullMove(previouslyFrozenIndex)
        
        // Verify undo effects
        assertEquals(FAST_WHITE, board.atMove) // Back to white
        assertEquals(0, board.moveNumber) // Move number should decrement
        assertEquals(pieceIndex, board.getFrozenPieceIndex(FAST_WHITE)) // Piece should be frozen again
        assertTrue("Piece should be frozen again", board.getAt(pieceIndex).getFrozen())
    }

    @Test
    fun testNullMoveWithNoFrozenPiece() {
        val board = FastBoard()
        
        // Set up initial state: white to move, no frozen pieces
        assertEquals(FAST_WHITE, board.atMove)
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE))
        assertEquals(0, board.moveNumber)
        
        // Make null move when no piece is frozen
        val previouslyFrozenIndex = board.makeNullMove()
        
        // Should return -1 since no piece was frozen
        assertEquals(-1, previouslyFrozenIndex)
        assertEquals(FAST_BLACK, board.atMove) // Should still switch turns
        assertEquals(1, board.moveNumber) // Move number should still increment
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE)) // Still no frozen pieces
        
        // Undo null move
        board.undoNullMove(previouslyFrozenIndex)
        
        // Should restore state correctly even when no piece was unfrozen
        assertEquals(FAST_WHITE, board.atMove)
        assertEquals(0, board.moveNumber)
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE))
    }

    @Test
    fun testNullMovePreservesOtherPlayersFrozenPiece() {
        val board = FastBoard()
        
        // Set up: white has frozen piece, black has frozen piece
        val whitePieceIndex = indexOf(2, 3)
        val blackPieceIndex = indexOf(5, 4)
        
        board.setAt(whitePieceIndex, packPieceState(FAST_PAWN, FAST_WHITE, FAST_FROZEN))
        board.setFrozenPieceIndex(FAST_WHITE, whitePieceIndex)
        
        board.setAt(blackPieceIndex, packPieceState(FAST_ROOK, FAST_BLACK, FAST_FROZEN))
        board.setFrozenPieceIndex(FAST_BLACK, blackPieceIndex)
        
        // Make null move (white's turn)
        val previouslyFrozenIndex = board.makeNullMove()
        
        // White piece should be unfrozen, black piece should remain frozen
        assertEquals(whitePieceIndex, previouslyFrozenIndex)
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE))
        assertEquals(blackPieceIndex, board.getFrozenPieceIndex(FAST_BLACK))
        assertEquals(FAST_NORMAL, board.getAt(whitePieceIndex).getFrozen())
        assertTrue("Black piece should remain frozen", board.getAt(blackPieceIndex).getFrozen())
        
        // Undo null move
        board.undoNullMove(previouslyFrozenIndex)
        
        // Both pieces should be frozen again
        assertEquals(whitePieceIndex, board.getFrozenPieceIndex(FAST_WHITE))
        assertEquals(blackPieceIndex, board.getFrozenPieceIndex(FAST_BLACK))
        assertTrue("White piece should be frozen again", board.getAt(whitePieceIndex).getFrozen())
        assertTrue("Black piece should remain frozen", board.getAt(blackPieceIndex).getFrozen())
    }

}