package de.tadris.flang_lib.script

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.analysis.AnalysisListener
import de.tadris.flang_lib.analysis.ComputerAnalysis
import de.tadris.flang_lib.analysis.MoveInfo
import de.tadris.flang_lib.bot.Engine
import de.tadris.flang_lib.bot.fast.FastFlangBot
import de.tadris.flang_lib.puzzle.PuzzleGenerator
import kotlin.math.min

fun main(){
    val randomGames = generateRandomGames(10, 5)
    val fmns = randomGames.joinToString(separator = "\n") + "\n" + """
        
    """.trimIndent()
    val games = fmns.lines().map { it.trim() }.filter { it.isNotEmpty() }.map { Board.fromFMN(it).getFMN2() }

    val engine = FastFlangBot(5, 8, ttSizeMB = 1024)
    val searcher = object : Engine{
        override fun findBestMove(
            board: Board,
            printTime: Boolean
        ) = engine.findBestMoveIterative(board, printTime, 500)
    }

    val puzzles = games.flatMapIndexed { index, fmn ->
        println("Processing game ${index + 1}/${games.size}")
        val analysis = ComputerAnalysis(fmn, searcher, object : AnalysisListener {
            override fun onMoveAnalyzed(currentMove: Int, totalMoves: Int, moveInfo: MoveInfo?) {
                println("Move $currentMove/$totalMoves -> ${moveInfo?.depth}")
            }
        })
        analysis.analyze()
        val puzzleGenerator = PuzzleGenerator(analysis.getFullEvaluationDataAfterAnalysis(), engine)
        puzzleGenerator.searchPuzzles()
    }
    println("Found ${puzzles.size} puzzles:")
    puzzles.forEach {
        println("- $it")
    }
    puzzles.forEach {
        println("            Puzzle(1, \"${it.startFMN}\", \"${it.puzzleFMN}\", 1300.0, 50),")
    }
}

private fun generateRandomGames(count: Int, depth: Int): List<String> {
    val engine = FastFlangBot(min(5, depth), depth, ttSizeMB = 1024)
    return buildList {
        repeat(count){
            val board = Board()
            repeat(5){
                val randomMove = board.findAllMoves(board.atMove).random()
                board.executeOnBoard(randomMove)
            }
            while (!board.gameIsComplete()){
                val move = engine.findBestMove(board, printTime = false)
                board.executeOnBoard(move.bestMove.move)
            }
            val fmn = board.getFMN2()
            println("Generated $fmn")
            add(fmn)
        }
    }
}