package de.tadris.flang_lib.script

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.bot.*
import de.tadris.flang_lib.bot.fast.FastFlangBot
import de.tadris.flang_lib.bot.fast.SimpleFastEvaluation
import java.io.File
import java.util.concurrent.Executors
import kotlin.math.roundToInt

fun main(){

    println("Loading starting positions")
    /*val startingPositions = File("doc/startingPos.txt")
        .readLines()
        .map { Board.fromFMN(it) }
        .take(50)*/

    val startingPositions = buildList { repeat(50){ add(Board()) } }

    val timePerMove = 2000L // ms

    val lock = Object()

    fun createBot1() = FastFlangBot(4, 10, false, { FastNeoBoardEvaluation() }, threads = 1)
    fun createBot2() = FastFlangBot(4, 10, false, { SimpleFastEvaluation() }, threads = 1)

    class Stats(
        var winsWhite: Int = 0,
        var winsBlack: Int = 0,
        var depths: MutableList<Int> = mutableListOf(),
    )

    val bot1Stats = Stats()
    val bot2Stats = Stats()
    var playedGames = 0

    fun makeGame(board: Board, white: FastFlangBot, black: FastFlangBot, bot1IsWhite: Boolean) {
        val whiteDepths = mutableListOf<Int>()
        val blackDepths = mutableListOf<Int>()

        while(!board.gameIsComplete()){
            val bot = if(board.atMove == Color.WHITE) white else black
            val moveEval = bot.findBestMoveIterative(board, false, maxTimeMs = timePerMove).bestMove
            if(board.atMove == Color.WHITE){
                whiteDepths += moveEval.depth
            }else{
                blackDepths += moveEval.depth
            }
            board.executeOnBoard(moveEval.move)
            if(board.moveNumber > 100){
                println("\nMove number too high: ${board.getFMN2()}")
                break
            }
        }

        synchronized(lock){
            val whiteStats = if(bot1IsWhite) bot1Stats else bot2Stats
            val blackStats = if(bot1IsWhite) bot2Stats else bot1Stats
            when(board.getWinningColor()){
                Color.WHITE -> whiteStats.winsWhite++
                Color.BLACK -> blackStats.winsBlack++
                null -> { }
            }
            whiteStats.depths.addAll(whiteDepths)
            blackStats.depths.addAll(blackDepths)
            playedGames++
        }
    }

    fun test(board: Board){
        makeGame(board.clone(true), createBot1(), createBot2(), true)
        makeGame(board.clone(true), createBot2(), createBot1(), false)
    }

    val executor = Executors.newFixedThreadPool(24)
    startingPositions.forEach {
        executor.submit {
            try {
                test(it)
            }catch (e: Exception){
                e.printStackTrace()
            }
        }
    }

    print("Testing...")
    while (playedGames < startingPositions.size * 2){
        print("\rTesting $playedGames/${startingPositions.size * 2}...")
        Thread.sleep(1000)
    }
    executor.shutdown()

    println("\rFinished testing!    ")

    fun getScore(num: Int, total: Int) = (((num.toDouble() / total) - 0.5) * 2000).roundToInt() / 10.0

    println("Bot 1 score: ${getScore(bot1Stats.winsWhite + bot1Stats.winsBlack, playedGames)} white: ${getScore(bot1Stats.winsWhite, playedGames / 2)} black: ${getScore(bot1Stats.winsBlack, playedGames / 2)}")
    println("Bot 2 score: ${getScore(bot2Stats.winsWhite + bot2Stats.winsBlack, playedGames)} white: ${getScore(bot2Stats.winsWhite, playedGames / 2)} black: ${getScore(bot2Stats.winsBlack, playedGames / 2)}")

    println("====")

    fun depthsFormatted(depths: List<Int>) = buildString {
        append("Avg: ")
        append(depths.sum() / depths.size.toDouble())
        append(" | ")
        append(depths.groupBy { it }.entries.sortedBy { it.key }.associate { it.key to it.value.size })
    }

    println("Avg Depth:")
    println("    Bot 1: " + depthsFormatted(bot1Stats.depths))
    println("    Bot 2: " + depthsFormatted(bot2Stats.depths))

}