package de.tadris.flang_lib.fast

import de.tadris.flang_lib.*

typealias MoveAction = (index: FastBoardIndex) -> Unit

class MoveBuffer(val capacity: Int = 200) {
    private val buffer = LongArray(capacity)
    private var count = 0
    
    fun clear() {
        count = 0
    }
    
    fun add(move: FastMove) {
        if (count >= capacity) {
            throw IllegalStateException("Move buffer overflow: $count >= $capacity")
        }
        buffer[count++] = move
    }

    fun get(index: Int) = buffer[index]
    
    fun toList(): List<FastMove> = buffer.take(count)

    fun size(): Int = count
}

class FastMoveGenerator(var board: FastBoard, val includeOwnPieces: Boolean, val kingRange: Int = 1) {
    
    fun loadMovesToBuffer(color: FastColor?, moveBuffer: MoveBuffer) {
        moveBuffer.clear()
        if(board.gameIsComplete()) return

        board.eachPiece(color){ index, piece ->
            forEachMove(index, piece){ move ->
                moveBuffer.add(move)
            }
        }
    }

    inline fun forEachMove(fromIndex: FastBoardIndex, state: FastPieceState, action: (FastMove) -> Unit){
        forEachTargetLocation(fromIndex, state){ toIndex ->
            action(packMove(fromIndex, toIndex, state, board.getAt(toIndex), board.getFrozenPieceIndex(state.getColor())))
        }
    }

    inline fun forEachTargetLocation(fromIndex: FastBoardIndex, state: FastPieceState, action: MoveAction) {
        if(state.getFrozen()) return
        when(state.getType()){
            FAST_PAWN -> forEachMoveForPawn(fromIndex, state, action)
            FAST_KING -> forEachKingMove(fromIndex, state, action)
            else -> forEachPossibleTargetLocation(fromIndex, state, action)
        }
    }

    inline fun forEachMoveForPawn(fromIndex: FastBoardIndex, state: FastPieceState, action: MoveAction) {
        // Optimized method for pawns

        val color = state.getColor()
        val yDirection = color.evaluationNumber
        val x = fromIndex.x
        val y = fromIndex.y

        if(checkTarget(x, y + yDirection, color)){
            action(indexOf(x, y + yDirection))
        }
        if(checkTarget(x + 1, y + yDirection, color)){
            action(indexOf(x + 1, y + yDirection))
        }
        if(checkTarget(x - 1, y + yDirection, color)){
            action(indexOf(x - 1, y + yDirection))
        }

    }

    inline fun forEachKingMove(fromIndex: FastBoardIndex, state: FastPieceState, action: MoveAction) {
        // Optimized method for kings

        val x = fromIndex.x
        val y = fromIndex.y
        val color = state.getColor()

        for(dx in -kingRange..kingRange){
            for(dy in -kingRange..kingRange){
                if(dx == 0 && dy == 0) continue
                if(checkTarget(x + dx, y + dy, color)){
                    action(indexOf(x + dx, y + dy))
                }
            }
        }
    }

    val targets = IntArray(32){ -1 }
    var targetsSize = 0

    inline fun forEachPossibleTargetLocation(fromIndex: FastBoardIndex, state: FastPieceState, action: MoveAction) {
        val color = state.getColor()
        val type = state.getType()
        if(type.hasDoubleMoves) {
            for(i in targets.indices){
                if(targets[i] != -1){
                    targets[i] = -1
                } else break
            }
            targetsSize = 0
        }

        type.moves.forEach { batch ->
            forEachPossibleTargetLocation(fromIndex, color, batch) { toIndex ->
                var targetAlreadyCalled = false
                if(type.hasDoubleMoves){
                    for(i in 0..<targetsSize){
                        if(targets[i] == toIndex) {
                            targetAlreadyCalled = true
                            break
                        }
                    }
                }

                if(!targetAlreadyCalled){
                    if(type.hasDoubleMoves) {
                        targets[targetsSize] = toIndex
                        targetsSize++
                    }
                    action(toIndex)
                }
            }
        }
    }

    inline fun forEachPossibleTargetLocation(pieceIndex: FastBoardIndex, color: FastColor, batch: Array<Vector>, action: MoveAction) {
        batch.forEach {
            val x = pieceIndex.x + it.x
            val y = pieceIndex.y + it.y
            if(checkTarget(x, y, color)){
                action(indexOf(x, y))
                if(!isEmpty(x, y)){
                    return
                }
            }else{
                return
            }
        }
    }

    /**
     * Checks if a given color piece can go to this target field
     */
    fun checkTarget(x: Int, y: Int, color: FastColor): Boolean {
        return isValid(x, y) && (includeOwnPieces || isEmpty(x, y) || (isWhite(x, y) != color))
    }

    fun isEmpty(x: Int, y: Int): Boolean {
        return board.getAt(x, y).getType() == FAST_NONE
    }

    fun isWhite(x: Int, y: Int): Boolean {
        return board.getAt(x, y).getColor()
    }

    private fun isValid(x: Int, y: Int) = x >= 0 && y >= 0 && x < Board.BOARD_SIZE && y < Board.BOARD_SIZE

}