package de.tadris.flang_lib.fast

/**
 * FastMove Int 64bit = 8 bytes
 * - 1 byte from index (byte 0, bits 0-7)
 * - 1 byte to index (byte 1, bits 8-15)
 * - 1 byte from piece state (byte 2, bits 16-23)
 * - 1 byte to piece state (byte 3, bits 24-31)
 * - 1 byte previously frozen piece index (byte 4, bits 32-39)
 * - 3 bytes unused (bytes 5-7, bits 40-63)
 */
typealias FastMove = Long

// Bit masks for each field (using Long to handle 64-bit operations)
private const val FROM_INDEX_MASK: Long = 0x00000000000000FFL        // bits 0-7
private const val TO_INDEX_MASK: Long = 0x000000000000FF00L          // bits 8-15
private const val FROM_PIECE_STATE_MASK: Long = 0x0000000000FF0000L  // bits 16-23
private const val TO_PIECE_STATE_MASK: Long = 0x00000000FF000000L    // bits 24-31
private const val PREV_FROZEN_INDEX_MASK: Long = 0x000000FF00000000L // bits 32-39

// Bit shifts for each field
private const val FROM_INDEX_SHIFT = 0
private const val TO_INDEX_SHIFT = 8
private const val FROM_PIECE_STATE_SHIFT = 16
private const val TO_PIECE_STATE_SHIFT = 24
private const val PREV_FROZEN_INDEX_SHIFT = 32

/**
 * Creates a FastMove from individual components
 * @param fromIndex source position index (0-255)
 * @param toIndex destination position index (0-255)
 * @param fromPieceState state of piece at source position
 * @param toPieceState state of piece at destination position
 * @param previouslyFrozenPieceIndex index of previously frozen piece (0-254) or -1 if no piece was frozen
 * @return FastMove Long with encoded values
 */
fun packMove(
    fromIndex: FastBoardIndex,
    toIndex: FastBoardIndex,
    fromPieceState: FastPieceState,
    toPieceState: FastPieceState,
    previouslyFrozenPieceIndex: FastBoardIndex
): FastMove {
    var move = 0L

    // Set from index (byte 0)
    move = move or (fromIndex.toLong() and 0xFF)

    // Set to index (byte 1)
    move = move or ((toIndex.toLong() and 0xFF) shl TO_INDEX_SHIFT)

    // Set from piece state (byte 2)
    move = move or ((fromPieceState.toLong() and 0xFF) shl FROM_PIECE_STATE_SHIFT)

    // Set to piece state (byte 3)
    move = move or ((toPieceState.toLong() and 0xFF) shl TO_PIECE_STATE_SHIFT)

    // Set previously frozen piece index (byte 4)
    move = move or ((previouslyFrozenPieceIndex.toLong() and 0xFF) shl PREV_FROZEN_INDEX_SHIFT)

    return move
}

/**
 * Extracts the from index from FastMove
 * @return source position index (0-255)
 */
fun FastMove.getFromIndex(): FastBoardIndex {
    return (this and FROM_INDEX_MASK).toInt()
}

/**
 * Extracts the to index from FastMove
 * @return destination position index (0-255)
 */
fun FastMove.getToIndex(): FastBoardIndex {
    return ((this and TO_INDEX_MASK) shr TO_INDEX_SHIFT).toInt()
}

/**
 * Extracts the from piece state from FastMove
 * @return state of piece at source position
 */
fun FastMove.getFromPieceState(): FastPieceState {
    return ((this and FROM_PIECE_STATE_MASK) shr FROM_PIECE_STATE_SHIFT).toByte()
}

/**
 * Extracts the to piece state from FastMove
 * @return state of piece at destination position
 */
fun FastMove.getToPieceState(): FastPieceState {
    return ((this and TO_PIECE_STATE_MASK) shr TO_PIECE_STATE_SHIFT).toByte()
}

/**
 * Extracts the previously frozen piece index from FastMove
 * @return index of previously frozen piece (0-254) or -1 if no piece was frozen
 */
fun FastMove.getPreviouslyFrozenPieceIndex(): FastBoardIndex {
    val value = ((this and PREV_FROZEN_INDEX_MASK) shr PREV_FROZEN_INDEX_SHIFT).toInt()
    return if (value == 255) -1 else value
}

/**
 * Creates a string representation of the FastMove for debugging
 * @return formatted string showing all components
 */
fun FastMove.toDebugString(): String {
    return "FastMove(fromIndex=${getFromIndex()}, toIndex=${getToIndex()}, " +
            "fromPieceState=0x${getFromPieceState().toString(16).padStart(2, '0')}, " +
            "toPieceState=0x${getToPieceState().toString(16).padStart(2, '0')}, " +
            "previouslyFrozenPieceIndex=${getPreviouslyFrozenPieceIndex()})"
}