package de.tadris.flang.game

import android.app.Activity
import de.tadris.flang.network_api.model.GameConfiguration
import de.tadris.flang.network_api.model.GameInfo
import de.tadris.flang.network_api.model.GamePlayerInfo
import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.bot.FlangBot
import de.tadris.flang_lib.action.Move
import de.tadris.flang_lib.action.Resign
import de.tadris.flang_lib.bot.BoardEvaluation
import de.tadris.flang_lib.bot.FastBoardEvaluation
import de.tadris.flang_lib.bot.FastNeoBoardEvaluation
import de.tadris.flang_lib.bot.NeoBoardEvaluation
import de.tadris.flang_lib.bot.ParallelFlangBot
import de.tadris.flang_lib.bot.fast.FastFlangBot
import kotlinx.coroutines.delay
import kotlin.concurrent.thread
import kotlin.math.min

open class OfflineBotGameController(activity: Activity) : AbstractGameController(activity) {

    companion object {
        const val NAME = "Flangbot Classic"
        const val EASY_STRENGTH = 2
        const val DEFAULT_STRENGTH = 5
        const val MIN_OFFLINE_THINK_TIME = 1000
    }

    protected var color = Color.WHITE // Color of user
    protected var board = Board(Board.DEFAULT_BOARD)

    protected var evaluationFactory: () -> FastBoardEvaluation = { FastNeoBoardEvaluation() }
        set(value) {
            field = value
            updateBot()
        }
    protected var strength = DEFAULT_STRENGTH
        set(value) {
            field = value
            updateBot()
        }
    protected var bot = FastFlangBot(strength, strength, strength > 3, evaluationFactory)
    protected var name = "Flangbot Classic#$strength"

    private fun updateBot(){
        bot = if(strength > 0){
            FastFlangBot(
                minDepth = min(5, strength),
                maxDepth = strength,
                useOpeningDatabase = strength > 3,
                evaluationFactory = evaluationFactory,
                useLME = true,
                lmeMaxExtension = min(strength, 3)
            )
        }else{
            FastFlangBot(5, 20, true, evaluationFactory, useLME = true, lmeMaxExtension = 3)
        }
    }

    fun updateConfiguration(name: String, evaluationFactory: () -> FastBoardEvaluation, strength: Int){
        this.name = name
        this.evaluationFactory = evaluationFactory
        this.strength = strength
        updateGameState()
    }

    override fun onMoveRequested(move: Move, newBoardRequest: Board?, onCancel: (() -> Unit)?) {
        if(newBoardRequest != null){
            // Accept new board
            board = newBoardRequest
        }
        board.executeOnBoard(move)
        callback.onUpdate(move)
        botTurn()
    }

    override fun requestGame() {
        color = if(Math.random() > 0.5) Color.WHITE else Color.BLACK
        updateGameState()
        if(color == Color.BLACK){
            botTurn()
        }
    }

    private fun updateGameState(){
        val player1Info = GamePlayerInfo("Player1", 0f, -1, 0f, false, "")
        val player2Info = GamePlayerInfo(this.name, 0f, -1, 0f, true, "")
        val whiteInfo = if(color == Color.WHITE) player1Info else player2Info
        val blackInfo = if(color == Color.WHITE) player2Info else player1Info
        callback.onGameRequestSuccess(GameInfo(-1, whiteInfo, blackInfo, "", 0, running = true, GameConfiguration(
            isRated = false,
            infiniteTime = true,
            time = 0,
            isBotRequest = false
        ), lastAction = 0, won = 0, spectatorCount = 0), true, color, board.clone(true))
    }

    private fun botTurn(){
        if(board.gameIsComplete()){
            return
        }
        thread {
            val start = System.currentTimeMillis()
            val botMoves = if(strength > 0){
                bot.findBestMove(board)
            }else{
                bot.findBestMoveIterative(board, true, -(strength * 1000L))
            }
            println(botMoves.evaluations)
            println("Evaluations: " + bot.totalEvaluations)
            val diff = System.currentTimeMillis() - start
            Thread.sleep((MIN_OFFLINE_THINK_TIME - diff).coerceAtLeast(0))
            val botMove = botMoves.bestMove.move
            board.executeOnBoard(botMove)
            activity.runOnUiThread {
                callback.onUpdate(botMove)
            }
        }
    }

    override fun resignGame() {
        callback.onUpdate(Resign(color))
    }

    override fun isCreativeGame() = true
}