package de.tadris.flang

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.fast.*
import org.junit.Assert.*
import org.junit.Test

class TestFastBoard {

    @Test
    fun testFastBoardCreation() {
        val board = FastBoard()
        assertEquals(Board.ARRAY_SIZE, board.pieces.size)
        assertEquals(FAST_WHITE, board.atMove)
        assertEquals(-1, board.frozenWhiteIndex)
        assertEquals(-1, board.frozenBlackIndex)
        assertEquals(0, board.moveNumber)
    }

    @Test
    fun testFastBoardCreationWithInvalidSize() {
        try {
            FastBoard(ByteArray(50))
            fail("Should throw IllegalArgumentException for wrong size")
        } catch (e: IllegalArgumentException) {
            assertTrue(e.message!!.contains("Cannot create board with size"))
        }
    }

    @Test
    fun testBoardConversion() {
        val originalBoard = Board()
        val fastBoard = originalBoard.fast()
        
        assertEquals(originalBoard.atMove.fast(), fastBoard.atMove)
        assertEquals(Board.ARRAY_SIZE, fastBoard.pieces.size)
        
        // Check that piece states are properly converted
        for (i in 0 until Board.ARRAY_SIZE) {
            val piece = originalBoard.getAt(i * 2)
            val fastPieceState = fastBoard.getAt(i)
            
            if (piece != null) {
                assertEquals(piece.type.fast(), fastPieceState.getType())
                assertEquals(piece.color.fast(), fastPieceState.getColor())
                assertEquals(piece.state.fast(), fastPieceState.getFrozen())
            } else {
                assertEquals(FAST_NONE, fastPieceState.getType())
            }
        }
    }

    @Test
    fun testGetAndSetAt() {
        val board = FastBoard()
        
        // Test setting and getting by index
        val pieceState = packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL)
        board.setAt(0, pieceState)
        assertEquals(pieceState, board.getAt(0))
        
        // Test clearing
        board.clearAt(0)
        assertEquals(0.toByte(), board.getAt(0))
        
        // Test getting by coordinates
        board.setAt(indexOf(3, 4), pieceState)
        assertEquals(pieceState, board.getAt(3, 4))
    }

    @Test
    fun testUpdateAt() {
        val board = FastBoard()
        
        // Test normal piece update
        board.updateAt(0, FAST_KING, FAST_WHITE)
        val piece = board.getAt(0)
        assertEquals(FAST_KING, piece.getType())
        assertEquals(FAST_WHITE, piece.getColor())
        assertEquals(FAST_NORMAL, piece.getFrozen())
        
        // Test pawn promotion
        board.updateAt(indexOf(0, 7), FAST_PAWN, FAST_WHITE) // white winning Y is 7
        val promotedPiece = board.getAt(indexOf(0, 7))
        assertEquals(FAST_UNI, promotedPiece.getType())
        assertEquals(FAST_WHITE, promotedPiece.getColor())
        assertEquals(FAST_FROZEN, promotedPiece.getFrozen())
        
        // Test clearing with FAST_NONE
        board.updateAt(0, FAST_NONE, FAST_WHITE)
        assertEquals(0.toByte(), board.getAt(0))
    }

    @Test
    fun testFindIndex() {
        val board = FastBoard()
        
        // Place a king
        board.setAt(10, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        assertEquals(10, board.findIndex(FAST_KING, FAST_WHITE))
        assertEquals(-1, board.findIndex(FAST_KING, FAST_BLACK))
        
        // Test findKingIndex
        assertEquals(10, board.findKingIndex(FAST_WHITE))
        assertEquals(-1, board.findKingIndex(FAST_BLACK))
    }

    @Test
    fun testFrozenPieceHandling() {
        val board = FastBoard()
        
        // Test setting frozen piece indices
        board.setFrozenPieceIndex(FAST_WHITE, 10)
        board.setFrozenPieceIndex(FAST_BLACK, 20)
        
        assertEquals(10, board.getFrozenPieceIndex(FAST_WHITE))
        assertEquals(20, board.getFrozenPieceIndex(FAST_BLACK))
        
        // Test unfreezing
        board.setAt(10, packPieceState(FAST_KING, FAST_WHITE, FAST_FROZEN))
        board.unfreezeOnBoard(FAST_WHITE)
        
        val piece = board.getAt(10)
        assertEquals(FAST_NORMAL, piece.getFrozen())
        assertEquals(-1, board.getFrozenPieceIndex(FAST_WHITE))
    }

    @Test
    fun testHasWon() {
        val board = FastBoard()
        
        // Place white king in winning position (y=7)
        board.setAt(indexOf(4, 7), packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        // Place black king not in winning position
        board.setAt(indexOf(3, 3), packPieceState(FAST_KING, FAST_BLACK, FAST_NORMAL))
        
        assertTrue(board.hasWon(FAST_WHITE))
        assertFalse(board.hasWon(FAST_BLACK))
        
        // Test win by elimination
        board.clearAt(indexOf(3, 3)) // Remove black king
        assertTrue(board.hasWon(FAST_WHITE))
        
        // Test no win when no king
        val emptyBoard = FastBoard()
        assertFalse(emptyBoard.hasWon(FAST_WHITE))
        assertFalse(emptyBoard.hasWon(FAST_BLACK))
    }

    @Test
    fun testGameIsComplete() {
        val board = FastBoard()
        
        // Game not complete initially
        assertFalse(board.gameIsComplete())
        
        // Place white king in winning position
        board.setAt(indexOf(4, 6), packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        
        assertTrue(board.gameIsComplete())
    }

    @Test
    fun testExecuteMove() {
        val board = FastBoard()
        
        // Set up initial position
        val fromIndex = indexOf(4, 1)
        val toIndex = indexOf(4, 2)
        val pieceState = packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL)
        board.setAt(fromIndex, pieceState)
        
        // Create move
        val move = packMove(fromIndex, toIndex, pieceState, 0, -1)
        
        // Execute move
        board.executeOnBoard(move)
        
        // Check results
        assertEquals(0.toByte(), board.getAt(fromIndex)) // From square should be empty
        assertEquals(FAST_PAWN, board.getAt(toIndex).getType()) // To square should have pawn
        assertEquals(FAST_WHITE, board.getAt(toIndex).getColor())
        assertEquals(FAST_BLACK, board.atMove) // Should switch to black
        assertEquals(1, board.moveNumber)
    }

    @Test
    fun testRevertMove() {
        val board = FastBoard()
        
        // Set up initial position
        val fromIndex = indexOf(4, 1)
        val toIndex = indexOf(4, 2)
        val pieceState = packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL)
        board.setAt(fromIndex, pieceState)
        
        // Create and execute move
        val move = packMove(fromIndex, toIndex, pieceState, 0, -1)
        board.executeOnBoard(move)
        
        // Revert move
        board.revertMove(move)
        
        // Check results
        assertEquals(pieceState, board.getAt(fromIndex)) // Piece should be back
        assertEquals(0.toByte(), board.getAt(toIndex)) // To square should be empty
        assertEquals(FAST_WHITE, board.atMove) // Should be back to white
        assertEquals(0, board.moveNumber)
    }

    @Test
    fun testGetFBN() {
        val board = FastBoard()
        
        // Place some pieces
        board.setAt(0, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        board.setAt(1, packPieceState(FAST_PAWN, FAST_BLACK, FAST_FROZEN))
        
        val fbn = board.getFBN()
        
        // Check that FBN contains expected characters
        assertTrue(fbn.startsWith("K+"))
        assertTrue(fbn.contains("p-"))
        assertEquals(Board.ARRAY_SIZE * 2, fbn.length) // Each piece has char + state
    }

    @Test
    fun testToString() {
        println(FastBoard())
        println(Board().fast())
    }

    @Test
    fun testEachPiece() {
        val board = FastBoard()
        
        // Place pieces
        board.setAt(0, packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL))
        board.setAt(1, packPieceState(FAST_PAWN, FAST_BLACK, FAST_NORMAL))
        board.setAt(2, packPieceState(FAST_ROOK, FAST_WHITE, FAST_NORMAL))
        
        // Count all pieces
        var allPieceCount = 0
        board.eachPiece(null) { _, piece ->
            if (piece.getType() != FAST_NONE) allPieceCount++
        }
        assertEquals(3, allPieceCount)
        
        // Count only white pieces
        var whitePieceCount = 0
        board.eachPiece(FAST_WHITE) { _, piece ->
            if (piece.getType() != FAST_NONE) whitePieceCount++
        }
        assertEquals(2, whitePieceCount)
        
        // Count only black pieces
        var blackPieceCount = 0
        board.eachPiece(FAST_BLACK) { _, piece ->
            if (piece.getType() != FAST_NONE) blackPieceCount++
        }
        assertEquals(1, blackPieceCount)
    }

    @Test
    fun testMoveNumberTracking() {
        val board = FastBoard()
        assertEquals(0, board.moveNumber)
        
        // Set up and execute several moves
        val move1 = packMove(indexOf(0, 1), indexOf(0, 2), 
            packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL), 0, -1)
        board.setAt(indexOf(0, 1), packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL))
        board.executeOnBoard(move1)
        assertEquals(1, board.moveNumber)
        
        val move2 = packMove(indexOf(1, 6), indexOf(1, 5), 
            packPieceState(FAST_PAWN, FAST_BLACK, FAST_NORMAL), 0, -1)
        board.setAt(indexOf(1, 6), packPieceState(FAST_PAWN, FAST_BLACK, FAST_NORMAL))
        board.executeOnBoard(move2)
        assertEquals(2, board.moveNumber)
        
        // Revert moves
        board.revertMove(move2)
        assertEquals(1, board.moveNumber)
        
        board.revertMove(move1)
        assertEquals(0, board.moveNumber)
    }
}