package de.tadris.flang

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.analysis.AnalysisFormatter
import de.tadris.flang_lib.analysis.AnalysisListener
import de.tadris.flang_lib.analysis.AnalysisResult
import de.tadris.flang_lib.analysis.ComputerAnalysis
import de.tadris.flang_lib.analysis.MoveInfo
import de.tadris.flang_lib.bot.BotResult
import de.tadris.flang_lib.bot.Engine
import de.tadris.flang_lib.bot.FlangBot
import de.tadris.flang_lib.bot.fast.FastFlangBot
import org.junit.Test

class TestComputerAnalysis {

    @Test
    fun testAnalysis(){
        val fmn = "Uf3 b7a6 b3 hc6 b2 ub5 h2g3 e7f6 Uc3 h5 g1 he5 Hf3 fc6 a1 b7 e2e3 kb6 Ph4 hf3 Ff3 h4 g2g3 e3 d2e3 h6 Kg2 h5 Pg4 d7c6 Kg3 fg6 Pf4 c6b5 g5 uh5 g6 g6 g2 pg5 g5 pg5 Rh1 f4 Pg3 c4 Kh2 f5 Pc4 kc6 g2 kd6 Rf1 pg7 Pb3 d4 d4 e5 Kf3 d4 Rf2 d3 Pg4 a7b6 g3 e3 g5 g6 b4 d6 g6 b6b5 c4b5 e5 g7 b5 b5 d3 g8 e3 b6 d3 Ua2 e3 h4 d3 g5 e3 g6 kd4 g7 c4 h8"
        
        val listener = object : AnalysisListener {
            override fun onAnalysisStarted(totalMoves: Int) {
                println("Starting analysis of $totalMoves moves...")
            }
            
            override fun onMoveAnalyzed(currentMove: Int, totalMoves: Int, moveInfo: MoveInfo?) {
                val progress = (currentMove * 100) / totalMoves
                println("Analyzing move $currentMove/$totalMoves ($progress%): ${moveInfo?.moveNotation ?: "unknown"}")
            }
            
            override fun onAnalysisCompleted(result: AnalysisResult) {
                println("Analysis completed in ${result.analysisTimeMs}ms")
            }
        }

        val bot = FastFlangBot(5, 20)
        val analysis = ComputerAnalysis(fmn, object : Engine {
            override fun findBestMove(board: Board, printTime: Boolean) =
                bot.findBestMoveIterative(board, true, 2000)
        }, listener)
        val result = analysis.analyze()
        
        println("=".repeat(50))
        println("ANALYSIS RESULTS")
        println("=".repeat(50))
        
        // Print formatted summary
        println(AnalysisFormatter.formatGameSummary(result))
        println()
        
        // Print detailed move analysis
        println("DETAILED MOVE ANALYSIS:")
        println("-".repeat(50))
        result.moves.forEach { moveInfo ->
            println(AnalysisFormatter.formatMoveInfo(moveInfo))
        }
        
        println()
        println("WHITE PLAYER ANALYSIS:")
        println("-".repeat(30))
        println(AnalysisFormatter.formatPlayerAccuracy(result.whiteAccuracy))
        
        println()
        println("BLACK PLAYER ANALYSIS:")
        println("-".repeat(30))
        println(AnalysisFormatter.formatPlayerAccuracy(result.blackAccuracy))
    }
    
    @Test
    fun testShortGame() {
        // Test with a shorter game for quicker testing
        val shortFmn = "PG2-F3 pE7-F6 HE1-D3 pF8-G7 HD3-F4"
        
        val analysis = ComputerAnalysis(shortFmn, 2)
        val result = analysis.analyze()
        
        println("Short game analysis:")
        println("Moves analyzed: ${result.totalMoves}")
        println("White accuracy: ${String.format("%.1f", result.whiteAccuracy.accuracy)}%")
        println("Black accuracy: ${String.format("%.1f", result.blackAccuracy.accuracy)}%")
        
        // Verify basic functionality
        assert(result.moves.isNotEmpty()) { "Should have analyzed moves" }
        assert(result.whiteAccuracy.totalMoves > 0) { "White should have moves" }
        assert(result.blackAccuracy.totalMoves > 0) { "Black should have moves" }
    }

}