package de.tadris.flang_lib.fast

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.Location
import de.tadris.flang_lib.Piece
import de.tadris.flang_lib.PieceState
import de.tadris.flang_lib.Type
import de.tadris.flang_lib.Vector
import de.tadris.flang_lib.action.Move

fun Board.fast() = FastBoard(
    buildList { eachLocation { add(packPieceState(
        it.piece?.type?.fast() ?: FAST_NONE,
        it.piece?.color?.fast() ?: FAST_COLOR_UNDEFINED, // Default to FAST_WHITE for empty squares (shouldn't matter since type is FAST_NONE)
        it.piece?.state?.fast() ?: FAST_NORMAL,
    )) } }.toByteArray(),
    atMove = atMove.fast(),
    frozenWhiteIndex = findFrozenFigure(Color.WHITE),
    frozenBlackIndex = findFrozenFigure(Color.BLACK),
    moveNumber = moveNumber
)

fun FastBoard.decode() = Board(
    getFBN().toCharArray(),
    atMove.decodeColor(),
    moveNumber = moveNumber
)

fun FastMove.decode(board: Board): Move {
    val from = getFromIndex()
    val to = getToIndex()
    val fromState = getFromPieceState()
    return Move(
        Piece(Location(board, from.x, from.y), fromState.getType().decodeType()!!, fromState.getColor().decodeColor(), if(fromState.getFrozen()) PieceState.FROZEN else PieceState.NORMAL),
        Vector(to.x, to.y)
    )
}

fun Type.fast() = when(this){
    Type.KING -> FAST_KING
    Type.PAWN -> FAST_PAWN
    Type.HORSE -> FAST_HORSE
    Type.ROOK -> FAST_ROOK
    Type.UNI -> FAST_UNI
    Type.FLANGER -> FAST_FLANGER
}

fun FastType.decodeType() = when(this){
    FAST_KING -> Type.KING
    FAST_PAWN -> Type.PAWN
    FAST_HORSE -> Type.HORSE
    FAST_ROOK -> Type.ROOK
    FAST_UNI -> Type.UNI
    FAST_FLANGER -> Type.FLANGER
    FAST_NONE -> null
    else -> throw Exception("Unknown type $this")
}

fun Color.fast() = when(this){
    Color.WHITE -> FAST_WHITE
    Color.BLACK -> FAST_BLACK
}

fun FastColor.decodeColor() = if(this) Color.WHITE else Color.BLACK

fun PieceState.fast() = when(this){
    PieceState.NORMAL -> FAST_NORMAL
    PieceState.FROZEN -> FAST_FROZEN
}

fun Move.fast() = packMove(
    piece.location.fast(),
    target.fast(),
    piece.fast(),
    target.toLocation(piece.location.board).piece.fast(),
    piece.location.board.findFrozenFigure(piece.color)
)

fun Vector.fast(): FastBoardIndex = indexOf(x, y)

fun Piece?.fast(): FastPieceState =
    if(this != null) packPieceState(type.fast(), color.fast(), state.fast())
    else 0