package de.tadris.flang_lib.fast

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.Board.Companion.BOARD_SIZE
import de.tadris.flang_lib.Board.Companion.EMPTY
import de.tadris.flang_lib.Board.FreezeMethod
import de.tadris.flang_lib.Color
import de.tadris.flang_lib.Piece
import de.tadris.flang_lib.PieceState
import de.tadris.flang_lib.Type
import de.tadris.flang_lib.action.Move

/**
 * @property pieces stores [FastPieceState]
 */
class FastBoard(
    val pieces: ByteArray = ByteArray(Board.ARRAY_SIZE),
    var atMove: FastColor = FAST_WHITE,
    var frozenWhiteIndex: FastBoardIndex = -1,
    var frozenBlackIndex: FastBoardIndex = -1,
    var moveNumber: Int = 0,
) {

    init {
        if(pieces.size != Board.ARRAY_SIZE) throw IllegalArgumentException(
            "Cannot create board with size ${pieces.size}. Must be ${Board.ARRAY_SIZE}"
        )
    }

    fun executeOnBoard(move: FastMove) {
        val piece = move.getFromPieceState()
        unfreezeOnBoard(piece.getColor())
        clearAt(move.getFromIndex())
        updateAt(move.getToIndex(), piece.getType(), piece.getColor())
        atMove = atMove.getOpponent()
        moveNumber++
    }

    fun revertMove(move: FastMove){
        val fromPieceState = move.getFromPieceState()
        val toPieceState = move.getToPieceState()
        setAt(move.getFromIndex(), fromPieceState)
        setAt(move.getToIndex(), toPieceState)

        val frozenPieceIndex = move.getPreviouslyFrozenPieceIndex()
        setFrozenPieceIndex(fromPieceState.getColor(), frozenPieceIndex)
        if(frozenPieceIndex != -1){
            val frozenPiece = getAt(frozenPieceIndex)
            setAt(frozenPieceIndex, packPieceState(frozenPiece.getType(), frozenPiece.getColor(), FAST_FROZEN))
        }

        atMove = atMove.getOpponent()
        moveNumber--
    }

    fun unfreezeOnBoard(color: FastColor){
        val index = getFrozenPieceIndex(color)
        if(index == -1) return
        val currentPiece = getAt(index)
        setAt(index, packPieceState(currentPiece.getType(), currentPiece.getColor(), FAST_NORMAL))
    }

    fun getFrozenPieceIndex(color: FastColor): FastBoardIndex = if(color) frozenWhiteIndex else frozenBlackIndex

    fun setFrozenPieceIndex(color: FastColor, index: FastBoardIndex) {
        if(color) {
            frozenWhiteIndex = index
        } else{
            frozenBlackIndex = index
        }
    }

    fun gameIsComplete() = hasWon(FAST_WHITE) || hasWon(FAST_BLACK)

    fun hasWon(color: FastColor): Boolean {
        val index = findKingIndex(color)
        if(index == -1) return false // has no king, so hasn't won
        return if(findKingIndex(color.getOpponent()) == -1){
            true // opponent has no king, so won
        }else index.y == color.winningY
    }

    fun findKingIndex(color: FastColor) = findIndex(FAST_KING, color)

    fun findIndex(type: FastType, color: FastColor): Int {
        for (index in pieces.indices) {
            val element = pieces[index]
            if (element.getType() == type && element.getColor() == color) {
                return index
            }
        }
        return -1
    }

    fun getAt(index: Int): FastPieceState = pieces[index]

    fun getAt(x: Int, y: Int): FastPieceState = pieces[indexOf(x, y)]

    fun updateAt(index: Int, type: FastType, color: FastColor){
        if(type != FAST_NONE){
            val writtenType = if(type == FAST_PAWN && index.y == color.winningY) FAST_UNI else type // pawn promotion
            setAt(index, packPieceState(writtenType, color, writtenType.hasFreeze))
        }else{
            clearAt(index)
        }
    }

    fun clearAt(index: Int){
        setAt(index, 0)
    }

    fun setAt(index: Int, state: FastPieceState){
        pieces[index] = state
        if(index == frozenWhiteIndex){
            val stillFrozen = state.getColor() && state.getFrozen()
            if(!stillFrozen) frozenWhiteIndex = -1
        }else if(index == frozenBlackIndex){
            val stillFrozen = !state.getColor() && state.getFrozen()
            if(!stillFrozen) frozenBlackIndex = -1
        }

        if(state.getFrozen()){
            setFrozenPieceIndex(state.getColor(), index)
        }
    }

    fun getFBN() = buildString {
        eachLocation{ index ->
            val state = getAt(index)
            val char = state.getType().decodeType()?.getChar(state.getColor().decodeColor()) ?: ' '
            append(char)
            append(
                if(state.getFrozen()) '-' else '+'
            )
        }
    }

    override fun toString() = getFBN()

    inline fun eachLocation(action: (FastBoardIndex) -> Unit){
        for(index in 0..pieces.lastIndex){
            action(index)
        }
    }

    inline fun eachPiece(color: FastColor?, action: (FastBoardIndex, FastPieceState) -> Unit){
        for(index in 0..pieces.lastIndex){
            val piece = getAt(index)
            if(piece != 0.toByte() && (color == null || piece.getColor() == color)){
                action(index, piece)
            }
        }
    }

    /**
     * Make a null move (pass turn without making actual move)
     */
    fun makeNullMove() {
        atMove = atMove.getOpponent()
        moveNumber++
    }

    /**
     * Undo a null move
     */
    fun undoNullMove() {
        atMove = atMove.getOpponent()
        moveNumber--
    }

    /**
     * Check if current player has non-pawn material
     * Used for null move pruning - don't use null move in endgames with only pawns
     */
    fun hasNonPawnMaterial(): Boolean {
        val currentColor = atMove
        
        for (i in 0..pieces.lastIndex) {
            val piece = getAt(i)
            if (piece.getType() != FAST_NONE && piece.getColor() == currentColor) {
                val pieceType = piece.getType()
                if (pieceType != FAST_PAWN) {
                    return true
                }
            }
        }
        
        return false
    }

}