package de.tadris.flang_lib.bot.fast

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.bot.FastNeoBoardEvaluation
import de.tadris.flang_lib.bot.NeoBoardEvaluation

/**
 * Test iterative deepening functionality
 */
fun main() {
    println("Testing Iterative Deepening")
    println("=" * 40)
    
    val board = Board.fromFMN("PC1-B2 pB7-A6 PC2-B3 pD7-E6 UG1-F3 hD8-C6")
    println("Board: ${board.getFBN2()}")
    println()
    
    val fastBot = FastFlangBot(
        minDepth = 2,
        maxDepth = 6,
        evaluationFactory = { FastNeoBoardEvaluation() },
        ttSizeMB = 64
    )
    
    // Test 1: Fixed depth vs iterative deepening
    println("1. Fixed depth search (depth 6):")
    fastBot.clearTranspositionTable()
    val result1 = fastBot.findBestMove(board)
    println("Move: ${result1.bestMove.move}, Eval: ${result1.bestMove.evaluation}")
    println()
    
    // Test 2: Iterative deepening with no time limit (should reach max depth)
    println("2. Iterative deepening (no time limit):")
    fastBot.clearTranspositionTable()
    val result2 = fastBot.findBestMoveIterative(board)
    println("Move: ${result2.bestMove.move}, Eval: ${result2.bestMove.evaluation}")
    println("Final depth: ${result2.bestMove.depth}")
    println()
    
    // Test 3: Iterative deepening with time limit
    println("3. Iterative deepening with 2 second time limit:")
    fastBot.clearTranspositionTable()
    val result3 = fastBot.findBestMoveIterative(board, true, 2000)
    println("Move: ${result3.bestMove.move}, Eval: ${result3.bestMove.evaluation}")
    println("Reached depth: ${result3.bestMove.depth}")
    println()
    
    // Test 4: Very short time limit
    println("4. Iterative deepening with 0.5 second time limit:")
    fastBot.clearTranspositionTable()
    val result4 = fastBot.findBestMoveIterative(board, true, 500)
    println("Move: ${result4.bestMove.move}, Eval: ${result4.bestMove.evaluation}")
    println("Reached depth: ${result4.bestMove.depth}")
    
    println("\nIterative deepening test completed!")
}

private operator fun String.times(n: Int): String = this.repeat(n)