package de.tadris.flang_lib.bot

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.fast.FAST_BLACK
import de.tadris.flang_lib.fast.FAST_NONE
import de.tadris.flang_lib.fast.FAST_WHITE
import de.tadris.flang_lib.fast.FastBoard
import de.tadris.flang_lib.fast.FastBoardIndex
import de.tadris.flang_lib.fast.FastColor
import de.tadris.flang_lib.fast.FastMoveGenerator
import de.tadris.flang_lib.fast.evaluationNumber
import de.tadris.flang_lib.fast.getColor
import de.tadris.flang_lib.fast.getType
import de.tadris.flang_lib.fast.indexOf
import de.tadris.flang_lib.fast.pieceValue
import de.tadris.flang_lib.fast.x
import de.tadris.flang_lib.fast.y
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.pow

/**
 * This is equivalent to [NeoBoardEvaluation] but uses [FastBoard].
 */
class FastNeoBoardEvaluation : FastBoardEvaluation {

    var board = FastBoard()

    private val evaluationMatrix = arrayOfNulls<LocationEvaluation>(Board.ARRAY_SIZE)

    init {
        for(i in evaluationMatrix.indices){
            evaluationMatrix[i] = LocationEvaluation()
        }
    }

    private val blackStats = OverallStats()
    private val whiteStats = OverallStats()

    private val moveGenerator = FastMoveGenerator(board, includeOwnPieces = true, kingRange = 2)

    private var kingsEvalNum = 0.0

    private var pieceValueEval = 0.0
        private set

    override fun evaluate(board: FastBoard): Double {
        this.board = board
        this.moveGenerator.board = board
        return evaluate()
    }

    fun evaluate(): Double {
        if(board.hasWon(FAST_WHITE)){
            return NeoBoardEvaluation.MATE_EVAL
        }
        if(board.hasWon(FAST_BLACK)){
            return -NeoBoardEvaluation.MATE_EVAL
        }
        prepare()
        for(index in 0..<Board.ARRAY_SIZE){
            evaluateLocation(index)
        }
        var evaluation = 0.0
        val movementEval = (whiteStats.movements.toDouble() / blackStats.movements) - (blackStats.movements.toDouble() / whiteStats.movements)
        pieceValueEval = (whiteStats.pieceValue.toDouble() / blackStats.pieceValue) - (blackStats.pieceValue.toDouble() / whiteStats.pieceValue)
        kingsEvalNum = getKingsEval()
        val matrixEval = evaluationMatrix.sumOf { it!!.evaluateField() }

        evaluation+=   5 * matrixEval
        evaluation+=  10 * movementEval
        evaluation+=  60 * pieceValueEval
        evaluation+=   1 * kingsEvalNum

        return evaluation
    }

    private fun prepare(){
        evaluationMatrix.forEach {
            it?.reset()
        }
        blackStats.reset()
        whiteStats.reset()
        kingsEvalNum = 0.0
    }

    private fun evaluateLocation(index: FastBoardIndex){
        val eval = getAt(index)
        val piece = board.getAt(index)
        val type = piece.getType()
        if(type != FAST_NONE){
            val color = piece.getColor()
            val stats = getStats(color)
            val value = type.pieceValue
            stats.pieceValue += value
            eval.occupiedBy = (value * color.evaluationNumber).toDouble()
            moveGenerator.forEachTargetLocation(index, piece){ targetIndex ->
                getAt(targetIndex).addThreat(color, 1.0 / value)
                stats.movements++
            }
        }
    }

    private fun getKingsEval(): Double {
        val whiteKing = board.findKingIndex(FAST_WHITE)
        val blackKing = board.findKingIndex(FAST_BLACK)
        if(whiteKing == -1 || blackKing == -1){
            throw IllegalStateException("Cannot find kings in board $board")
        }
        val whiteKingX = whiteKing.x
        val whiteKingY = whiteKing.y
        val blackKingX = blackKing.x
        val blackKingY = blackKing.y

        val whiteEval = 20 * 2.0.pow(whiteKingY - 6)
        val blackEval = 20 * 2.0.pow(-blackKingY + 1)

        for(y in whiteKingY until (Board.BOARD_SIZE - 1)){
            val field = getAt(whiteKingX, y)
            field.weight+= whiteEval
        }
        for(y in blackKingY downTo 1){
            val field = getAt(blackKingX, y)
            field.weight+= blackEval
        }

        return (whiteEval / blackEval) - (blackEval / whiteEval)
    }

    private fun getAt(x: Int, y: Int): LocationEvaluation {
        return getAt(indexOf(x, y));
    }

    private fun getAt(index: Int): LocationEvaluation {
        return evaluationMatrix[index]!!
    }

    private fun getStats(color: FastColor) = if(color) whiteStats else blackStats

    fun evaluateBreakdown() = BoardEvaluationBreakdown(evaluationMatrix)

    class OverallStats(var movements: Int = 1, var pieceValue: Int = 1){

        override fun toString(): String {
            return "pieces=$pieceValue, moves=$movements"
        }

        fun reset(){
            movements = 1
            pieceValue = 1
        }

    }

    class LocationEvaluation(
        var occupiedBy: Double = 0.0,
        var whiteControl: Double = 0.0,
        var blackControl: Double = 0.0,
        var weight: Double = 1.0, // multiplier
        var bonus: Double = 0.0, // bonus for the winning party
    ){

        fun addThreat(color: FastColor, threat: Double){
            if(color){ // test if white
                whiteControl+= threat
            }else{
                blackControl+= threat
            }
        }

        fun evaluateField(): Double {
            val whiteControl = this.whiteControl + 1
            val blackControl = this.blackControl + 1
            val controlRate = (whiteControl / blackControl) - (blackControl / whiteControl)

            val result = when {
                occupiedBy > 0 -> {
                    ((1 + controlRate) * if(blackControl > whiteControl) occupiedBy else 1.0) * weight
                }
                occupiedBy < 0 -> {
                   ((-1 + controlRate) * if(whiteControl > blackControl) abs(occupiedBy) else 1.0) * weight
                }
                else -> {
                    controlRate * weight
                }
            }
            val winningColor = if(result > 0.5) 1 else if(result < -0.5) -1 else 0
            return result + winningColor * bonus
        }

        fun reset(){
            occupiedBy = 0.0
            whiteControl = 0.0
            blackControl = 0.0
            weight = 1.0
            bonus = 0.0
        }

    }

    fun printMatrix(){
        val d = evaluate()
        println("White: $whiteStats")
        println("Black: $blackStats")
        println("+-----------------+")
        for(y in 0 until Board.BOARD_SIZE){
            print("| ")
            for(x in 0 until Board.BOARD_SIZE){
                val s = (evaluationMatrix[y*8 + x]!!.evaluateField() * 2).toInt().toString()
                print(" ".repeat(max(0, 2-s.length)) + s)
            }
            println("|")
        }
        println("+-----------------+")
        println("Rating: $d")
    }

    class BoardEvaluationBreakdown(val evaluationMatrix: Array<LocationEvaluation?>)

}