package de.tadris.flang

import de.tadris.flang_lib.fast.*
import org.junit.Assert.*
import org.junit.Test

class TestFastPieceState {

    @Test
    fun testPackPieceState() {
        val pieceState = packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL)
        
        assertEquals(FAST_KING, pieceState.getType())
        assertEquals(FAST_WHITE, pieceState.getColor())
        assertEquals(FAST_NORMAL, pieceState.getFrozen())
    }

    @Test
    fun testAllPieceTypes() {
        val types = arrayOf(FAST_NONE, FAST_KING, FAST_PAWN, FAST_HORSE, FAST_ROOK, FAST_UNI, FAST_FLANGER)
        
        for (type in types) {
            val pieceState = packPieceState(type, FAST_WHITE, FAST_NORMAL)
            assertEquals(type, pieceState.getType())
            assertEquals(FAST_WHITE, pieceState.getColor())
            assertEquals(FAST_NORMAL, pieceState.getFrozen())
        }
    }

    @Test
    fun testBothColors() {
        val whiteKing = packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL)
        val blackKing = packPieceState(FAST_KING, FAST_BLACK, FAST_NORMAL)
        
        assertEquals(FAST_WHITE, whiteKing.getColor())
        assertEquals(FAST_BLACK, blackKing.getColor())
        assertEquals(FAST_KING, whiteKing.getType())
        assertEquals(FAST_KING, blackKing.getType())
    }

    @Test
    fun testFrozenStates() {
        val normalPiece = packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL)
        val frozenPiece = packPieceState(FAST_KING, FAST_WHITE, FAST_FROZEN)
        
        assertEquals(FAST_NORMAL, normalPiece.getFrozen())
        assertEquals(FAST_FROZEN, frozenPiece.getFrozen())
        assertEquals(FAST_KING, normalPiece.getType())
        assertEquals(FAST_KING, frozenPiece.getType())
        assertEquals(FAST_WHITE, normalPiece.getColor())
        assertEquals(FAST_WHITE, frozenPiece.getColor())
    }

    @Test
    fun testAllCombinations() {
        val types = arrayOf(FAST_NONE, FAST_KING, FAST_PAWN, FAST_HORSE, FAST_ROOK, FAST_UNI, FAST_FLANGER)
        val colors = arrayOf(FAST_WHITE, FAST_BLACK)
        val frozenStates = arrayOf(FAST_NORMAL, FAST_FROZEN)
        
        for (type in types) {
            for (color in colors) {
                for (frozen in frozenStates) {
                    val pieceState = packPieceState(type, color, frozen)
                    assertEquals("Type mismatch for $type, $color, $frozen", type, pieceState.getType())
                    assertEquals("Color mismatch for $type, $color, $frozen", color, pieceState.getColor())
                    assertEquals("Frozen mismatch for $type, $color, $frozen", frozen, pieceState.getFrozen())
                }
            }
        }
    }

    @Test
    fun testZeroPieceState() {
        val emptyState: FastPieceState = 0
        
        assertEquals(FAST_NONE, emptyState.getType())
        assertEquals(FAST_BLACK, emptyState.getColor()) // false = BLACK
        assertEquals(FAST_NORMAL, emptyState.getFrozen()) // false = NORMAL
    }

    @Test
    fun testMaximumValues() {
        // Test with maximum type value (7)
        val maxType: FastType = 7
        val pieceState = packPieceState(maxType, FAST_WHITE, FAST_FROZEN)
        
        assertEquals(maxType, pieceState.getType())
        assertEquals(FAST_WHITE, pieceState.getColor())
        assertEquals(FAST_FROZEN, pieceState.getFrozen())
    }

    @Test
    fun testInvalidTypeThrowsException() {
        try {
            packPieceState(8, FAST_WHITE, FAST_NORMAL) // 8 is outside valid range (0-7)
            fail("Should throw IllegalArgumentException for type > 7")
        } catch (e: IllegalArgumentException) {
            assertTrue(e.message!!.contains("Type must be between 0 and 7"))
        }
        
        try {
            packPieceState(-1, FAST_WHITE, FAST_NORMAL) // -1 is outside valid range (0-7)
            fail("Should throw IllegalArgumentException for type < 0")
        } catch (e: IllegalArgumentException) {
            assertTrue(e.message!!.contains("Type must be between 0 and 7"))
        }
    }

    @Test
    fun testBitMasking() {
        // Test that bits don't interfere with each other
        val whiteFrozenKing = packPieceState(FAST_KING, FAST_WHITE, FAST_FROZEN)
        val blackNormalKing = packPieceState(FAST_KING, FAST_BLACK, FAST_NORMAL)
        
        // Both should have the same type but different color and frozen state
        assertEquals(FAST_KING, whiteFrozenKing.getType())
        assertEquals(FAST_KING, blackNormalKing.getType())
        assertEquals(FAST_WHITE, whiteFrozenKing.getColor())
        assertEquals(FAST_BLACK, blackNormalKing.getColor())
        assertEquals(FAST_FROZEN, whiteFrozenKing.getFrozen())
        assertEquals(FAST_NORMAL, blackNormalKing.getFrozen())
    }

    @Test
    fun testPieceStateIndependence() {
        // Test that different piece states don't interfere
        val pawn = packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL)
        val king = packPieceState(FAST_KING, FAST_BLACK, FAST_FROZEN)
        
        // Each should maintain its own values
        assertEquals(FAST_PAWN, pawn.getType())
        assertEquals(FAST_WHITE, pawn.getColor())
        assertEquals(FAST_NORMAL, pawn.getFrozen())
        
        assertEquals(FAST_KING, king.getType())
        assertEquals(FAST_BLACK, king.getColor())
        assertEquals(FAST_FROZEN, king.getFrozen())
    }

    @Test
    fun testByteValueRange() {
        // Test that byte values are in expected range
        val maxPiece = packPieceState(FAST_FLANGER, FAST_WHITE, FAST_FROZEN)
        
        // Should be a valid byte value
        assertTrue(maxPiece >= Byte.MIN_VALUE)
        assertTrue(maxPiece <= Byte.MAX_VALUE)
        
        // Should preserve all components
        assertEquals(FAST_FLANGER, maxPiece.getType())
        assertEquals(FAST_WHITE, maxPiece.getColor())
        assertEquals(FAST_FROZEN, maxPiece.getFrozen())
    }

    @Test
    fun testCommonGamePieces() {
        // Test creating common game pieces
        val whitePawn = packPieceState(FAST_PAWN, FAST_WHITE, FAST_NORMAL)
        val blackPawn = packPieceState(FAST_PAWN, FAST_BLACK, FAST_NORMAL)
        val whiteKing = packPieceState(FAST_KING, FAST_WHITE, FAST_NORMAL)
        val blackKing = packPieceState(FAST_KING, FAST_BLACK, FAST_NORMAL)
        val frozenRook = packPieceState(FAST_ROOK, FAST_WHITE, FAST_FROZEN)
        
        // Verify all pieces are created correctly
        assertEquals(FAST_PAWN, whitePawn.getType())
        assertEquals(FAST_WHITE, whitePawn.getColor())
        assertEquals(FAST_NORMAL, whitePawn.getFrozen())
        
        assertEquals(FAST_PAWN, blackPawn.getType())
        assertEquals(FAST_BLACK, blackPawn.getColor())
        assertEquals(FAST_NORMAL, blackPawn.getFrozen())
        
        assertEquals(FAST_KING, whiteKing.getType())
        assertEquals(FAST_WHITE, whiteKing.getColor())
        assertEquals(FAST_NORMAL, whiteKing.getFrozen())
        
        assertEquals(FAST_KING, blackKing.getType())
        assertEquals(FAST_BLACK, blackKing.getColor())
        assertEquals(FAST_NORMAL, blackKing.getFrozen())
        
        assertEquals(FAST_ROOK, frozenRook.getType())
        assertEquals(FAST_WHITE, frozenRook.getColor())
        assertEquals(FAST_FROZEN, frozenRook.getFrozen())
    }
}