package de.tadris.flang_lib.bot.fast

import de.tadris.flang_lib.Board
import de.tadris.flang_lib.bot.FastNeoBoardEvaluation
import de.tadris.flang_lib.bot.NeoBoardEvaluation
import de.tadris.flang_lib.bot.ParallelFlangBot

/**
 * Simple test for FastFlangBot to verify transposition table functionality
 */
fun main() {
    println("Testing FastFlangBot with Transposition Tables")
    println("=" * 50)
    
    // Create a default board
    val board = Board.fromFMN("g2g3 uc6 Uf3 b7b6 Kg2 g7 Kh3 h8 Ph4 f3 Ff3 fb5 b2 kb7 Pb3")
    println("Initial board position:")
    println(board.getFBN2())
    println()

    val depth = 6
    
    // Test with different bots
    println("Testing Original vs Fast Bot Performance:")
    
    // Original ParallelFlangBot
    println("\n1. Original ParallelFlangBot:")
    val originalBot = ParallelFlangBot(
        minDepth = depth,
        maxDepth = depth,
        evaluationFactory = { NeoBoardEvaluation(board) }
    )
    
    val start1 = System.currentTimeMillis()
    val result1 = originalBot.findBestMove(board)
    val end1 = System.currentTimeMillis()
    
    println("Original Bot Result:")
    println("Best move: ${result1.bestMove.move}")
    println("Evaluation: ${result1.bestMove.evaluation}")
    println("Evaluations: ${result1.count}")
    println("Time: ${end1 - start1}ms")
    
    // FastFlangBot with transposition tables
    println("\n2. FastFlangBot with Transposition Tables:")
    val fastBot = FastFlangBot(
        minDepth = depth,
        maxDepth = depth,
        evaluationFactory = { FastNeoBoardEvaluation() },
        ttSizeMB = 256
    )
    
    val start2 = System.currentTimeMillis()
    val result2 = fastBot.findBestMove(board)
    val end2 = System.currentTimeMillis()
    
    println("Fast Bot Result:")
    println("Best move: ${result2.bestMove.move}")
    println("Evaluation: ${result2.bestMove.evaluation}")
    println("Evaluations: ${result2.count}")
    println("Time: ${end2 - start2}ms")
    println("TT Stats: ${fastBot.getTranspositionTableStats()}")
    
    // Compare performance
    println("\n3. Performance Comparison:")
    val speedup = (end1 - start1).toDouble() / (end2 - start2).toDouble()
    val evalReduction = (result1.count.toDouble() - result2.count.toDouble()) / result1.count.toDouble() * 100
    
    println("Speedup: ${String.format("%.2f", speedup)}x")
    println("Evaluation reduction: ${String.format("%.1f", evalReduction)}%")
    
    // Test second search (should be much faster due to TT)
    println("\n4. Second search (TT should help):")
    val start3 = System.currentTimeMillis()
    val result3 = fastBot.findBestMove(board)
    val end3 = System.currentTimeMillis()
    
    println("Second search time: ${end3 - start3}ms")
    println("TT Stats: ${fastBot.getTranspositionTableStats()}")
    
    println("\nTest completed successfully!")
}

private operator fun String.times(n: Int): String = this.repeat(n)