//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


#include <alloca.h>
#include <stdlib.h>
#include <resources.h>
#include <machine.h>
#include "uiapi.h"
#include "ui.h"
#include "uimon.h"
#include "c64ui.h"
#include "petui.h"
#include "vic20ui.h"
#include "uicmdline.h"
#include "jnihelpers.h"
#include "logginghelpers.h"
#include "drive.h"
#include "jnihelpers.h"
#include "javascript.h"
#include "c128.h"
#include "gamepad.h"

#define GEORAMsize "GEORAMsize"
#define GEORAM "GEORAM"
#define REU "REU"
#define REUsize "REUsize"

#define UNITIALIZED (-1 )

static char *vic20_memory_map = NULL;
static char* c64_c128_memory_ext = NULL;

static char* ram_banks[] = {"RAMBlock0", "RAMBlock1", "RAMBlock2", "RAMBlock3", "RAMBlock5"};
static int ram_banks_count = sizeof (ram_banks)/sizeof (ram_banks[0]);

static int set_c64_c128_memory_ext_name(const char *val, __attribute__((unused)) void *param) {
    if (c64_c128_memory_ext) {
        lib_free(c64_c128_memory_ext);
    }
    if (val) {
        c64_c128_memory_ext = lib_strdup(val);
    } else {
        c64_c128_memory_ext = NULL;
    }
    int georam = 0;
    int georam_size = 0;
    int reu = 0;
    int reu_size = 0;
    char *work = val ? lib_strdup(val) : NULL;
    if (work) {
        char *blank = strstr(work, " ");
        char *dummy_for_strtol;
        if (blank) {
            int size = (int) strtol(blank + 1, &dummy_for_strtol, 10);
            if (size) {
                *blank = (char) 0;
                if (strcmp(work, REU) == 0) {
                    reu = 1;
                    reu_size = size;
                }
                if (strcmp(work, GEORAM) == 0) {
                    georam = 1;
                    georam_size = size;
                }
            }
        }
        lib_free(work);
    }
    LOGV("REU = %d, REUsize = %d, GEORAM = %d, GEORAMsize = %d", reu, reu_size, georam, georam_size);
    int ret = resources_set_int(GEORAM, georam);
    if (georam_size) {
       ret = ret != 0 ? ret : resources_set_int(GEORAMsize, georam_size);
    }
    if (reu_size) {
        ret = ret != 0 ? ret : resources_set_int(REUsize, reu_size);
    }
    ret = ret != 0 ? ret : resources_set_int(REU, reu);
    return ret;
}
static int set_vic20_memory_map_name(const char *val, __attribute__((unused)) void *param) {
	int reset = 0;
	if (strlen(val) != ram_banks_count) {
		return 1;
	}
	if (vic20_memory_map == NULL) {
		reset = 1;
	} else {
		reset = strcmp(vic20_memory_map, val);
		lib_free(vic20_memory_map);
	}
	vic20_memory_map = lib_strdup(val);
	//strncpy(vic20_memory_map, val, ram_banks_count);
	for (int i=0;i<ram_banks_count;i++) {
		switch(val[i]) {
			case (char)'0':
				resources_set_int(ram_banks[i],0);
				break;
			case (char)'1':
				resources_set_int(ram_banks[i],1);
				break;
			default:
				return 2;
		}
	}
	if (reset) {
		machine_trigger_reset(MACHINE_RESET_MODE_HARD);
	}
	return 0;
}

static int c128_machine_type;

static int set_c128machine_type(int val, __attribute__((unused)) void *param)
{
    int machine_type;
    int ret = resources_get_int("MachineType", &machine_type);
    if  (ret != 0) {
        LOGV("set_c128machine_type, get MachineType returned %d", ret);
    } else {
        LOGV("set_c128machine_type, comparing %d with %d", val, machine_type);
        if (val != machine_type) {
            ret = resources_set_int("MachineType", val);
            if (ret != 0) {
                LOGV("set_c128machine_type, set MachineType returned %d", ret);
            } else {
                static jmethodID mth = NULL;
                if (!mth) {
                    LOGV("set_c128machine_type getting method");
                    mth = GetMethodID(GetObjectClass(CurrentActivity()), "resetToIntlKeyboard",
                                      "()V");
                }
                if (mth) {
                    LOGV("set_c128machine_type calling method");
                    CallVoidMethod(CurrentActivity(), mth);

                }
            }
        }
        LOGV("set_c128machine_type done");
    }
    return ret;
}
static const resource_string_t resources_string_c64_c128_memory_ext[] = {
        { "C64C128MemoryExtension", "00000", RES_EVENT_SAME, NULL,
          &c64_c128_memory_ext, set_c64_c128_memory_ext_name, NULL },
        RESOURCE_STRING_LIST_END
};
static const resource_string_t resources_string_vic[] = {
    { "Vic20MemoryMap", "00000", RES_EVENT_SAME, NULL,
      &vic20_memory_map, set_vic20_memory_map_name, NULL },
    RESOURCE_STRING_LIST_END
};
static const resource_int_t resources_int_c128[] = {
    {"C128MachineType", C128_MACHINE_INT, RES_EVENT_SAME, NULL,
     &c128_machine_type, set_c128machine_type, NULL},
    RESOURCE_INT_LIST_END
};
static int vic20_resources_init(void) {
	return resources_register_string(resources_string_vic);
}

extern int ui_resources_init(void)
{
    LOGV("ui_resources_init() for %s, cmp %d", machine_get_name(), strcmp(machine_get_name(), "C128"));
    if (strcmp(machine_get_name(), "VIC20") == 0) {
        return vic20_resources_init();
    }
    if (strcmp(machine_get_name(), "C64") == 0) {
        return resources_register_string(resources_string_c64_c128_memory_ext);
    }
    if (strcmp(machine_get_name(), "C128") == 0) {
        int ret = resources_register_int(resources_int_c128);
        ret = ret != 0 ? ret :resources_register_string(resources_string_c64_c128_memory_ext);
        return ret;
    }
    return 0;
}
extern void ui_resources_shutdown(void)
{

}
extern int ui_cmdline_options_init(void)
{
	return 0;
}
extern t_shared_data* g_shareddata;
static char* default_useropts=NULL;
void set_initial_resource_filename(const char* value);
extern int ui_init(__unused int *argc, __unused char **argv)
{
	t_shared_data* shared_data = g_shareddata;
	jmethodID mth_get_useropts=GetMethodID(GetObjectClass(shared_data->instance),"getUseropts","()Ljava/lang/String;");
	jstring juseropts=CallObjectMethod(shared_data->instance,mth_get_useropts);
	const char* s = GetStringUTFChars((jstring) juseropts, NULL);

	if (s) {
		default_useropts = lib_strdup(s);
		set_initial_resource_filename(default_useropts);
	}
	ReleaseStringUTFChars(juseropts, s);
	return 0;
}

extern int ui_init_finish(void)
{
	return 0;
}
void vsyncarch_init(void);
extern int ui_init_finalize(void)
{
	jmethodID  mth=GetMethodID(GetObjectClass(CurrentActivity()),"uiInitFinalize","()V");
	CallVoidMethod(CurrentActivity(),mth);
	vsyncarch_init();
	return 0;
}
extern void ui_shutdown(void)
{
	if (default_useropts) {
		lib_free(default_useropts);
		default_useropts = NULL;
	}
	js_cleanup();
}
/* Print an error message.  */
extern void ui_error(__unused const char *format,...)
{
    va_list args;
    va_start(args, format);
    __android_log_vprint(ANDROID_LOG_ERROR, "ViceEmulation (JNI", format, args);
    va_end(args);
}

/* Let the user browse for a filename; display format as a titel */
extern char* ui_get_file(__unused const char *format,...)
{
	return 0;
}

/* Drive related UI.  */
extern void ui_enable_drive_status(__attribute__((unused)) ui_drive_enable_t state,
								   __unused int *drive_led_color)
{
}
extern void ui_display_drive_track(__unused unsigned int drive_number,
								   __unused unsigned int drive_base,
								   __unused unsigned int half_track_number)
{
}

/* The pwm value will vary between 0 and 1000.  */
extern void ui_display_drive_led(unsigned int drive_number, __attribute__((unused)) unsigned int drive_base, unsigned int led_pwm1,
                                 __attribute__((unused)) unsigned int led_pwm2)
{
	static jmethodID mth=NULL;
	static int oldPwm1[DRIVE_UNIT_MAX - DRIVE_UNIT_MIN + 1];
	static int oldPwm1_initialized=0;
	if (!mth) {
		mth = GetMethodID(GetObjectClass(CurrentActivity()), "uiDisplayDriveLed", "(II)V");
	}
	if (!oldPwm1_initialized)
	{
		memset(oldPwm1, UNITIALIZED, sizeof(oldPwm1) / sizeof(oldPwm1[0]));
		oldPwm1_initialized = 1;
	}
	//LOGV("ui_display_drive_led (%d, %d, %d, %d", drive_number, drive_base, led_pwm1, led_pwm2);
	if (oldPwm1[drive_number] > 0 ? 1 : 0 != led_pwm1 > 0 ? 1 : 0)
	{
		CallVoidMethod(CurrentActivity(), mth, DRIVE_UNIT_MIN + drive_number, led_pwm1);
		oldPwm1[drive_number]= led_pwm1 > 0 ? 1 : 0;
	}

}

extern void ui_display_drive_current_image(__unused unsigned int unit_number, __unused unsigned int drive_number,
										   __unused const char *image)
{
}

extern int ui_extend_image_dialog(void)
{
	return 0;
}

/* Tape related UI */
//useless
extern void ui_set_tape_status(__unused int tape_status)
{
}
extern void ui_display_tape_motor_status(int motor)
{
	static jmethodID mth=NULL;
	if (!mth)
	{
		mth = GetMethodID(GetObjectClass(CurrentActivity()), "uiDisplayTapemotorStatus", "(I)V");
	}
	CallVoidMethod(CurrentActivity(),mth,motor);

}
extern void ui_display_tape_control_status(int control)
{
	static int oldControl=UNITIALIZED;
	static jmethodID mth=NULL;
	if (!mth)
	{
		mth = GetMethodID(GetObjectClass(CurrentActivity()), "uiDisplayTapecontrolStatus", "(I)V");
	}
	if (control!=oldControl) {
		CallVoidMethod(CurrentActivity(), mth, control);
		oldControl = control;
	}
}
extern void ui_display_tape_counter(int counter)
{
	static int oldCounter=UNITIALIZED;
	static jmethodID mth=NULL;
	if (!mth)
	{
		mth = GetMethodID(GetObjectClass(CurrentActivity()), "uiDisplayTapecounter", "(I)V");
	}
	if (counter != oldCounter) {
		CallVoidMethod(CurrentActivity(), mth, counter);
		oldCounter = counter;
	}

}
extern void ui_display_tape_current_image(__unused const char *image)
{
}

/* Show a CPU JAM dialog.  */
extern ui_jam_action_t ui_jam_dialog(const char *format, ...)
{
	(*getAactiveenv())->GetObjectClass(getAactiveenv(),CurrentActivity());
	static jmethodID mth=NULL;
	char* msg=strdup(format);
	jstring jstr=NULL;
	if (!mth) {
		mth = (*getAactiveenv())->GetMethodID(getAactiveenv(),
											  (*getAactiveenv())->GetObjectClass(getAactiveenv(),
																				 CurrentActivity()),
											  "postError", "(Ljava/lang/String;)V");
	}
	jstr=(*getAactiveenv())->NewStringUTF(getAactiveenv(),msg);
	(*getAactiveenv())->CallVoidMethod(getAactiveenv(),CurrentActivity(),mth,jstr);
	(*getAactiveenv())->ReleaseStringUTFChars(getAactiveenv(),jstr,msg);
	return UI_JAM_HARD_RESET;
}

/* Recording UI */
extern void ui_display_playback(__unused int playback_status, __unused char *version)
{
}
extern void ui_display_recording(__unused int recording_status)
{
}
extern void ui_display_event_time(__unused unsigned int current, __unused unsigned int total)
{
}
/* Joystick UI */
extern void ui_display_joyport(__unused BYTE *joyport)
{
}
/* Volume UI */
void ui_display_volume(__unused int vol)
{
}


extern struct console_s *uimon_window_open(void)
{
	return NULL;
}
extern void uimon_window_suspend(void)
{
}
extern struct console_s *uimon_window_resume(void)
{
	return NULL;
}
extern void uimon_window_close(void)
{

}

extern int uimon_out(__unused const char *buffer)
{
	return 0;
}
extern void uimon_notify_change(void)
{

}
extern void uimon_set_interface(__unused struct monitor_interface_s ** pmi, __unused int i)
{

}
extern char *uimon_get_in(__unused char ** pc, __unused const char *c)
{
	return "";
}
int c64ui_init_early(void)
{
	return 0;
}
int c128ui_init_early(void)
{
    return 0;
}

extern int vic20ui_init_early(void)
{
	return 0;
}
extern int vic20ui_init(void)
{
	return 0;
}
extern void vic20ui_shutdown(void)
{

}
extern int c128ui_init(void) {
    return 0;
}
extern void c128ui_shutdown(void) {

}

extern int petui_init_early(void) {
    return 0;
}
extern int petui_init(void) {
    return 0;
}
extern void petui_shutdown(void) {

}
extern int plus4ui_init_early(void) {
    return 0;
}
extern int plus4ui_init(void) {
    return 0;
}
extern void plus4ui_shutdown(void) {

}
