//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------

#include <time.h>
#include <string.h>
#include "timemachine.h"
#include "lib.h"
#include "machine.h"
#include "fmemopen.h"
#include "interrupt.h"
#include "jnihelpers.h"
#include "logginghelpers.h"
#include "temporary_snapshot.h"
#include "vsync.h"
#include "timemachine.h"

timemachine* active_timemachine = NULL;

static void save_temporary_snapshot_trap(__unused WORD unused_addr, void* data) {
    char memfile[64];
    moment *target;
    static int tmpbuffersize=1;
    snprintf(memfile, sizeof(memfile) - 1, "malloc::%d@temporary_snapshot", tmpbuffersize);
    int ret = machine_write_snapshot(memfile, 1, 1, 0);
    //LOGV("Snapshot-Chain: Saving Snapshot to %p returned %d", data, ret);
    if (ret == 0) {
        long size;
        void *temporary_data;
        get_kept_data("temporary_snapshot", &size, &temporary_data);
        target = (moment *) data;
        if (size > target->alloc_size) {
            if (target->data) {
                target->data = lib_realloc(target->data, size);
            } else {
                target->data = lib_malloc(size);
            }
            target->alloc_size = size;
        }
        target->size = size;
        if (size>tmpbuffersize) {
            tmpbuffersize = size;
        }
        //LOGV("Snapshot Chain: allocated %ud bytes for snapshot %p at %p and copied from %p", size, target, target->data, temporary_data);
        memcpy(target->data, temporary_data, size);
        target->valid = 1;
        clean_kept_data("temporary_snapshot");
    } else {
        LOGE("wrintgmachine_write_snapshot (%s) returned %d", memfile, ret);
    }
}

typedef struct {
    char* filename;
    jobject runnable;
    jobject runnable2;
    JNIEnv *env;
} travel_data;
static void timetravel_snapshot_trap(__unused WORD unused_addr, void* p)
{
    travel_data* data = (travel_data*)p;
    int ret = load_snapshot_trap(unused_addr,lib_strdup(data->filename));
    LOGV("Snapshot-Chain: Loading Snapshot from %p returned %d", data, ret);
    callRun(data->env, data->runnable);
    lib_free(data->filename);
    lib_free(data);
}

static void load_temporary_snapshot_trap(__unused WORD unused_addr, void* data) {
    moment* source = (moment*) data;
    char filename[1024];
    snprintf(filename,1024,"memmap::%p:%lu",source->data, (long)source->size);


}
static char* timemachine_get_moment_path(moment* source) {
    char* filename;
    interrupt_maincpu_trigger_trap(load_temporary_snapshot_trap, source);
    filename = lib_malloc(1024);
    snprintf(filename,1024,"memmap::%p:%d",source->data, (int)source->size);
    filename = lib_realloc(filename, strlen(filename)+1); // NOLINT(*-suspicious-realloc-usage)
    return filename;
}
static void set_screendata(screenshot* target, int w, int h, int d, jbyte* data) {
    if (w*h*d > target->widht * target->height * target->depth) {
        if (target->pixeldata) {
            target->pixeldata = lib_realloc(target->pixeldata, w * h * d / 8);
        } else {
            target->pixeldata = lib_malloc(w * h * d / 8);
        }
    }
    target->widht = w;
    target->height = h;
    target->depth = d;
    memcpy(target->pixeldata,data, w*h*d/8);

}

static void tm_shutdown(void* p) {
    timemachine* t = (timemachine*)p;
    for (int i = 0; i < t->_ringbuffer_size; i++) {
        moment* m = t->_ringbuffer+i;
        if (m->data) {
            lib_free(m->data);
        }
        screenshot* s = &(m->screen);
        if (s->pixeldata) {
            lib_free(s->pixeldata);
        }
    }
    if (t->_ringbuffer) {
        lib_free(t->_ringbuffer);
    }
    if (t->_last_tick_time) {
        lib_free(t->_last_tick_time);
    }
    if (t->_screenshot.pixeldata) {
        lib_free(t->_screenshot.pixeldata);
    }
    lib_free(p);
}
static void tm_set_paused(void* p, int paused) {
    timemachine* t = (timemachine*)p;
    t->_ispaused = paused;
    if (!paused) {
        gettimeofday(active_timemachine->_last_tick_time, NULL);
    }
}

static void tm_tick(void* p) {
    timemachine* t = (timemachine*)p;

    struct timeval now;
    moment* createdmoment;
    if (!t->_ispaused) {
        gettimeofday(&now, NULL);
        if (t->_first_after_timetravel) {
            memcpy(t->_last_tick_time, &now, sizeof(struct timeval));
            t->_first_after_timetravel = 0;
        } else {
            uint64_t now_millis = (now.tv_sec * (uint64_t) 1000) + (now.tv_usec / 1000);
            uint64_t last_millis =
                    (t->_last_tick_time->tv_sec * (uint64_t) 1000)
                    + (t->_last_tick_time->tv_usec / 1000);
            active_timemachine->_elapsed_time += (now_millis - last_millis);
        }
        if (!active_timemachine->_counter) {
            if (!(maincpu_int_status->global_pending_int & IK_TRAP)) {
                createdmoment = active_timemachine->_ringbuffer + active_timemachine->_ringbuffer_cursor;
                createdmoment->valid=0;
                interrupt_maincpu_trigger_trap(save_temporary_snapshot_trap, (void*)createdmoment);
                //createdmoment->valid=1;
                createdmoment->timestamp = active_timemachine->_elapsed_time;
                screenshot src = active_timemachine->_screenshot;
                set_screendata(&(createdmoment->screen),src.widht, src.height, src.depth, src.pixeldata);
                active_timemachine->_counter = (int) (vsync_get_refresh_frequency() * active_timemachine->_snapshot_interval / 1000);
                active_timemachine->_ringbuffer_cursor = (active_timemachine->_ringbuffer_cursor + 1) % active_timemachine->_ringbuffer_size; // NOLINT(cppcoreguidelines-narrowing-conversions)
            }
        } else {
            active_timemachine->_counter--;
        }
    }
    gettimeofday(active_timemachine->_last_tick_time, NULL);
}

static void tm_travel(void* p, int minimal_elapsed_seconds, jobject runnable) {
    timemachine* t = (timemachine*)p;
    moment* m;
    moment* found = NULL;
    for (int i = 0; i < t->_ringbuffer_size; i++) {
        int pos = (active_timemachine->_ringbuffer_cursor + i) % active_timemachine->_ringbuffer_size;
        m = t->_ringbuffer + pos;
        if (m->valid && m->timestamp <= active_timemachine->_elapsed_time - minimal_elapsed_seconds) {
            found = m;
        }
    }
    if (found) {
        //LOGV("travelling %d seconds from %llu to %llu", minimal_elapsed_seconds, active_timemachine->_elapsed_time, found->timestamp);
        travel_data *data = lib_malloc(sizeof(travel_data));
        data->filename = timemachine_get_moment_path(found);
        data->env = getAactiveenv();
        data->runnable = (*data->env)->NewGlobalRef(data->env, runnable);
        interrupt_maincpu_trigger_trap(timetravel_snapshot_trap, data);
        for (int i = 0; i < active_timemachine->_ringbuffer_size; i++) {
            moment *m2 = active_timemachine->_ringbuffer + i;

            //LOGV("Checking %d with %llu vs. %llu", i, m2->timestamp, found->timestamp);
            if (m2->timestamp > found->timestamp) {
                //LOGV("invalidating %d with %llu", i, m2->timestamp);
                m2->valid = 0;
            }

        }
    }
    struct timeval now;
    memcpy(t->_last_tick_time, &now, sizeof(struct timeval));
    for (int i = 0; i < active_timemachine->_ringbuffer_size; i++) {
        moment *m2 = active_timemachine->_ringbuffer + i;
        if (m2->valid) {
            m2->timestamp+=minimal_elapsed_seconds;
        }
    }
    t->_first_after_timetravel=1;
}
static moment* get_moment(void* p, int minimal_elapsed_time) {
    timemachine* t = (timemachine*)p;
    moment* m;
    moment* found = NULL;
    for (int i = 0; i < t->_ringbuffer_size; i++) {
        int pos = (active_timemachine->_ringbuffer_cursor + i) % t->_ringbuffer_size;
        m = t->_ringbuffer + pos;
        if (m->valid && m->timestamp <= t->_elapsed_time - minimal_elapsed_time) {
            found = m;
        }
    }
    return found;

}
static int tm_write_entry(void* p, int ellapsed_time, const char* path) {
    timemachine* t = (timemachine*)p;
    size_t written = 0;
    moment* m = get_moment(t, ellapsed_time);
    if (m) {
        FILE* f = fopen(path, "wb");
        if (f) {
            written = fwrite(m->data, m->size, 1, f);
            fclose(f);
        }
    }
    return written == 1 ? 0 : -1;
}
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wformat"
static int tm_write_newest_entry(void* p, const char* path) {
    timemachine* t = (timemachine*)p;
    int newest_pos=0;
    uint64_t compare = 0;
    for (int i = 0; i < t->_ringbuffer_size; i++) {
        if (t->_ringbuffer[i].valid && t->_ringbuffer[i].timestamp > compare) {
            compare = t->_ringbuffer[i].timestamp;
            newest_pos = i;
        }
    }
    int pos = newest_pos;
    while (1) {
        if (t->_ringbuffer[pos].valid && t->_elapsed_time > t->_ringbuffer[pos].timestamp) {
            FILE *f = fopen(path, "wb");
            size_t written = fwrite((t->_ringbuffer + pos)->data, (t->_ringbuffer + pos)->size, 1,
                                    f);
            fclose(f);
            if (written == 1) {
                LOGV("written timestamp from %llu ms ago to %s.", t->_elapsed_time - t->_ringbuffer[pos].timestamp, path);
                return 0;
            }
        }
        pos = (pos -1) % t->_ringbuffer_size;
        if (pos == newest_pos) {
            return -1;
        }
    }
}
#pragma clang diagnostic pop
static void tm_set_current_screendata(void* p, int w, int h, int d, jbyte* data) {
    timemachine* t = (timemachine*)p;
    set_screendata(&(t->_screenshot),w,h,d,data);
}
__attribute__((used)) void timemachine_init(int number_of_snapshots, int interval_in_ms) {
    if (active_timemachine) {
        active_timemachine->shutdown(active_timemachine);
    }
    active_timemachine = lib_malloc(sizeof(timemachine));
    memset(active_timemachine, 0, sizeof(timemachine));
    active_timemachine->_snapshot_interval = interval_in_ms;
    active_timemachine->_ringbuffer_size = number_of_snapshots;
    active_timemachine->_ringbuffer=lib_malloc(sizeof(moment) * active_timemachine->_ringbuffer_size);
    memset(active_timemachine->_ringbuffer, 0, sizeof(moment) * active_timemachine->_ringbuffer_size);
    active_timemachine->_last_tick_time = lib_malloc(sizeof(struct timeval));
    gettimeofday(active_timemachine->_last_tick_time, NULL);
    active_timemachine->set_paused=tm_set_paused;
    active_timemachine->travel=tm_travel;
    active_timemachine->tick = tm_tick;
    active_timemachine->shutdown = tm_shutdown;
    active_timemachine->write_newest_entry=tm_write_newest_entry;
    active_timemachine->set_current_screendata = tm_set_current_screendata;
    active_timemachine->write_entry = tm_write_entry;


}
__attribute__((used)) void timemachine_tick() {
    if (active_timemachine) {
        active_timemachine->tick(active_timemachine);
    }
}

__attribute__((used)) void timemachine_set_paused (int paused) {
    if (active_timemachine) {
        active_timemachine->set_paused(active_timemachine, paused);
    }
}
__attribute__((used)) int timemachine_is_paused () {
    if (active_timemachine) {
        return active_timemachine->_ispaused;
    } else {
        return 0;
    }
}
__attribute__((used)) void timemachine_travel(int minimal_elapsed_seconds, jobject runnable) {
    if (active_timemachine) {
        active_timemachine->travel(active_timemachine, minimal_elapsed_seconds, runnable);
    }
}

__attribute__((used)) int oldest_moment_elapsed_time() {

    if (active_timemachine) {
        //LOGV("oldest_moment_elapsed_time starting with %llu", active_timemachine->_elapsed_time);
        uint64_t now = active_timemachine->_elapsed_time; // active_timemachine->_ringbuffer[active_timemachine->_ringbuffer_cursor].timestamp;
        moment* found = NULL;
        for (int i = 0; i< active_timemachine->_ringbuffer_size; i++) {
            /*
            LOGV("moment [%d] is%s valid has timestamp %llu", i,
                 active_timemachine->_ringbuffer[i].populated ? "" : " not",
                 active_timemachine->_ringbuffer[i].timestamp);
            */
            moment* m = active_timemachine->_ringbuffer+i;
            if (m->valid) {
                if (!found) {
                    found = m;
                    //LOGV("found initialized with [%d], %llu", i, m->timestamp);
                } else {
                    if (found->timestamp > m->timestamp) {
                        found = m;
                        //LOGV("found updated with [%d], %llu", i, m->timestamp);
                    }
                }
            }
        }
        if (found) {
            return (int)(now - found->timestamp);
        } else {
            return 0;
        }
    }
    return 0;
}
__attribute__((used)) int newest_moment_elapsed_time() {
    if (active_timemachine) {
        uint64_t compare = 0;
        for (int i = 0; i < active_timemachine->_ringbuffer_size; i++) {
            if (active_timemachine->_ringbuffer[i].valid && active_timemachine->_ringbuffer[i].timestamp > compare) {
                compare = active_timemachine->_ringbuffer[i].timestamp;
            }
        }
        return (int)(active_timemachine->_elapsed_time - compare);
    }
    return 0;
}

__attribute__((used)) int timemachine_get_width(int ellapsed_time) {
    if (active_timemachine) {
        moment* m = get_moment(active_timemachine, ellapsed_time);
        if (m) {
            return m->screen.widht;
        }
    }
    return 0;
}
__attribute__((used)) int timemachine_get_height(int ellapsed_time) {
    if (active_timemachine) {
        moment* m = get_moment(active_timemachine, ellapsed_time);
        if (m) {
            return m->screen.height;
        }
    }
    return 0;
}
__attribute__((used)) int timemachine_get_depth(int ellapsed_time) {
    if (active_timemachine) {
        moment* m = get_moment(active_timemachine, ellapsed_time);
        if (m) {
            return m->screen.depth;
        }
    }
    return 0;
}

__attribute__((used)) void timemachine_get_pixels(int ellapsed_time, jbyte** data, int* size) {
    if (active_timemachine) {
        moment* m = get_moment(active_timemachine, ellapsed_time);
        if (m) {
            *data = m->screen.pixeldata;
            *size = m->screen.widht*m->screen.height*m->screen.depth/8;
        }
    }
}
__attribute__((used)) int timemachine_store_entry(int ellapsed_time, const char* path) {
    if (active_timemachine) {
        if (ellapsed_time < 0) {
            return active_timemachine->write_newest_entry(active_timemachine, path);
        } else {
            return active_timemachine->write_entry(active_timemachine, ellapsed_time, path);
        }
    }
    return -1;
}

__attribute__((used)) int timemachine_store_last_entry(const char* path) {
    if (active_timemachine) {
        return active_timemachine->write_newest_entry(active_timemachine, path);
    }
    return -1;
}
static void timemachine_store_new_entry_trap(__unused WORD unused_addr, void* p) {
    LOGV("snapshot: timemachine_store_new_entry");

    travel_data *data = (travel_data *) p;
    data->env = getAactiveenv();
    moment* createdmoment = lib_malloc(sizeof (moment));
    memset(createdmoment,0,sizeof(moment));
    LOGV("Saving moment");
    save_temporary_snapshot_trap(0, createdmoment);
    size_t written = 0;
    if (createdmoment) {
        FILE* f = fopen(data->filename, "wb");
        LOGV("Writing moment");
        if (f) {
            written = fwrite(createdmoment->data, createdmoment->size, 1, f);
            fclose(f);
        }
        if (written == 1) {
            LOGV("Success, written to %s, calling runnable 1", data->filename);
            callRun(data->env, data->runnable);
            LOGV("Deleting unused runnable");
            (*data->env)->DeleteGlobalRef(data->env, data->runnable2);
            LOGV("Done");
        } else {
            LOGV("Error, calling runnable 2");
            callRun(data->env, data->runnable2);
            LOGV("Deleting unused runnable");
            (*data->env)->DeleteGlobalRef(data->env, data->runnable);
            LOGV("Done");
        }
        lib_free(createdmoment->data);
        lib_free(createdmoment);
    }

    lib_free(data->filename);
    lib_free(data);

}
__attribute__((used)) void timemachine_store_new_entry(const char* path, jobject runOnSuccess, jobject runOnError) {
    travel_data *data = lib_malloc(sizeof(travel_data));
    data->filename = lib_strdup(path);
    data->env = getAactiveenv();

    data->runnable = (*data->env)->NewGlobalRef(data->env, runOnSuccess);
    data->runnable2 = (*data->env)->NewGlobalRef(data->env, runOnError);
    interrupt_maincpu_trigger_trap(timemachine_store_new_entry_trap, data);

}