//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


extern "C"
{
#include "jnihelpers.h"


}
/*

 */
#include <fcntl.h>
#include <oboe/Oboe.h>
#include <jni.h>
#include <logginghelpers.h>
#include "jnihelpers.h"

#ifdef LOG_CALLS
struct EntryRaiiObject {
    EntryRaiiObject(const char *f) : f_(f) { LOGV("AudioEngine:Entered into %s", f_); } // NOLINT(google-explicit-constructor)
    ~EntryRaiiObject() { LOGV("AudioEngine:Exited from %s", f_); }
    const char *f_;
};
#define ENTRY EntryRaiiObject obj ## __LINE__ (__FUNCTION__);
#else
#define ENTRY
#endif


class AudioEngine : public oboe::AudioStreamDataCallback, public oboe::AudioStreamErrorCallback {
public:
    void onErrorAfterClose(oboe::AudioStream* audioStream, oboe::Result error) override { ENTRY
        createStream(&outputchannels, &stream);
        stream->requestStart();
    }
    void setAudioTestCallback(jobject obj) { ENTRY
        std::lock_guard<std::mutex> guard(myMutex);
        JNIEnv* env = getAactiveenv();
        audio_test_callback = env->NewGlobalRef(obj);
        //audio_test_callback = obj;
    }
    oboe::DataCallbackResult onAudioReady(
            oboe::AudioStream *oboeStream,
            void *audioData,
            int32_t numFrames) override { ENTRY
        int32_t raw_size;
        if (stream != nullptr) {
            if (reader > 0) {
                if (stream->getFormat() == oboe::AudioFormat::I16) {
                    raw_size = numFrames * (int32_t)sizeof(int16_t);
                } else {
                    raw_size = numFrames * (int32_t)sizeof(float);
                }
                ssize_t l = read(reader, audioData, raw_size);
                if (l < 0) {
                    if (errno != EAGAIN) {
                        return oboe::DataCallbackResult::Stop;
                    }
                }
            }
        }
        return oboe::DataCallbackResult::Continue;
    }

    int writeData(int16_t *pbuf, size_t pbuf_len){ ENTRY
        std::lock_guard<std::mutex> guard(myMutex);
        static auto floatbuffer=(float*)malloc(1);
        static size_t floatbuffer_len=0;
        if (skipAudiobuffer >= pbuf_len) {
            skipAudiobuffer -= pbuf_len;
            return 0;
        } else {
            skipAudiobuffer = 0;
        }
        if (stream != nullptr) {
            if (oldMute == 0 && newMute == 1) {
                stream->requestPause();
                changeState(oboe::StreamState::Pausing);
                stream->requestFlush();
                changeState(oboe::StreamState::Flushing);
                oldMute = 1;
            }
            if (oldMute == 1 && newMute == 0) {
                stream->requestStart();
                changeState(oboe::StreamState::Starting);
                oldMute = 0;
            }
#pragma clang diagnostic push
#pragma ide diagnostic ignored "ConstantConditionsOC"
            if (oldMute == 0 && newMute == 0) {
                if (stream->getState() == oboe::StreamState::Started) {
                    if (stream->getDataCallback() == this) {
                        if (stream->getFormat() == oboe::AudioFormat::I16) {
                            write(writer, pbuf, pbuf_len * sizeof(int16_t));
                        } else {
                            if (floatbuffer_len < pbuf_len) {
                                floatbuffer = (float *) realloc(floatbuffer, pbuf_len * sizeof(float)); // NOLINT(*-suspicious-realloc-usage)
                                floatbuffer_len = pbuf_len;
                            }
                            oboe::convertPcm16ToFloat(pbuf, floatbuffer, pbuf_len); // NOLINT(cppcoreguidelines-narrowing-conversions)
                            write(writer, floatbuffer, pbuf_len * sizeof(float));
                        }
                    } else {
                        stream->write(pbuf, pbuf_len, 0); // NOLINT(cppcoreguidelines-narrowing-conversions)
                    }
                }
            }
#pragma clang diagnostic pop
        }
        if (audio_test_callback && pbuf && pbuf_len) {
            int16_t v0 = pbuf[0];
            for (int i = 0;i<pbuf_len;i++) {
                if (pbuf[i] != v0) {
                    JNIEnv* env = getAactiveenv();
                    jclass cls = env->FindClass("java/lang/Runnable");
                    if (cls) {
                        jmethodID mth = env->GetMethodID(cls, "run", "()V");
                        if (mth) {
                            env->CallVoidMethod(audio_test_callback, mth);
                        }
                    }
                    env->DeleteGlobalRef(audio_test_callback);
                    audio_test_callback = nullptr;
                    break;
                }
            }
        }
        return 0;
    }
    int suspend() {
        std::lock_guard<std::mutex> guard(myMutex);
        return suspendImpl();
    }
#pragma clang diagnostic push
#pragma ide diagnostic ignored "ConstantFunctionResult"
    int init(const char *param, int *speed, int *fragsize, int *fragnr,
              int *channels) { ENTRY
        std::lock_guard<std::mutex> guard(myMutex);
        static int opensl_initialized = 0;
        if (!opensl_initialized) {
            jmethodID mthSetError = GetMethodID(GetObjectClass(CurrentActivity()), "setAudioDefaults", "()V");
            CallVoidMethod(CurrentActivity(), mthSetError);
            opensl_initialized=1;
        }
        if (stream) {
            if (stream->getState() == oboe::StreamState::Started) {
                stream->requestStop();
                LOGV("oboe-stream closing: %p",stream);
                stream->close();
            }
        }
        if (param) {
            /* make comiler smile */
        }
        createpipes();
        jmethodID mthGetError = GetMethodID(GetObjectClass(CurrentActivity()), "isAudioFailing", "()Z");

        JNIEnv* currentenv= getAactiveenv();
        if (currentenv->CallBooleanMethod(CurrentActivity(), mthGetError) == JNI_FALSE) {
            oboe::Result result = createStream(channels, &stream);
            if (result == oboe::Result::OK) {
                LOGV("oboe-stream created: %p",stream);
                samplerate = stream->getSampleRate();
                int nr = (*fragnr) * *speed * *fragsize;
                *speed = stream->getSampleRate();

                *fragsize = stream->getFramesPerBurst();
                *fragnr = nr / (*speed * *fragsize);
                stream->requestStart();
                changeState(oboe::StreamState::Starting);
                *channels = stream->getChannelCount();
                jmethodID mth = GetMethodID(GetObjectClass(CurrentActivity()), "getAudioFocus", "()V");
                CallVoidMethod(CurrentActivity(), mth);
                return 0;
            } else {
                LOGV("oboe-stream creation failed: %p",stream);
                if (result == oboe::Result::ErrorInternal) {
                    jmethodID mth = GetMethodID(GetObjectClass(CurrentActivity()), "onAudioFailure", "()V");
                    CallVoidMethod(CurrentActivity(), mth);
                }
                if (stream != nullptr) {
                    LOGV("oboe-stream closing: %p",stream);
                    stream->close();
                    stream = nullptr;
                }
            }
        }
        return 0;


    }
#pragma clang diagnostic pop
    int resume() { ENTRY
        std::lock_guard<std::mutex> guard(myMutex);
        if (stream == nullptr) {
            return 0;
        }
        if (stream->getState() == oboe::StreamState::Started) {
            return 0;
        }
        oboe::Result result = stream->requestStart();
        if (result==oboe::Result::OK) {
            result = changeState(oboe::StreamState::Starting);
        }
        createpipes();
        return result==oboe::Result::OK?0:1;
    }
    void closeStream() {
        std::lock_guard<std::mutex> guard(myMutex);
        if (stream) {
            suspendImpl();
            samplerate = 0;
            changeState(oboe::StreamState::Closing);
            LOGV("oboe-stream closing: %p",stream);
            stream->close();
            stream = nullptr;
        }

    }
    void mute() { ENTRY
        std::lock_guard<std::mutex> guard(myMutex);
        if (stream) {
            if (oldMute == 0) {
                stream->requestPause();
                changeState(oboe::StreamState::Pausing);
                stream->requestFlush();
                changeState(oboe::StreamState::Flushing);
                oldMute = 1;
            }
            newMute = 1;
        }
    }
    void unmute() { ENTRY
        std::lock_guard<std::mutex> guard(myMutex);
        newMute = 0;
    }
    static void setDefaultValues(int sampleRate, int framesPerBurst) { ENTRY
        oboe::DefaultStreamValues::SampleRate = (int32_t) sampleRate;
        oboe::DefaultStreamValues::FramesPerBurst = (int32_t) framesPerBurst;
    }

    void ignoreAudiobuffer(size_t len) {
        skipAudiobuffer = len;
    }

private:
    int createpipes() { ENTRY
        int fd[2];
        if (pipe(fd) < 0)
            return 1;

        // error checking for fcntl


        if (fcntl(fd[0], F_SETFL, O_NONBLOCK) < 0) {
            return 2;
        }

        if (fcntl(fd[1], F_SETFL, O_NONBLOCK) < 0) {
            return 3;
        }
        reader=fd[0];
        writer=fd[1];
        return 0;
    }

    oboe::Result createStream(int* channels, oboe::AudioStream** newStream) { ENTRY // NOLINT(readability-non-const-parameter)
        oboe::AudioStreamBuilder builder;

        builder.setDirection(oboe::Direction::Output);
        builder.setPerformanceMode(oboe::PerformanceMode::LowLatency);
        builder.setSharingMode(oboe::SharingMode::Exclusive);
        builder.setDataCallback(this);
        builder.setErrorCallback(this);
        if (samplerate) {
            builder.setSampleRate(samplerate);
        }
        builder.setChannelCount(*channels==1?oboe::ChannelCount::Mono:oboe::ChannelCount::Stereo);
        oboe::Result result = builder.openStream(newStream);
        outputchannels = *channels;
        return result;

    }
    int suspendImpl() { ENTRY
        if (stream == nullptr) {
            return 0;
        }
        if (stream->getState() == oboe::StreamState::Stopped) {
            return 0;
        }
        if (reader>0) {
            close(reader);
            reader=-1;
        }
        if (writer>0) {
            close(writer);
            writer=-1;
        }

        oboe::Result result = stream->requestPause();
        if (result==oboe::Result::OK) {
            result=changeState(oboe::StreamState::Stopping);
        }
        return (result==oboe::Result::OK?0:1);
    }
    oboe::AudioStream *stream = nullptr;
    int writer=-1;
    int reader=-1;
    int oldMute = 0;
    int newMute = 0;
    int outputchannels = 0;
    int samplerate = 0;
    size_t skipAudiobuffer = 0;
    std::mutex myMutex;
    jobject audio_test_callback = nullptr;
    oboe::Result changeState(oboe::StreamState state) { ENTRY
        if (stream) {
            oboe::StreamState inputState = state;
            oboe::StreamState nextState = oboe::StreamState::Uninitialized;
            int64_t timeoutNanos = 100 * oboe::kNanosPerMillisecond;
            return stream->waitForStateChange(inputState, &nextState, timeoutNanos);
        }
        return oboe::Result::OK;
    }
};
static AudioEngine myAudioEngine;

#pragma clang diagnostic push
#pragma ide diagnostic ignored "ConstantFunctionResult"
static int oboe_init(const char *param, int *speed, int *fragsize, int *fragnr,
                     int *channels)
{
    return myAudioEngine.init(param, speed, fragsize, fragnr, channels);
}
#pragma clang diagnostic pop
#pragma clang diagnostic push
#pragma ide diagnostic ignored "ConstantFunctionResult"
static int oboe_write(int16_t *pbuf, size_t pbuf_len)
{
    return myAudioEngine.writeData(pbuf, pbuf_len);
}
#pragma clang diagnostic pop
int oboe_suspend() {
    return myAudioEngine.suspend();
}
int oboe_resume()
{
    return myAudioEngine.resume();
}
static void oboe_close() {
    myAudioEngine.closeStream();
}


extern "C"
{
__attribute__((used)) void set_audio_test_callback(jobject callback) {
    myAudioEngine.setAudioTestCallback(callback);
}

void oboe_mute()
{
    myAudioEngine.mute();
}
void oboe_unmute()
{
    myAudioEngine.unmute();
}
__attribute__((used)) void oboe_set_default_values(int sampleRate, int framesPerBurst) {
    AudioEngine::setDefaultValues(sampleRate, framesPerBurst);
}

#include "sound.h"

sound_device_t oboe_device = {
                "oboe",
                oboe_init,
                oboe_write,
                nullptr,
                nullptr,
                nullptr,
                oboe_close,
                oboe_suspend,
                oboe_resume,
                1,
                2,
                false
        };
void ignore_audiobuffer(size_t size) {
    myAudioEngine.ignoreAudiobuffer(size);
}
}