//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;

import android.annotation.SuppressLint;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.SwitchCompat;
import de.rainerhock.eightbitwonders.Emulation;
import de.rainerhock.eightbitwonders.EmulationUi;
import de.rainerhock.eightbitwonders.KeyboardFragment;
import de.rainerhock.eightbitwonders.MappedSpinner;
import de.rainerhock.eightbitwonders.R;
import de.rainerhock.eightbitwonders.SettingsFragment;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * Fragment with the virtual keyboard for VICE-based emulations.
 */
public final class ViceKeyboardFragment extends KeyboardFragment {
    private final List<KeyButton> mAllButtons = new LinkedList<>();
    void setController(final ViewGroup vg, final ViceKeyboardFragment k) {
        for (int i = 0; i < vg.getChildCount(); i++) {
            View c = vg.getChildAt(i);
            if (c instanceof ViewGroup) {
                setController((ViewGroup) vg.getChildAt(i), k);
            }
            if (c instanceof KeyButtonBase) {
                ((KeyButtonBase) vg.getChildAt(i)).setKeyboardController(k);
                if (c instanceof KeyButton) {
                    mAllButtons.add((KeyButton) c);
                }
            }
        }
        ViceEmulation.getInstance().setKeyButtons(mAllButtons);
    }
    List<KeyButton> getKeys() {
        return mAllButtons;
    }
    void reset() {
        for (KeyButtonBase k : mKeysDown) {
            highlightButton(k, false);
        }
        ViceEmulation.getInstance().keyboardKeyClear();

    }

    private static final String TAG = ViceKeyboardFragment.class.getName();
    private void setKeyPressed(final KeyButtonBase btn, final boolean val) {
        final int row = btn.getRow();
        if (val) {
            ViceEmulation.getInstance().keypressed(btn.getId(),
                    row, btn.getCol(), btn.getShiftState());
        } else {
            ViceEmulation.getInstance().keyreleased(btn.getId(),
                    row, btn.getCol(), btn.getShiftState());
        }
        if (val) {
            mKeysDown.add(btn);
        } else {
            mKeysDown.remove(btn);
        }

    }

    private void highlightButton(final KeyButtonBase btn, final boolean highlight) {
        btn.setHighlighted(highlight);
        if (highlight) {
            btn.setBackgroundColor(getResources().getColor(R.color.button_highlight));
        } else {
            btn.setBackground(btn.getOriginalColor());
        }

    }

    private final LinkedList<KeyButtonBase> mKeysTouched = new LinkedList<>();
    private final LinkedList<KeyButtonBase> mKeysDown = new LinkedList<>();
    private final LinkedList<KeyButtonBase> mUsedModificators = new LinkedList<>();

    void dumpstate() {
        Log.d(TAG, "keysDown: " + mKeysDown);
        Log.d(TAG, "keysTouched: " + mKeysTouched);
        Log.d(TAG, "usedModificators" + mUsedModificators);

    }
    void higlightToggledKeyPressed(final ToggledKeyButton btn) {
        if (ViceEmulation.getInstance().isToggleKeyPressed(btn.getRow(), btn.getCol())) {
            highlightButton(btn, true);
            mKeysTouched.add(btn);
        }
    }
    void pressKey(final ToggledKeyButton btn) {
        if (mKeysTouched.contains(btn)) {
            highlightButton(btn, false);
            setKeyPressed(btn, btn.getRow() < 0);
            mKeysTouched.remove(btn);
        } else {
            internalPressKey(btn);
        }
        ViceEmulation e = ViceEmulation.getInstance();
        View v = requireView();
        e.setToggleKeyPressed(btn.getRow(), btn.getCol(),
                mKeysTouched.contains(btn));
        if (e.toggleKeyset(btn.getRow(), btn.getCol())) {
            int[] hide = e.hideKeysAfterKeypress(btn.getRow(), btn.getCol());
            int[] show = e.showKeysAfterKeypress(btn.getRow(), btn.getCol());

            for (int id : hide) {
                v.findViewById(id).setVisibility(View.GONE);
            }
            for (int id : show) {
                v.findViewById(id).setVisibility(View.VISIBLE);
            }

        }


    }
    void pressKey(final KeyButton btn) {
        internalPressKey(btn);
    }
    private void internalPressKey(final KeyButtonBase btn) {
        if (btn.isModificator()) {
            if (btn.isShiftLock()) {
                if (mKeysTouched.contains(btn)) {
                    highlightButton(btn, false);
                    setKeyPressed(btn, false);
                    mKeysTouched.remove(btn);
                } else {
                    if (!btn.isShift() || !isShiftLockPressed()) {

                        setKeyPressed(btn, true);
                        mKeysTouched.add(btn);
                    }
                    highlightButton(btn, true);
                }
            } else {
                if (mKeysDown.contains(btn)) {
                    highlightButton(btn, false);
                    setKeyPressed(btn, false);
                    mKeysTouched.remove(btn);
                } else {
                    highlightButton(btn, true);
                    setKeyPressed(btn, true);
                    mKeysTouched.add(btn);
                    mUsedModificators.remove(btn);
                }
            }
            for (KeyButtonBase k : mKeysDown) {
                if (k.isModificator() && k != btn && !k.isShiftLock()) {
                    if ((btn.isCbm() && k.isShift()) || (btn.isShift() && k.isCbm())
                            && !mUsedModificators.contains(k)) {
                        mUsedModificators.add(k);
                    }
                }
            }

        } else {
            for (KeyButtonBase k : mKeysDown) {
                if (k.isModificator() && k != btn && !k.isShiftLock()) {
                    if (!mUsedModificators.contains(k)) {
                        mUsedModificators.add(k);
                    }

                }
            }

            highlightButton(btn, true);
            setKeyPressed(btn, true);
            mKeysTouched.add(btn);
        }


    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private boolean isShiftLockPressed() {
        for (KeyButtonBase k : mKeysTouched) {
            if (k.isShiftLock()) {
                return true;
            }
        }
        return false;
    }
    void releaseKey(final KeyButton btn) {
        LinkedList<KeyButtonBase> releaseafter = new LinkedList<>();
        boolean shiftWasPressed = false;
        boolean cbmWasPressed = false;
        if (!btn.isShiftLock()) {
            for (KeyButtonBase k : mKeysDown) {
                if (k.isModificator()) {
                    if (k.isShift()) {
                        shiftWasPressed = true;
                    }
                    if (k.isCbm()) {
                        cbmWasPressed = true;
                    }
                    if (!k.isShiftLock()) {
                        if (!mKeysTouched.contains(k)) {

                            releaseafter.add(k);
                        }
                    }
                }
            }
            if (btn.isShift()) {
                shiftWasPressed = true;
            }
            if (btn.isCbm()) {
                cbmWasPressed = true;
            }
            boolean releaseThisModificator = (btn.isModificator()
                    && mUsedModificators.contains(btn)) || (shiftWasPressed && cbmWasPressed);
            if (releaseThisModificator) {
                if (!btn.isShift() || !isShiftLockPressed()) {
                    setKeyPressed(btn, false);
                    mUsedModificators.remove(btn);
                }
                highlightButton(btn, false);


            }
            if (!btn.isModificator()) {

                setKeyPressed(btn, false);
                highlightButton(btn, false);
            }
            for (KeyButtonBase k : releaseafter) {
                if (!k.isShift() || !isShiftLockPressed()) {
                    setKeyPressed(k, false);
                    mUsedModificators.remove(btn);
                }
                highlightButton(k, false);

            }
            mKeysTouched.remove(btn);
        }
        ViceEmulation e = ViceEmulation.getInstance();
        View v = getView();
        if (v != null && e.toggleKeyset(btn.getRow(), btn.getCol())) {
            int[] hide = e.hideKeysAfterKeypress(btn.getRow(), btn.getCol());
            int[] show = e.showKeysAfterKeypress(btn.getRow(), btn.getCol());
            for (int id : hide) {
                v.findViewById(id).setVisibility(View.GONE);
            }
            for (int id : show) {
                v.findViewById(id).setVisibility(View.VISIBLE);
            }

        }
    }

    @Override
    public void onStart() {
        super.onStart();
        ViewGroup vg = (ViewGroup) getView();
        if (vg != null) {
            setController(vg, this);
        }
    }
    private void setDynamicVisibility(final View root, final ArrayList<Integer> ids,
                                      final int visibility) {
        if (ids != null) {
            for (int id : ids) {
                Log.v(getClass().getSimpleName(), visibility == 0 ? "showing " : "hiding "
                        + root.findViewById(id));
                View v = root.findViewById(id);
                if (v != null) {
                    root.findViewById(id).setVisibility(visibility);
                }
            }
        }

    }
    @Override
    public View onCreateView(@NonNull final LayoutInflater inflater,
                             final ViewGroup container,
                             final Bundle savedInstanceState) {
        mC128LayoutFixed = false;
        View root = inflater.inflate(
                requireArguments().getInt("layout_id"), container, false);
        if (root instanceof ViewGroup) {
            final ViewGroup vg = (ViewGroup) root;
            final SwitchCompat sw = vg.findViewById(R.id.sw_alpha_num);
            final View alpha = vg.findViewById(R.id.part_alpha);
            final View num = vg.findViewById(R.id.part_num);
            if (sw != null && alpha != null && num != null) {
                sw.setOnCheckedChangeListener((compoundButton, b) -> {
                    alpha.setVisibility(b ? View.GONE : View.VISIBLE);
                    num.setVisibility(b ? View.VISIBLE : View.GONE);
                    sw.post(() -> fixC128Layout(root));
                });
                final boolean b = requireArguments().getBoolean("show_num", false);
                alpha.setVisibility(b ? View.GONE : View.VISIBLE);
                num.setVisibility(b ? View.VISIBLE : View.GONE);

                sw.setChecked(false);
            }
            setHighlights(root,
                    requireArguments().getIntegerArrayList("keys_to_highlight"), true);
            setHighlights(root,
                    requireArguments().getIntegerArrayList("keys_to_dehighlight"), false);
            setDynamicVisibility(root,
                    requireArguments().getIntegerArrayList("keys_to_hide"), View.GONE);
            setDynamicVisibility(root,
                    requireArguments().getIntegerArrayList("keys_to_show"), View.VISIBLE);
            setDynamicVisibility(root,
                    requireArguments().getIntegerArrayList("dynamic_keys_to_hide"), View.GONE);
            setDynamicVisibility(root,
                    requireArguments().getIntegerArrayList("dynamic_keys_to_show"), View.VISIBLE);
        }
        return root;
    }

    @Override
    public void onViewCreated(final @NonNull View view, final @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        view.post(() -> fixC128Layout(view));
    }
    private boolean mC128LayoutFixed = false;
    private void fixC128Layout(final @NonNull View view) {
        final View num = view.findViewById(R.id.part_num);
        if (num != null) {
            final View source = num.findViewById(R.id.c128key_num_minus);
            final View target = num.findViewById(R.id.c128key_enter);
            if (source != null && target != null) {
                view.post(() -> {
                    ViewGroup.LayoutParams lp = target.getLayoutParams();
                    int width = source.getMeasuredWidth();
                    if (width > 0 & !mC128LayoutFixed) {
                        lp.width = width;
                        mC128LayoutFixed = true;
                        target.setLayoutParams(lp);
                    }
                });
            }
        }
    }

    private void setHighlights(final View root, final ArrayList<Integer> keysToHighlight,
                               final boolean val) {
        if (keysToHighlight != null) {
            for (int id : keysToHighlight) {
                View v = root.findViewById(id);
                if (/*v.getVisibility() == View.VISIBLE &&*/ v instanceof KeyButtonBase) {
                    KeyButtonBase kb = (KeyButtonBase) v;
                    highlightButton(kb, val);
                }
            }
        }
    }

    public static final class KeyboardSettingsFragment extends SettingsFragment {
        @SuppressLint("InflateParams")
        @Nullable
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {

            return inflater.inflate(R.layout.activity_part_hardwarekeyboard, null);
        }

        @Override
        protected void initWidgets(final View rootView) {
            super.initWidgets(rootView);
            populateSpinner(ViceEmulation.getInstance().getModel());
        }
        private void populateSpinner(final int model) {
            List<EmulationUi.SettingSpinnerElement> elements = getSettingSpinnerElements(model);
            MappedSpinner ms = requireView().findViewById(R.id.sp_keyboard_mapping);
            ms.populate(elements);
            String activeElementId = ViceEmulation.getInstance().getEmulationActivity()
                    .getCurrentUseropts().getStringValue((String) ms.getTag(),
                            ms.getDefaultElementId());
            for (EmulationUi.SettingSpinnerElement e : elements) {
                if (ViceEmulation.getInstance().isMatchingKeyboardMapping(e, activeElementId)) {
                    ms.setSelection(e.getId());
                }
            }

        }


        private @NonNull List<EmulationUi.SettingSpinnerElement> getSettingSpinnerElements(
                final int model) {
            List<EmulationUi.SettingSpinnerElement> elements = new LinkedList<>();
            for (Emulation.HardwareKeyboardFunctions.KeyboardMapping mapping
                    : ViceEmulation.getInstance().getHardwareKeyboardFunctions()
                    .getMappings()) {
                elements.add(new EmulationUi.SettingSpinnerElement() {
                    @Override
                    public String getId() {
                        return mapping.getId();
                    }

                    @Override
                    public String getText() {
                        return mapping.getName();
                    }
                });
            }
            return elements;
        }
        @Override
        public void handleOption(final String key, final String value) {
            if ("Model".equals(key)) {
                populateSpinner(Integer.parseInt(value));
            }
            super.handleOption(key, value);
        }


    }
}
