//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;

import android.app.Activity;
import android.content.Context;
import android.graphics.Typeface;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.CheckBox;
import android.widget.ListView;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import de.rainerhock.eightbitwonders.Emulation;
import de.rainerhock.eightbitwonders.R;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;


/**
 * This activity contains the vice-specific user-interface for the selection whic action
 * has to be done with a select file (e.g. autostart a program from a disk image or
 * simply attach it).
 */
public final class ViceFileActionActivity extends ViceMediaActionActivity {
    private void showAttachDisk(final Uri uri) {
        List<Integer> devices = getActiveDiskDrives();
        if (!devices.isEmpty()) {
            findViewById(R.id.tv_attach_disk_image_label).setVisibility(View.VISIBLE);
            findViewById(R.id.attach_disk).setVisibility(View.VISIBLE);
            findViewById(R.id.cb_readonly).setVisibility(View.VISIBLE);

            for (int viewId : new int[]{R.id.bn_drive8, R.id.bn_drive9,
                    R.id.bn_drive10, R.id.bn_drive11}) {

                if (devices.contains(Integer.valueOf(findViewById(viewId).getTag().toString()))) {
                    findViewById(viewId).setOnClickListener(v -> {
                        ViceEmulation.getInstance().addTask(() -> {
                            if (ViceEmulation.getInstance().setDriveImage(
                                    Integer.valueOf((String) v.getTag()),
                                    uri, getReadonlyState())) {
                                runOnUiThread(() -> {
                                    setResult(Activity.RESULT_OK);
                                    finish();
                                });
                            } else {
                                runOnUiThread(() ->
                                        Toast.makeText(ViceFileActionActivity.this,
                                                        R.string.IDMES_CANNOT_ATTACH_FILE,
                                                        Toast.LENGTH_LONG)
                                                .show());
                            }
                        });
                        ViceEmulation.getInstance().setPaused(false);
                    });
                    findViewById(viewId).setVisibility(View.VISIBLE);
                }
            }
        }
    }

    private void showAttachTape(final Uri uri) {
        if (uri.getLastPathSegment() != null) {
            findViewById(R.id.attach_tape).setVisibility(
                    uri.getLastPathSegment().endsWith(".t64") ? View.GONE : View.VISIBLE);
        } else {
            findViewById(R.id.attach_tape).setVisibility(View.GONE);
        }
        findViewById(R.id.tv_attach_tape_image_label).setVisibility(
                findViewById(R.id.attach_tape).getVisibility());
        findViewById(R.id.cb_readonly).setVisibility(
                findViewById(R.id.attach_tape).getVisibility());
        findViewById(R.id.bn_tape).setOnClickListener(v -> {
            Log.v("showAttachTape", "onClickListener");
            ViceEmulation.getInstance().addTask(() -> {
                if (ViceEmulation.getInstance().setTapeImage(1, uri,
                        getReadonlyState())) {
                    ViceEmulation.getInstance().setPaused(true);
                    runOnUiThread(() -> {
                        setResult(Activity.RESULT_OK);
                        finish();
                    });

                } else {
                    ViceEmulation.getInstance().setPaused(true);
                    runOnUiThread(() ->
                            Toast.makeText(
                                    ViceFileActionActivity.this,
                                    R.string.IDMES_CANNOT_ATTACH_FILE, Toast.LENGTH_LONG)
                                    .show());
                }
            });
            ViceEmulation.getInstance().setPaused(false);
        });

    }

    @Override
    public void finish() {
        //ViceEmulation.getInstance().getSoundFunctions().mute();
        super.finish();
    }

    private static class CartridgeArrayElement implements Emulation.DataEntry {
        private final Integer mType;
        private  final String mName;

        CartridgeArrayElement(final Integer type, final String name) {
            mType = type;
            mName = name;
        }

        @Override
        public String getText() {
            return mName;
        }
    }

    private void showAutorun(final Uri uri,
                             final ArrayList<ViceEmulation.DirEntry> imagecontent) {
        findViewById(R.id.autorun).setVisibility(View.VISIBLE);
        ArrayAdapter<ViceEmulation.DirEntry> aa =
                new ArrayAdapter<ViceEmulation.DirEntry>(this, 0) {
                @NonNull
                @Override
                public View getView(final int position,
                                    final View convertView,
                                    @NonNull final ViewGroup parent) {
                    final Typeface typeface =
                            Typeface.createFromAsset(getContext().getApplicationContext()
                                    .getAssets(), "fonts/cbm.ttf");
                    LayoutInflater inflater =
                            (LayoutInflater) getContext().getSystemService(
                                    Context.LAYOUT_INFLATER_SERVICE);
                    View rowView;
                    if (convertView != null) {
                        rowView = convertView;
                    } else {
                        rowView = inflater.inflate(
                                R.layout.fragment_viceimage_list_item, parent, false);
                    }
                    if (getItem(position) != null) {
                        TextView name = rowView.findViewById(R.id.tv_name);
                        name.setText(Objects.requireNonNull(getItem(position)).getText());
                        name.setTypeface(typeface);
                        rowView.setEnabled((!Objects.requireNonNull(
                                getItem(position)).getType().equals("PRG")));
                        rowView.setVisibility(
                                name.getText().length() > 0 ? View.VISIBLE : View.GONE);
                    }
                    return rowView;
                }
            };
        aa.addAll(imagecontent);
        ListView lv = findViewById(R.id.lv_imagecontents);
        lv.setAdapter(aa);
        lv.setOnItemClickListener((parent, view, position, id) ->
                ViceEmulation.getInstance().autostart(
                        uri,
                    (ViceEmulation.DirEntry) parent.getItemAtPosition(position),
                    () -> {
                        setResult(Activity.RESULT_OK);
                        finish();
                    },
                    () -> Toast.makeText(
                            ViceFileActionActivity.this,
                            R.string.IDMES_CANNOT_ATTACH_FILE, Toast.LENGTH_LONG)
                            .show()
        ));
    }

    private void showAttachCartridge(final Uri uri,
                                     final Map<String, Integer> cartridges,
                                     final Map<String, Integer> flags,
                                     final Map<String, Integer> filters) {
        findViewById(R.id.attach_cartridge).setVisibility(View.VISIBLE);
        ArrayAdapter<CartridgeArrayElement> aa =
                new ArrayAdapter<CartridgeArrayElement>(
                    ViceFileActionActivity.this, android.R.layout.simple_list_item_1) {
                    @Override
                    public @NonNull  View  getView(final int position,
                                                   @Nullable final View convertView,
                                                   @NonNull final ViewGroup parent) {
                        View view;
                        if (convertView == null) {
                            //noinspection ConstantValue
                            view = super.getView(position, convertView, parent);
                        } else {
                            view = convertView;
                        }
                        CartridgeArrayElement data = getItem(position);
                        if (data != null) {
                            ((TextView) view).setText(data.mName);
                            view.setTag(data.mType);
                        }
                        return view;
                    }
                };
        for (String s : cartridges.keySet()) {
            aa.add(new CartridgeArrayElement(cartridges.get(s), s));
        }
        ListView lv = findViewById(R.id.lv_cartridgetypes);
        if (filters != null) {

            ArrayAdapter<CartridgeArrayElement> aafilters =
                    new ArrayAdapter<CartridgeArrayElement>(ViceFileActionActivity.this,
                            android.R.layout.simple_list_item_1) {
                        @Override
                        public @NonNull View getView(final int position,
                                            final View convertView,
                                            @NonNull final ViewGroup parent) {
                            View view;
                            if (convertView == null) {
                                //noinspection ConstantValue
                                view = super.getView(position, convertView, parent);
                            } else {
                                view = convertView;
                            }
                            CartridgeArrayElement data = getItem(position);
                            if (data != null) {
                                ((TextView) view).setText(data.mName);
                                view.setTag(data.mType);
                            }
                            return view;
                        }
                    };
            for (String s : filters.keySet()) {
                aafilters.add(new CartridgeArrayElement(filters.get(s), s) {
                    @NonNull
                    @Override
                    public String toString() {
                        return s;
                    }
                });
            }
            Spinner sp = findViewById(R.id.sp_cartridge_filter);
            //noinspection ComparatorCombinators
            aafilters.sort((lhs, rhs) -> lhs.mType - rhs.mType);
            sp.setAdapter(aafilters);
            sp.setVisibility(View.VISIBLE);
            sp.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
                @Override
                public void onItemSelected(final AdapterView<?> parent, final View view,
                                           final int position, final long id) {
                    aa.clear();
                    Integer filter = (Integer) (view.getTag());
                    for (String s : cartridges.keySet()) {
                        if (filter == 0 | filter.equals(flags.get(s))) {
                            aa.add(new CartridgeArrayElement(cartridges.get(s), s));
                        }
                    }
                    //noinspection ComparatorCombinators
                    aa.sort((lhs, rhs) -> lhs.mName.compareTo(rhs.mName));
                    lv.setAdapter(aa);
                }

                @Override
                public void onNothingSelected(final AdapterView<?> adapterView) {
                }
            });
            findViewById(R.id.tv_cartridge_filter).setVisibility(View.VISIBLE);
        }
        //lv.setAdapter(aa);
        lv.setOnItemClickListener((parent, view, position, id) -> {
            ViceEmulation.getInstance().addTask(() -> {
                Runnable runOnSuccess = () -> runOnUiThread(() -> {
                    setResult(Activity.RESULT_OK, null);
                    finish();

                });
                Log.v(getClass().getSimpleName(), "Creating toast about cartridge error");
                Runnable runOnError = () -> runOnUiThread(() -> {
                        Log.v(getClass().getSimpleName(), "Showing toast about cartridge error");
                        Toast.makeText(
                                        ViceFileActionActivity.this,
                                        R.string.IDMES_CANNOT_ATTACH_FILE,
                                        Toast.LENGTH_LONG)
                                .show();

                    });
                ViceEmulation.getInstance().setCartridge((Integer) view.getTag(), uri,
                        runOnSuccess, runOnError);
            });
            ViceEmulation.getInstance().setPaused(false);
        });
    }

    private boolean getReadonlyState() {
        CheckBox view = findViewById(R.id.cb_readonly);
        return view.isChecked();
    }

    private void showReadOnly(final boolean fileIsReadOnly) {
        if (fileIsReadOnly) {
            CheckBox view = findViewById(R.id.cb_readonly);
            view.setEnabled(false);
            view.setChecked(true);
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
    }

    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_vice_file_action);
        int typ = getIntent().getIntExtra("type", 0);
        String filename = getIntent().getStringExtra("file");
        setTitle(getIntent().getStringExtra("title"));
        if (filename != null) {
            Uri uri = Uri.parse(filename);
            if (typ == 1) {
                showAttachDisk(uri);
            }
            if (typ == 2) {
                showAttachTape(uri);
            }
            if (typ == 1 || typ == 2) {
                if (Objects.requireNonNull(uri.getLastPathSegment()).endsWith(".t64")) {
                    findViewById(R.id.cb_readonly).setVisibility(View.GONE);
                } else {
                    showReadOnly(getIntent().getBooleanExtra("readonly", false));
                }
            }
            @SuppressWarnings("unchecked")
            ArrayList<ViceEmulation.DirEntry> imagecontent =
                    (ArrayList<ViceEmulation.DirEntry>) getIntent()
                            .getSerializableExtra("imagecontent");
            if (imagecontent != null) {
                showAutorun(uri, imagecontent);
            }
            if (typ == 4) {
                @SuppressWarnings("unchecked")
                Map<String, Integer> cartridges =
                        (Map<String, Integer>) getIntent()
                                .getSerializableExtra("cartridges");
                @SuppressWarnings("unchecked")
                Map<String, Integer> filters =
                        (Map<String, Integer>) getIntent()
                                .getSerializableExtra("cartridge-filters");
                @SuppressWarnings("unchecked")
                Map<String, Integer> flags =
                        (Map<String, Integer>) getIntent()
                                .getSerializableExtra("cartridge-flags");
                if (cartridges != null) {
                    showAttachCartridge(uri, cartridges, flags, filters);
                }
            }
        }
    }
}
