//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;

import android.content.Context;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import de.rainerhock.eightbitwonders.DownloaderFactory;
import de.rainerhock.eightbitwonders.Emulation;
import de.rainerhock.eightbitwonders.EmulationConfiguration;
import de.rainerhock.eightbitwonders.EmulationFactory;
import de.rainerhock.eightbitwonders.EmulationUi;

/**
 * Information about the available VICE-based Emulations.
 */
public final class ViceFactory implements EmulationFactory {

    /**
     * List of all available emulations.
    */
    @Override
    public List<String> getEmulatorIds() {
        return Arrays.asList("C64", "VIC20", "PET", "C128", "PLUS4");
    }

    /**
     * Start an new VICE-based emulation.
     * @param ui implementation of {@link EmulationUi} the emulation interacts with
     * @param config Configuration to start
     * @return a new Emulation or null if an error occurred.
     */
    @Override
    public Emulation newEmulation(final EmulationUi ui,
                                  final EmulationConfiguration config) {
        if (config.getEmulatorId().equals("C64")) {
            return new C64Emulation(ui, config);
        }
        if (config.getEmulatorId().equals("C128")) {
            return new C128Emulation(ui, config);
        }
        if (config.getEmulatorId().equals("VIC20")) {
            return new Vic20Emulation(ui, config);
        }
        if (config.getEmulatorId().equals("PET")) {
            return new PetEmulation(ui, config);
        }
        if (config.getEmulatorId().equals("PLUS4")) {
            return new Plus4Emulation(ui, config);
        }

        return null;
    }
    @Override
    public DownloaderFactory.Downloader getPrepareEmulationRunnable(
            final Context context, final String emulatorId) {
        try {
            return new DownloaderFactory(context).createAssetDownloader(
                    Arrays.asList("VICE_RESOURCES/" + emulatorId,
                            "VICE_RESOURCES/DRIVES", "VICE_RESOURCES/PRINTER")
            );
        } catch (IOException e) {
            return null;
        }
    }


}
