//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;

import android.content.res.Resources;
import android.net.Uri;

import androidx.annotation.Keep;

import de.rainerhock.eightbitwonders.EmulationConfiguration;
import de.rainerhock.eightbitwonders.EmulationUi;
import de.rainerhock.eightbitwonders.R;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Map;

/**
 * Emulation for the VIC20/VC20/VIC1001.
 */
@Keep
final class Vic20Emulation extends ViceEmulation {
    /**
     * Constructor for VIC20-Emulation.
     * @param ui Implementation of the {@link de.rainerhock.eightbitwonders.EmulationUi}
     * interface for Callbacks to Userinterface
     * @param config Configuration
     */
    Vic20Emulation(final EmulationUi ui, final EmulationConfiguration config) {
        super(ui, config);
        addC64ToVicKeyMappings();
    }
    private void addC64ToVic20KeyMapping(final int c64row, final int c64column,
                                         final int vic20row, final int vic20column) {
        mMappings.add(new Mapper(this, c64row, c64column, vic20row, vic20column, true));
        mMappings.add(new Mapper(this, c64row, c64column, vic20row, vic20column, false));
    }
    //CHECKSTYLE DISABLE MagicNumber FOR 30 LINES
    private void addC64ToVicKeyMappings() {

        addC64ToVic20KeyMapping(7, 7, 0, 3);
        addC64ToVic20KeyMapping(7, 0, 0, 0);
        addC64ToVic20KeyMapping(7, 3, 0, 7);
        addC64ToVic20KeyMapping(1, 3, 1, 7);
        addC64ToVic20KeyMapping(2, 3, 2, 7);
        addC64ToVic20KeyMapping(3, 3, 3, 7);
        addC64ToVic20KeyMapping(4, 3, 4, 7);
        addC64ToVic20KeyMapping(5, 3, 5, 7);
        addC64ToVic20KeyMapping(0, 0, 7, 0);
        addC64ToVic20KeyMapping(7, 2, 0, 2);
        addC64ToVic20KeyMapping(7, 6, 0, 6);
        addC64ToVic20KeyMapping(0, 1, 7, 1);
        addC64ToVic20KeyMapping(0, 5, 7, 5);
        addC64ToVic20KeyMapping(7, 5, 0, 5);
        addC64ToVic20KeyMapping(7, 1, 0, 1);
        addC64ToVic20KeyMapping(1, 7, 1, 3);
        addC64ToVic20KeyMapping(2, 7, 2, 3);
        addC64ToVic20KeyMapping(3, 7, 3, 3);
        addC64ToVic20KeyMapping(4, 7, 4, 3);
        addC64ToVic20KeyMapping(5, 7, 5, 3);
        addC64ToVic20KeyMapping(6, 7, 6, 3);
        addC64ToVic20KeyMapping(7, 4, 0, 4);
        addC64ToVic20KeyMapping(0, 4, 7, 4);
        addC64ToVic20KeyMapping(0, 6, 7, 6);
        addC64ToVic20KeyMapping(0, 3, 7, 7);
        addC64ToVic20KeyMapping(6, 3, 6, 7);
        addC64ToVic20KeyMapping(0, 7, 7, 3);
        addC64ToVic20KeyMapping(0, 2, 7, 2);
    }
    @Override
    public HashMap<Integer, String> getJoystickports() {
        Resources res = getEmulationActivity().getContext().getResources();
        HashMap<Integer, String> ret = new LinkedHashMap<>();
        ret.put(1, res.getString(R.string.IDS_JOYSTICK));
        ret.put(3, res.getString(R.string.IDS_JOYSTICK_IN_USERPORT_PORT_1));
        ret.put(4, res.getString(R.string.IDS_JOYSTICK_IN_USERPORT_PORT_2));
        ret.put(KEYMAPPED_JOYSTICK_1, res.getString(R.string.map_to_keyset_1));
        ret.put(KEYMAPPED_JOYSTICK_2, res.getString(R.string.map_to_keyset_2));
        fixJoystickNames(ret);
        return ret;
    }

    static class Mapper {
        private static final int FLAG_KEY_IS_CONTROL = 16384;
        private final Vic20Emulation mEmu;
        private final int mC64Row;
        private final int mC64Column;
        private final int mVic20Row;
        private final int mVic20Column;
        private final boolean mKeypressed;

        Mapper(final Vic20Emulation emu, final int c64row, final int c64column,
               final int vic20row, final int vic20column,
               final boolean keypressed) {
            mEmu = emu;
            mC64Row = c64row;
            mC64Column = c64column;
            mVic20Row = vic20row;
            mVic20Column = vic20column;
            mKeypressed = keypressed;

        }

        boolean mappedKeyAction(final int buttonId, final int row, final int column,
                                final int shift, final boolean keypressed) {
            if (row == mC64Row && column == mC64Column && mKeypressed == keypressed
                    && (shift & FLAG_KEY_IS_CONTROL) == 0) {
                if (mKeypressed) {
                    mEmu.c64keypressed(buttonId, mVic20Row, mVic20Column, shift);
                } else {
                    mEmu.c64keyreleased(buttonId, mVic20Row, mVic20Column, shift);
                }
                return true;
            }
            return false;
        }
    }

    private final LinkedList<Mapper> mMappings = new LinkedList<>();

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private boolean doMappedKeys(final int buttonId, final int row, final int column,
                                 final int shift, final boolean isPressed) {
        for (Mapper m : mMappings) {
            if (m.mappedKeyAction(buttonId, row, column, shift, isPressed)) {
                return true;
            }
        }
        return false;
    }

    void c64keypressed(final int buttonId, final int row, final int column, final int shift) {
        super.keypressed(buttonId, row, column, shift);
    }

    void c64keyreleased(final int buttonId, final int row, final int column, final int shift) {
        super.keyreleased(buttonId, row, column, shift);
    }

    void keypressed(final int buttonId, final int row, final int column, final int shift) {
        if (!doMappedKeys(buttonId, row, column, shift, true)) {
            super.keypressed(buttonId, row, column, shift);
        }
    }

    void keyreleased(final int buttonId, final int row, final int column, final int shift) {
        if (!doMappedKeys(buttonId, row, column, shift, false)) {
            super.keyreleased(buttonId, row, column, shift);
        }
    }

    @Override
    protected int getCompactKeyboardLayoutId(final int model) {
        return R.layout.fragment_c64_keyboard;
    }
    protected int getExactKeyboardLayoutId(final int model) {
        return R.layout.fragment_c64_keyboard_fullsized;
    }

    private static final int VIC20_CARTRIDGE_AUTODETECT_FLAG = 32768;
    @Override
    int getCartridgeAutodetectFlag() {
        return VIC20_CARTRIDGE_AUTODETECT_FLAG;
    }

    private static final  Map<String, String> PARAMETER_MAPPING
            = new LinkedHashMap<String, String>() {{
        put("Model", "-model");
    }};
    @Override
    protected ViceMachineSettingsFunctions getViceMachineSettingsFunctions() {
        return new ViceMachineSettingsFunctions(
                Arrays.asList(R.id.gh_vic20settings, R.id.gh_vic_i_settings, R.id.sidsettings,
                        R.id.vic20model),
                Arrays.asList(R.id.gh_vic20settings, R.id.gh_vic_i_settings, R.id.sidsettings),
                PARAMETER_MAPPING) {
            @Override
            protected Map<String, String> getDeviceSpecificDefaultValues() {
                Map<String, String> ret = new HashMap<>(super.getDeviceSpecificDefaultValues());
                ret.put("MachineVideoStandard",
                        getEmulationActivity().isInNtscRegion() ? "2" : "1");
                ret.put("VICDoubleSize", "1");
                ret.put("VICFilter", "0");
                ret.putAll(super.getDeviceSpecificDefaultValues());

                return ret;
            }

        };

    }
    private static final int KBYTE = 1024;
    @Override
    protected boolean isCartridge(final Uri uri) {
        String uriAdress = uri.toString();
        if (uriAdress.endsWith(".prg") || uriAdress.endsWith(".bin")) {
            try {
                InputStream is = getEmulationActivity().getContext()
                        .getContentResolver().openInputStream(uri);
                if (is != null) {
                    boolean ret = is.available() % KBYTE == 2;
                    is.close();
                    return ret;
                }
                return false;

            } catch (IOException e) {
                return false;
            }
        } else {
            return false;
        }
    }
}
