//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;

import android.annotation.SuppressLint;
import android.content.res.Resources;
import android.net.Uri;

import androidx.annotation.Keep;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import de.rainerhock.eightbitwonders.EmulationConfiguration;
import de.rainerhock.eightbitwonders.EmulationUi;
import de.rainerhock.eightbitwonders.R;

/**
 * Emulation for the C64.
 */
@Keep
public final class Plus4Emulation extends ViceEmulation {
    /**
     * Constructor for C64-Emulation.
     * @param ui Implementation of the {@link EmulationUi}
     * interface for Callbacks to Userinterface
     * @param config Configuration
     */
    public Plus4Emulation(final EmulationUi ui, final EmulationConfiguration config) {
        super(ui, config);
    }
    @Override
    protected boolean checkRequiredFiles() {
        return true;
    }
    @Override
    protected int getCompactKeyboardLayoutId(final int model) {
        return R.layout.fragment_plus4_keyboard;
    }
    private static final int PLUS4MODEL_PLUS4_PAL = 2;
    private static final int PLUS4MODEL_PLUS4_NTSC = 3;
    private static final int PLUS4MODEL_C16_PAL = 0;
    private static final int PLUS4MODEL_C16_NTSC = 1;
    @Override
    protected int getExactKeyboardLayoutId(final int model) {
        switch (model) {
            case PLUS4MODEL_PLUS4_PAL:
            case PLUS4MODEL_PLUS4_NTSC:
                //noinspection DuplicateBranchesInSwitch
                return R.layout.fragment_plus4_keyboard_fullsized;
            case PLUS4MODEL_C16_PAL:
            case PLUS4MODEL_C16_NTSC:
                return R.layout.fragment_c16_keyboard_fullsized;
            default:
                return R.layout.fragment_plus4_keyboard_fullsized;
        }
    }

    @Override
    public HashMap<Integer, String> getJoystickports() {
        HashMap<Integer, String> ret = new LinkedHashMap<>();
        Resources res = getEmulationActivity().getContext().getResources();
        ret.put(ret.size() + 1, res.getString(R.string.IDS_JOYSTICK_IN_PORT_1));
        ret.put(ret.size() + 1, res.getString(R.string.IDS_JOYSTICK_IN_PORT_2));
        ret.put(ret.size() + 1, res.getString(R.string.IDS_JOYSTICK_IN_USERPORT_PORT_1));
        ret.put(ret.size() + 1, res.getString(R.string.IDS_JOYSTICK_IN_USERPORT_PORT_2));
        ret.put(KEYMAPPED_JOYSTICK_1, res.getString(R.string.map_to_keyset_1));
        ret.put(KEYMAPPED_JOYSTICK_2, res.getString(R.string.map_to_keyset_2));

        return ret;
    }

    @SuppressLint("DiscouragedApi")
    @Override
    protected LinkedHashMap<String, Integer> getCartrigeFilters() {
        return new LinkedHashMap<>();
    }

    @Override
    int getCartridgeAutodetectFlag() {
        return 0;
    }

    @Override
    protected boolean isCartridge(final Uri uri) {
        return nativeIntFunc("cartridge_detect", uri.toString()) >= 0;
    }

    @Override
    protected ViceMachineSettingsFunctions getViceMachineSettingsFunctions() {

        return new ViceMachineSettingsFunctions("plus4model_set",
                Arrays.asList("MachineVideoStandard", "RamSize", "Acia1Enable", "SpeechEnabled"),
                Arrays.asList("KernalName", "ChargenName", "FunctionLowName", "FunctionHighName",
                        "c2loName"),
                Arrays.asList(R.id.gh_plus4settings, R.id.ted_settings,
                        R.id.plus4_audio_options, R.id.sidsettings),
                Arrays.asList(R.id.ted_settings, R.id.plus4_audio_options, R.id.sidsettings),
                null) {
            @Override
            protected String getDeviceSpecificModelDefault() {
                return getEmulationActivity().isInNtscRegion() ? "3" : "2";
            }
            @Override
            protected Map<String, String> getDeviceSpecificDefaultValues() {
                LinkedHashMap<String, String> ret = new LinkedHashMap<>(
                        super.getDeviceSpecificDefaultValues());
                ret.put("TEDDoubleSize", "1");
                ret.put("TEDFilter", "0");
                return ret;
            }
        };
    }
}
