//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;


import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;
import android.util.Log;
import android.view.KeyEvent;
import android.view.MotionEvent;

import de.rainerhock.eightbitwonders.EmulationUi;


/**
 * View for a virtual keyboard-key.
 */
@SuppressLint({"ClickableViewAccessibility"})
public class KeyButton extends KeyButtonBase {
    private static final String TAG = KeyButtonBase.class.getSimpleName();

    /**
     * Simple constructor to use when creating a KeyButton from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public KeyButton(final Context context) {
        super(context);
        init(context, null);
    }
    /**
     * Constructor that is called when inflating a KeyButton from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */

    public KeyButton(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public KeyButton(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
        init(context, attrs);
    }
    @Override
    final void setKeyboardController(final ViceKeyboardFragment val) {
        mKeyboardFragment = val;
    }
    private boolean mKeyDown = false;
    private ViceKeyboardFragment mKeyboardFragment = null;
    @Override
    public final boolean onKeyDown(final int keyCode, final KeyEvent event) {
        boolean superRet = super.onKeyDown(keyCode, event);
        if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(event.getKeyCode())) {
            return true;
        }
        if (mKeyboardFragment.delegateToActivity(keyCode, event)) {
            return true;
        }
        if (!mKeyDown && !isRemoteButton(keyCode)) {
            mKeyboardFragment.pressKey(this);
            mKeyboardFragment.dumpstate();
            mKeyDown = true;
            return true;
        }

        return superRet;
    }

    @Override
    public final boolean onKeyUp(final int keyCode, final KeyEvent event) {
        boolean superRet = super.onKeyUp(keyCode, event);
        if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(event.getKeyCode())) {
            mKeyboardFragment.dismiss();

            return true;
        }
        if (mKeyDown && !isRemoteButton(keyCode)) {
            mKeyboardFragment.releaseKey(this);
            mKeyboardFragment.dumpstate();
            mKeyDown = false;
            return true;
        }
        if (mKeyboardFragment.delegateToActivity(keyCode, event)) {
            return true;
        }
        return superRet;
    }

    @Override
    public final boolean onTouchEvent(final MotionEvent event) {
        if (event.getActionMasked() == MotionEvent.ACTION_DOWN
                || event.getActionMasked() == MotionEvent.ACTION_POINTER_DOWN) {
            Log.d(TAG, "ksh.pressKey ('" + this.getMainText() + "')");
            mKeyboardFragment.pressKey(this);
            mKeyboardFragment.dumpstate();
        }
        if (event.getActionMasked() == MotionEvent.ACTION_UP
                || event.getActionMasked() == MotionEvent.ACTION_POINTER_UP) {
            Log.d(TAG, "ksh.releaseKey ('" + this.getMainText() + "')");
            mKeyboardFragment.releaseKey(this);
            mKeyboardFragment.dumpstate();

        }
        return super.onTouchEvent(event);
    }

}
