package de.rainerhock.eightbitwonders.vice;

import android.annotation.SuppressLint;
import android.content.res.Resources;
import android.content.res.TypedArray;
import android.graphics.Point;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import de.rainerhock.eightbitwonders.EmulationConfiguration;
import de.rainerhock.eightbitwonders.EmulationUi;
import de.rainerhock.eightbitwonders.R;
import de.rainerhock.eightbitwonders.Useropts;

class C128Emulation extends C64C128CommonBase {
    C128Emulation(final EmulationUi ui, final EmulationConfiguration conf) {

        super(ui, conf);
        Objects.requireNonNull(getDualMonitorFunctions())
                .setActiveMonitor(getDualMonitorFunctions().getDefaultMonitor());
    }

    @Override
    protected int getCompactKeyboardLayoutId(final int model) {
        return R.layout.fragment_c128_keyboard;
    }

    @Override
    protected int getExactKeyboardLayoutId(final int model) {
        return R.layout.fragment_c128_keyboard_fullsized;
    }

    private static final ArrayList<Integer> CHARSET_TOGGLE_KEYS =
            new ArrayList<>(Arrays.asList(R.id.ASCII_DIN, R.id.ASCII_CC, R.id.CAPSLOCK));
    @Nullable
    @Override
    protected ArrayList<Integer> getKeysToHighlight() {
        ArrayList<Integer> ret = new ArrayList<>();
        if (mInternational) {
            ret.addAll(CHARSET_TOGGLE_KEYS);
        }
        if (getEmulationActivity().getCurrentUseropts()
                .getBooleanValue("key_40_80_pressed", false)) {
            ret.add(R.id.toggle_40_80);
        }
        return ret;

    }

    @Nullable
    @Override
    protected ArrayList<Integer> getKeysToDeHighlight() {
        if (!mInternational) {
            return CHARSET_TOGGLE_KEYS;
        }

        return super.getKeysToDeHighlight();
    }

    @Override
    protected ViceMachineSettingsFunctions getViceMachineSettingsFunctions() {

        return new ViceMachineSettingsFunctions("c128model_set",
                Collections.emptyList(),
                Collections.emptyList(),
                Arrays.asList(R.id.gh_c128_settings, R.id.gh_vic_ii_vdc_settings, R.id.sidsettings,
                        R.id.c128model),
                Arrays.asList(R.id.gh_c128_settings, R.id.gh_vic_ii_vdc_settings, R.id.sidsettings),
                null) {

            @Override
            protected Map<String, String> getDeviceSpecificDefaultValues() {
                Map<String, String> ret = new LinkedHashMap<>();
                ret.put("VICIIDoubleSize", "1");
                ret.put("VICIIFilter", "0");
                ret.put("VDCDoubleSize", "1");
                ret.put("VDCFilter", "0");

                ret.put("Drive8Type", "1571");
                ret.put("Model", "0");
                return ret;
            }
            @Override
            protected String getDeviceSpecificModelDefault() {
                return "2";
            }
        };
    }
    @Override
    protected boolean checkRequiredFiles() {
        return true;
    }
    @Nullable
    @Override
    public DualMonitorFunctions getDualMonitorFunctions() {
        return new DualMonitorFunctions() {
            @NonNull
            @Override
            public Map<Integer, String> getDisplays() {
                Map<Integer, String> ret = new HashMap<>();
                String[] values = getEmulationActivity().getContext()
                        .getResources().getStringArray(R.array.c18displays);
                for (int i = 0; i < values.length; i++) {
                    ret.put(i, values[i]);
                }
                return ret;
            }

            @Override
            public int getDefaultMonitor() {
                Useropts opts = getEmulationActivity().getCurrentUseropts();
                if (opts.getBooleanValue("key_40_80_pressed", false)) {
                    return 0;
                } else {
                    return 1;
                }
            }

            @Override
            public void setActiveMonitor(final int monitor) {
                nativeVoidFunc("set_active_monitor", monitor);

            }
        };
    }

    @Nullable
    @Override
    protected ArrayList<Integer> getFixedKeysToHide() {
        ArrayList<Integer> ret = new  ArrayList<>(Arrays.asList(R.id.F1, R.id.F3, R.id.F5, R.id.F7,
                R.id.c64onlyspacer1, R.id.c64onlyspacer2, R.id.c64onlyspacer3,
                R.id.c64onlyspacer4, R.id.c64onlyspacer5, R.id.c64onlyspacer6,
                R.id.c64onlyspacer7, R.id.c64onlyspacer8));
        Resources res = getEmulationActivity().getContext().getResources();
        if (res.getBoolean(R.bool.c128_keyboard_medium_size)) {
            ret.add(R.id.numrowdummy2);
        }
        if (res.getBoolean(R.bool.c128_keyboard_large_size)) {
            ret.add(R.id.alpha_num_switch);
        }
        return ret;
    }

    @Nullable
    @Override
    protected ArrayList<Integer> getDynamicKeysToHide() {
        ArrayList<Integer> ret = new ArrayList<>();
        if (mInternational) {
            for (int i: languageSpecificKeys(true)) {
                ret.add(i);
            }
        }
        return ret;
    }

    @Nullable
    @Override
    protected ArrayList<Integer> getDynamicKeysToShow() {
        ArrayList<Integer> ret = new ArrayList<>();
        if (mInternational) {
            for (int i: languageSpecificKeys(false)) {
                ret.add(i);
            }
        }
        return ret;

    }

    @Nullable
    @Override
    protected ArrayList<Integer> getFixedKeysToShow() {
        ArrayList<Integer> ret = new ArrayList<>(
                Arrays.asList(R.id.c128specialkeys, R.id.c128onlyspacer1, R.id.c128onlyspacer2,
                        R.id.c128onlyspacer3, R.id.ESC, R.id.TAB, R.id.ALT));
        int index = Integer.parseInt(getEmulationActivity().getCurrentUseropts()
                .getStringValue("C128MachineType", "0"));
        if (index == 0) {
            ret.add(R.id.CAPSLOCK);
        } else if (index == 3) {
            ret.add(R.id.ASCII_DIN);
        } else {
            ret.add(R.id.ASCII_CC);
        }
        Resources res = getEmulationActivity().getContext().getResources();
        if (res.getBoolean(R.bool.c128_keyboard_large_size)) {
            ret.add(R.id.extra_spacer_largesize1);
            ret.add(R.id.extra_spacer_largesize2);
        }

        return ret;

    }
    private static final int KEY_40_80_ROW = -4;
    private static final int KEY_40_80_COL = 0;
    private static final int KEY_ASCII_CC_ROW = -4;
    private static final int KEY_ASCII_CC_COL = 1;

    static final Map<Point, String> TOGGLE_KEYS_TO_SAVE = new HashMap<Point, String>() {{
        put(new Point(KEY_40_80_ROW, KEY_40_80_COL), "key_40_80_pressed");
        //put(new Point(KEY_ASCII_CC_ROW, KEY_ASCII_CC_COL), "key_ascii_cc_pressed");
    }};
    @NonNull
    @Override
    protected List<String> extractSettingsToCmdLine(final Useropts opts) {
        List<String> ret = super.extractSettingsToCmdLine(opts);
        if (opts.getBooleanValue("key_40_80_pressed", false)) {
            ret.add("-80col");
        }
        return ret;
    }
    @Override
    boolean isToggleKeyPressed(final int row, final int col) {
        boolean ret = super.isToggleKeyPressed(row, col);
        if (!ret) {
            Point p = new Point(row, col);
            if (TOGGLE_KEYS_TO_SAVE.containsKey(p)) {
                String key = TOGGLE_KEYS_TO_SAVE.get(p);
                if (key != null) {
                    return getEmulationActivity().getCurrentUseropts().getBooleanValue(key, false);
                }
            }
        }
        return ret;
    }
    @Override
    Point setToggleKeyPressed(final int row, final int col, final boolean pressed) {
        Point ret = super.setToggleKeyPressed(row, col, pressed);
        if (TOGGLE_KEYS_TO_SAVE.containsKey(ret)) {
            String key = TOGGLE_KEYS_TO_SAVE.get(ret);
            if (key != null) {
                getEmulationActivity().getCurrentUseropts()
                        .setValue(Useropts.Scope.CONFIGURATION, key, pressed);
            }
        }

        return ret;
    }
    private int[] languageSpecificKeys(final boolean international) {
        Resources res = getEmulationActivity().getContext().getResources();
        int index = Integer.parseInt(getEmulationActivity().getCurrentUseropts()
                .getStringValue("C128MachineType", "0"));
        //noinspection resource
        @SuppressLint("Recycle")
        TypedArray ta = res.obtainTypedArray(international ? R.array.intl_keys : R.array.natl_keys);
        //noinspection resource
        @SuppressLint("Recycle")
        TypedArray ta2 = res.obtainTypedArray(ta.getResourceId(index, 0));

        int[] ret = new int[ta2.length()];
        for (int i = 0; i < ret.length; i++) {
            ret[i] = ta2.getResourceId(i, 0);
        }
        ta2.recycle();
        ta.recycle();
        return ret;


    }
    private boolean mInternational = false;
    @Override
    boolean toggleKeyset(final int row, final int col) {
        if (row == KEY_ASCII_CC_ROW && col == KEY_ASCII_CC_COL) {
            mInternational = !mInternational;
            return true;
        }
        return false;
    }

    @Override
    int[] hideKeysAfterKeypress(final int row, final int col) {
        if (row == KEY_ASCII_CC_ROW && col == KEY_ASCII_CC_COL) {

            return languageSpecificKeys(mInternational);
        }
        return null;
    }
    @Override
    int[] showKeysAfterKeypress(final int row, final int col) {
        if (row == KEY_ASCII_CC_ROW && col == KEY_ASCII_CC_COL) {
            return languageSpecificKeys(!mInternational);
        }
        return null;
    }
    /** @noinspection unused*/
    @Keep
    private void resetToIntlKeyboard() {
        mInternational = "5".equals(getEmulationActivity().getCurrentUseropts()
                .getStringValue("C128MachineType", null));
        setToggleKeyPressed(KEY_ASCII_CC_ROW, KEY_ASCII_CC_COL, mInternational);
    }
}
