//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;

import org.yaml.snakeyaml.TypeDescription;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;
import org.yaml.snakeyaml.error.YAMLException;

import java.io.Reader;
import java.io.Serializable;
import java.io.Writer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Class to store an retrieve useropts.
 * Useropts can be stored in one of the scopes in {@link Scope}
 * When retrieving them, the class will use the value stored to the most granular matching scope.
 */
@Keep
public final class Useropts implements Serializable {
    private final List<SharedPreferences> mSearchpath = new LinkedList<>();


    Useropts() {
        super();
    }
    Useropts(final Context context) {
        super();
        SharedPreferences global = context.getSharedPreferences(
                context.getPackageName() + ".global", Context.MODE_PRIVATE);
        mSharedPrefKeys.put(Scope.GLOBAL, global);
        mSearchpath.clear();
        mSearchpath.add(global);
    }

    static void delete(final Context context, final String emu, final String confId) {
        context.getSharedPreferences(
                context.getPackageName() + ".emu_" + emu + ".conf_" + confId,
                Context.MODE_PRIVATE).edit().clear().apply();
    }
    @SuppressLint("ApplySharedPref")
    static void deleteNow(final Context context, final String emu, final String confId) {
        context.getSharedPreferences(
                context.getPackageName() + ".emu_" + emu + ".conf_" + confId,
                Context.MODE_PRIVATE).edit().clear().commit();
    }

    /**
     * Scopes to share the useropts.
     */
    public enum Scope { SESSION, CONFIGURATION, EMULATOR, GLOBAL
    }

    private final Map<Scope, SharedPreferences> mSharedPrefKeys = new LinkedHashMap<>();

    private String mEmulatorId = null;
    private String mConfigurationId = null;
    void setCurrentEmulation(final Context context,
                             final String emulatorId,
                             final String configurationId) {


        mEmulatorId = emulatorId;
        mConfigurationId = configurationId;
        mSharedPrefKeys.put(Scope.EMULATOR, context.getSharedPreferences(
                context.getPackageName() + ".emu_" + emulatorId + ".emulator",
                Context.MODE_PRIVATE));
        if (configurationId != null) {
            mSharedPrefKeys.put(Scope.CONFIGURATION, context.getSharedPreferences(
                    context.getPackageName() + ".emu_" + emulatorId + ".conf_" + configurationId,
                    Context.MODE_PRIVATE));
        }
        mSharedPrefKeys.put(Scope.GLOBAL, context.getSharedPreferences(
                context.getPackageName() + ".global",
                Context.MODE_PRIVATE));
        synchronized (mSearchpath) {
            mSearchpath.clear();
            for (Scope scope : new Scope[]{Scope.CONFIGURATION, Scope.EMULATOR, Scope.GLOBAL}) {
                SharedPreferences val = mSharedPrefKeys.get(scope);
                if (val != null) {
                    mSearchpath.add(val);
                }

            }
        }
        resetCurrentEmulation();
    }

    void resetCurrentEmulation() {
        mStringPreferences.clear();
        mBooleanPreferences.clear();
        mIntegerPreferences.clear();
    }

    private SharedPreferences getWritePrefs(final Scope scope) {
        return mSharedPrefKeys.get(scope);
    }

    private SharedPreferences getReadPrefs(final String key) {
        synchronized (mSearchpath) {
            for (SharedPreferences prefs : mSearchpath) {
                if (prefs != null && prefs.contains(key)) {
                    return prefs;
                }
            }
            return null;
        }
    }

    interface Setter<T> {
        void setValue(SharedPreferences prefs, String key, T value);
    }

    interface Getter<T> {
        T getValue(SharedPreferences prefs, String key);
    }

    class TypedPreferences<T> implements Serializable {
        private final Setter<T> mSetter;
        private final Getter<T> mGetter;
        private final Map<String, T> mSessionValues;

        TypedPreferences(final Setter<T> setter, final Getter<T> getter) {
            mSetter = setter;
            mGetter = getter;
            mSessionValues = new LinkedHashMap<>();
        }

        void clear() {
            mSessionValues.clear();
        }

        T getValue(final String key, final T defaultval) {
            if (mSessionValues.containsKey(key)) {
                return mSessionValues.get(key);
            }
            SharedPreferences prefs = getReadPrefs(key);
            if (prefs != null) {
                return mGetter.getValue(prefs, key);
            }
            return defaultval;

        }

        void setValue(final Scope scope, final String key, final T value) {
            if (scope == Scope.SESSION) {
                mSessionValues.put(key, value);
                return;
            }

            SharedPreferences prefs = getWritePrefs(scope);
            if (prefs != null) {
                mSetter.setValue(prefs, key, value);
            }

        }
    }

    private final TypedPreferences<String> mStringPreferences = new TypedPreferences<>(
            (prefs, key, value) -> prefs.edit().putString(key, value).apply(),
            (prefs, key) -> prefs.getString(key, null)
    );
    private final TypedPreferences<Integer> mIntegerPreferences = new TypedPreferences<>(
            (prefs, key, value) -> prefs.edit().putInt(key, value).apply(),
            (prefs, key) -> prefs.getInt(key, 0)
    );
    private final TypedPreferences<Boolean> mBooleanPreferences = new TypedPreferences<>(
            (prefs, key, value) -> prefs.edit().putBoolean(key, value).apply(),
            (prefs, key) -> prefs.getBoolean(key, false)
    );

    /**
     * Check if useropts are stored.
     * @return true if there are stored useropts otherwise false.
     */
    public boolean isEmpty() {
        return getStringKeys().isEmpty()
                && getIntegerKeys().isEmpty()
                && getBooleanKeys().isEmpty();
    }
    Scope getScope(final String key) {
        if (key.startsWith("__") && key.endsWith("__")) {
            return Scope.CONFIGURATION;
        }
        for (Scope scope :Arrays.asList(Scope.SESSION, Scope.CONFIGURATION,
                Scope.EMULATOR, Scope.GLOBAL)) {
            SharedPreferences prefs = mSharedPrefKeys.get(scope);
            if (prefs != null && prefs.contains(key)) {
                return scope;
            }
        }
        return null;
    }
    /**
     * Check if useropts have values that differ from the configuration.
     * @param conf #EmulationConfiguration to compare with
     * @return true if there are keys that differ from configuration.
     */
    boolean isDifferent(final EmulationConfiguration conf) {
        for (String key : getIntegerKeys()) {
            if (Scope.CONFIGURATION.equals(getScope(key))) {
                try {
                    int compare = Integer.parseInt(conf.getProperty(key, "-1"));
                    if (getIntegerValue(key, compare) != compare) {
                        return true;
                    }
                } catch (NumberFormatException e) {
                    return true;
                }
            }
        }
        for (String key : getStringKeys()) {
            if (Scope.CONFIGURATION.equals(getScope(key))) {
                if (!getStringValue(key, conf.getProperty(key, ""))
                        .equals(conf.getProperty(key, ""))) {
                    return true;
                }
            }
        }
        for (String key : getBooleanKeys()) {
            if (Scope.CONFIGURATION.equals(getScope(key))) {
                if (Boolean.valueOf(conf.getProperty(key, "false"))
                        != getBooleanValue(key,
                        Boolean.valueOf(conf.getProperty(key, "false")))) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * get all string-based useropts for the current emulation and configuration.
     * @return useropt keys.
     */
    public Set<String> getStringKeys() {
        return getTypeKeys(String.class);
    }

    /**
     * get all boolean-based useropts for the current emulation and configuration.
     * @return useropt keys.
     */
    public Set<String> getBooleanKeys() {
        return getTypeKeys(Boolean.class);
    }

    private Set<String> getTypeKeys(final Class<?> cls) {
        List<String> values = new LinkedList<>();
        synchronized (mSearchpath) {
            for (SharedPreferences prefs : mSearchpath) {
                if (prefs != null) {
                    for (String key : prefs.getAll().keySet()) {
                        if (cls == String.class && prefs.getAll().get(key) instanceof String) {
                            values.add(key);
                        }
                        if (cls == Integer.class && prefs.getAll().get(key) instanceof Integer) {
                            values.add(key);
                        }
                        if (cls == Boolean.class && prefs.getAll().get(key) instanceof Boolean) {
                            values.add(key);
                        }

                    }
                }
            }
        }
        return new HashSet<>(values);
    }
    /**
     * get all integer-based useropts for the current emulation and configuration.
     * @return useropt keys.
     */

    public Set<String> getIntegerKeys() {
        return getTypeKeys(Integer.class);
    }

    /**
     * Set an integer value.
     * @param scope the scope where the value is to be placed.
     * @param key useropts-key
     * @param value value to be stored
     */
    public void setValue(final Scope scope, final String key, final Integer value) {
        mIntegerPreferences.setValue(scope, key, value);
    }
    /**
     * Set a boolean value.
     * @param scope the scope where the value is to be placed.
     * @param key useropts-key
     * @param value value to be stored
     */

    public void setValue(final Scope scope, final String key, final Boolean value) {
        mBooleanPreferences.setValue(scope, key, value);
    }
    /**
     * Set a string value.
     * @param scope the scope where the value is to be placed.
     * @param key useropts-key
     * @param value value to be stored
     */
    public void setValue(final Scope scope, final String key, final String value) {
        mStringPreferences.setValue(scope, key, value);
    }

    /**
     * Get a string value.
     * @param key useropts-key
     * @param defaultval value to be returned if the useropt has not been set.
     * @return the best-matching useropt or the defaultval, if no useropt matched.
     */
    public String getStringValue(final String key, final String defaultval) {
        try {
            return mStringPreferences.getValue(key, defaultval);
        } catch (ClassCastException e) {
            return defaultval;
        }
    }
    /**
     * Get an integer value.
     * @param key useropts-key
     * @param defaultval value to be returned if the useropt has not been set.
     * @return the best-matching useropt or the defaultval, if no useropt matched.
     */

    public Integer getIntegerValue(final String key, final Integer defaultval) {
        try {
            return mIntegerPreferences.getValue(key, defaultval);
        } catch (ClassCastException e) {
            return defaultval;
        }
    }
    /**
     * Get a boolean value.
     * @param key useropts-key
     * @param defaultval value to be returned if the useropt has not been set.
     * @return the best-matching useropt or the defaultval, if no useropt matched.
     */
    public Boolean getBooleanValue(final String key, final Boolean defaultval) {
        try {
            return mBooleanPreferences.getValue(key, defaultval);
        } catch (ClassCastException e) {
            return defaultval;
        }
    }
    void copyToConfiguration(final Context context, final String destConfig) {
        SharedPreferences source = mSharedPrefKeys.get(Scope.CONFIGURATION);
        SharedPreferences dest = context.getSharedPreferences(
                context.getPackageName() + ".emu_" + mEmulatorId + ".conf_" + destConfig,
                Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = dest.edit();
        editor.clear();
        if (source != null) {
            for (Map.Entry<String, ?> entry : source.getAll().entrySet()) {
                Object value = entry.getValue();
                String key = entry.getKey();
                if (value instanceof String) {
                    editor = editor.putString(key, ((String) value));
                } else if (value instanceof Integer) {
                    editor = editor.putInt(key, (Integer) value);
                } else if (value instanceof Long) {
                    editor = editor.putLong(key, (Long) value);
                } else if (value instanceof Float) {
                    editor = editor.putFloat(key, (Float) value);
                } else if (value instanceof Boolean) {
                    editor = editor.putBoolean(key, (Boolean) value);
                }

            }
        }
        editor.apply();
    }
    // CHECKSTYLE DISABLE MemberName FOR 10 LINES
    // CHECKSTYLE DISABLE VisibilityModifier FOR 10 LINES
    @Keep
    private static final class DataObject {
        String emulatorId = null;
        String configurationId = null;
        Map<String, String> stringValues = new LinkedHashMap<>();
        Map<String, Integer> intValues = new LinkedHashMap<>();
        Map<String, Boolean> boolValues = new LinkedHashMap<>();
    }
    void readFromYaml(final Context context, final Reader reader) {
        DataObject repr = new Yaml(getJsonReprConstructor()).load(reader);
        Scope scope;
        if (repr.configurationId != null && repr.emulatorId != null) {
            setCurrentEmulation(context, repr.emulatorId, repr.configurationId);
            scope = Scope.CONFIGURATION;
        } else if (repr.configurationId == null & repr.emulatorId != null) {
            setCurrentEmulation(context, repr.emulatorId, repr.configurationId);
            scope = Scope.EMULATOR;
        } else {
            scope = Scope.GLOBAL;
        }
        for (String key : repr.stringValues.keySet()) {
            setValue(scope, key, repr.stringValues.get(key));
        }
        for (String key : repr.intValues.keySet()) {
            setValue(scope, key, repr.intValues.get(key));
        }
        for (String key : repr.boolValues.keySet()) {
            setValue(scope, key, repr.boolValues.get(key));
        }
    }
    void yamlify(final Scope scope, final Writer writer) throws YAMLException {
        SharedPreferences prefs = getWritePrefs(scope);
        DataObject repr = new DataObject();
        if (scope != Scope.GLOBAL) {
            repr.emulatorId = mEmulatorId;
            }
        if (scope == Scope.CONFIGURATION) {
            repr.configurationId = mConfigurationId;
        }
        Map<String, ?> all = prefs.getAll();
        for (String key : all.keySet()) {
            if (getScope(key) == scope) {
                Object value = all.get(key);
                if (value instanceof String) {
                    repr.stringValues.put(key, (String) value);
                }
                if (value instanceof Integer) {
                    repr.intValues.put(key, (Integer) value);
                }
                if (value instanceof Boolean) {
                    repr.boolValues.put(key, (Boolean) value);
                }
            }
        }
        Constructor constructor = getJsonReprConstructor();
        Yaml yaml = new Yaml(constructor);
        yaml.dump(repr, writer);
    }

    private static @NonNull Constructor getJsonReprConstructor() {
        Constructor constructor = new Constructor(DataObject.class);
        TypeDescription customTypeDescription = new TypeDescription(DataObject.class);
        customTypeDescription.addPropertyParameters("stringValues", String.class, String.class);
        customTypeDescription.addPropertyParameters("intValues", String.class, Integer.class);
        customTypeDescription.addPropertyParameters("boolValues", String.class, Boolean.class);

        constructor.addTypeDescription(customTypeDescription);
        return constructor;
    }

}
