//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.Joystick.PORT_NOT_CONNECTED;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Rect;
import android.util.AttributeSet;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.RelativeLayout;

import androidx.annotation.NonNull;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * This view splits all multi-touch events into single events and sends them to the correct view.
 * All other solutions suck.
 */
public class TouchDisplayRelativeLayout extends RelativeLayout {

    private static final float MIN_OPACITY = 0.2f;
    private static final float PERCENT_DIVIDER = 100;

    /**
     * Simple constructor to use when creating a TouchDisplayRelativeLayout from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */

    public TouchDisplayRelativeLayout(final Context context) {
        super(context);
        init();
    }
    /**
     * Constructor that is called when inflating a TouchDisplayRelativeLayout from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public TouchDisplayRelativeLayout(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public TouchDisplayRelativeLayout(final Context context, final AttributeSet attrs,
                                      final int defStyle) {
        super(context, attrs, defStyle);
        init();
    }

    private void init() {
        mPointerPropForReUse[0] = new MotionEvent.PointerProperties();
        mPointerCoordForReuse[0] = new MotionEvent.PointerCoords();
        if (findViewById(R.id.fragment_dynamic_ui_elements) == null) {
            LayoutInflater inflater = (LayoutInflater) getContext()
                    .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            inflater.inflate(R.layout.view_controls, this, true);
        }
    }
    @Override
    public final boolean onInterceptTouchEvent(final MotionEvent event) {
        return true;
    }
    private final Set<View> mViews = new LinkedHashSet<>();
    private final LinkedHashMap<Integer, View> mTouchedView = new LinkedHashMap<>();
    private final LinkedHashMap<Integer, View> mJoystickView = new LinkedHashMap<>();
    private static final List<Integer> VIEW_IDS = Arrays.asList(
            R.id.jv_directions, R.id.jv_fire,
            R.id.jv_flipswitch, R.id.jv_wheel,
            R.id.jv_secondary,
            R.id.ib_hamburger_menu,  R.id.scroll,
            R.id.softkey_container);
    final List<Integer> getUiElements() {
        return VIEW_IDS;
    }

    private boolean sendEvent(final View v, final MotionEvent e) {
        Log.v(getClass().getSimpleName(), "delegating event " + e + " to " + v);
        if (v.isShown() && v.isEnabled()) {
            return v.dispatchTouchEvent(e);
        }
        return false;

    }
    private final Rect mTempRect = new Rect();
    private boolean containsPosition(final View v, final float x, final float y) {
        // Convert touch coordinates from sourceView to screen coordinates
        int[] sourceOffset = new int[2];
        getLocationOnScreen(sourceOffset);
        float screenX = x + sourceOffset[0];
        float screenY = y + sourceOffset[1];

        // Get targetView bounds on screen
        int[] targetOffset = new int[2];
        v.getLocationOnScreen(targetOffset);
        Rect targetRect = new Rect(
                targetOffset[0],
                targetOffset[1],
                targetOffset[0] + v.getWidth(),
                targetOffset[1] + v.getHeight()
        );

        return targetRect.contains((int) screenX, (int) screenY);    }
    private final MotionEvent.PointerProperties[] mPointerPropForReUse
            = new MotionEvent.PointerProperties[1];
    private static final List<Integer> PRESS_EVENTS = Arrays.asList(MotionEvent.ACTION_DOWN,
            MotionEvent.ACTION_POINTER_DOWN);
    private static final List<Integer> RELEASE_EVENTS = Arrays.asList(MotionEvent.ACTION_UP,
            MotionEvent.ACTION_POINTER_UP);
    private final MotionEvent.PointerCoords[] mPointerCoordForReuse
            = new MotionEvent.PointerCoords[1];
    private final int[] mWork = new int[]{0, 0};
    private float getRawX(final MotionEvent event, final int pointerIndex) {
        return event.getRawX() - event.getX() + event.getX(pointerIndex);
    }
    private float getRawY(final MotionEvent event, final int pointerIndex) {
        return event.getRawY() - event.getY() + event.getY(pointerIndex);
    }
    private final Map<Integer, TouchDisplayElement> mPressedView = new HashMap<>();
    @SuppressLint("ClickableViewAccessibility")
    @Override
    public final boolean onTouchEvent(final MotionEvent event) {
        int action = event.getAction() & MotionEvent.ACTION_MASK;
        int pointerIndex = event.getActionIndex();
        int pointerId = event.getPointerId(pointerIndex);
        boolean ret = false;
        for (View v : mViews) {
            if (v instanceof TouchDisplayElement) {
                v.getLocationOnScreen(mWork);
                v.measure(View.MeasureSpec.UNSPECIFIED,
                        View.MeasureSpec.UNSPECIFIED);
                if (RELEASE_EVENTS.contains(action)) {
                    TouchDisplayElement tde = mPressedView.get(pointerId);
                    if (tde != null) {
                        Log.v("pressed", tde + "was released from pointer "
                                + pointerIndex + " from " + event);
                        tde.release();
                    }
                }
                if (PRESS_EVENTS.contains(action)) {
                    ret = handlePress(event, v, pointerIndex, action, pointerId);
                }
                if (action == MotionEvent.ACTION_MOVE) {
                    for (int i = 0; i < event.getPointerCount(); i++) {
                        int id = event.getPointerId(i);
                        handlePress(event, v, i, action, id);
                    }
                }
            }
        }
        if (!ret) {
            handleNonJoystickEvents(event);
        }
        return true;
    }

    private boolean handlePress(final MotionEvent event, final View v, final int pointerIndex,
                                final int action, final int pointerId) {
        boolean ret = false;
        TouchDisplayElement tde = (TouchDisplayElement) v;
        float px = getRawX(event, pointerIndex) - mWork[0];
        float py = getRawY(event, pointerIndex) - mWork[1];
        // CHECKSTYLE DISABLE MagicNumber FOR 3 LINES
        float rx = 2 * ((px - 0.5f) / v.getMeasuredWidth()) - 1;
        float ry = 2 * ((py - 0.5f) / v.getMeasuredHeight()) - 1;
        if (rx > -1 && ry > -1 && rx < 1 && ry < 1) {
            if (action == MotionEvent.ACTION_MOVE) {
                if (mPressedView.get(pointerId) == tde) {
                    tde.moveTo(rx, ry);
                }
            } else {
                tde.press(rx, ry);
                mPressedView.put(pointerId, tde);
                Log.v("pressed", tde
                        + " was pressed by pointer with index " + pointerIndex + "from " + event);
            }
            ret = true;
        }
        return ret;
    }

    interface TouchDisplayElement {
        void press(float x, float y);
        void release();

        void moveTo(float rx, float ry);
    }
    private void handleNonJoystickEvents(final MotionEvent event) {

        int action = event.getAction() & MotionEvent.ACTION_MASK;
        int[] sourceOffset = new int[] {0, 0};
        getLocationOnScreen(sourceOffset);
        float screenX = sourceOffset[0];
        float screenY = sourceOffset[1];

        if (action == MotionEvent.ACTION_DOWN || action == MotionEvent.ACTION_POINTER_DOWN) {
            boolean viewSet = false;
            int pointerIndex = event.getActionIndex();
            int pointerId = event.getPointerId(event.getActionIndex());

            for (View v: mViews) {
                if (!(v instanceof TouchDisplayElement) && containsPosition(v,
                        event.getX(pointerIndex), event.getY(pointerIndex))) {
                    MotionEvent e = calcRelativeEvent(event, MotionEvent.ACTION_DOWN,
                            pointerIndex, v, screenX, screenY);
                    if (!viewSet) {
                        mTouchedView.put(pointerId, v);
                        viewSet = true;
                    }
                    if (v instanceof JoystickElementView) {
                        mJoystickView.put(pointerId, v);
                    }
                    sendEvent(Objects.requireNonNull(v), e);
                }
            }

        } else if (action == MotionEvent.ACTION_POINTER_UP || action == MotionEvent.ACTION_UP) {
            int pointerIndex = event.getActionIndex();
            int pointerId = event.getPointerId(pointerIndex);
            View[] handleViews;
            if (mJoystickView.get(pointerId) == mTouchedView.get(pointerId)) {
                handleViews = new View[] {mJoystickView.get(pointerId)};
            } else {
                handleViews = new View[] {mJoystickView.get(pointerId),
                        mTouchedView.get(pointerId)};
            }
            for (View v: handleViews) {
                if (v != null) {
                    MotionEvent e;
                    if (action == MotionEvent.ACTION_UP) {
                        e = event;
                    } else {
                        e = MotionEvent.obtain(event);
                        e.setAction(MotionEvent.ACTION_UP);
                    }
                    sendEvent(v, e);
                }
            }
            mTouchedView.remove(pointerId);
            mJoystickView.remove(pointerId);
        } else if (action == MotionEvent.ACTION_MOVE) {
            for (View v : mViews) {
                for (int pointerIndex = 0; pointerIndex < event.getPointerCount(); pointerIndex++) {
                    int pointerId = event.getPointerId(pointerIndex);
                    if (v != null && v.getGlobalVisibleRect(mTempRect)
                            && (mTouchedView.get(pointerId) == v && containsPosition(
                                    v, event.getX(pointerIndex), event.getY(pointerIndex)))) {
                        MotionEvent copy = calcRelativeEvent(event, MotionEvent.ACTION_MOVE,
                                pointerIndex, v, screenX, screenY);
                        if (sendEvent(Objects.requireNonNull(v), copy)) {
                            if (v instanceof JoystickElementView) {
                                mJoystickView.put(pointerId, v);
                            }
                            mTouchedView.put(pointerId, v);
                        }
                        break;
                    }

                }
            }
        }
    }

    private MotionEvent calcRelativeEvent(final MotionEvent event, final int action,
                                          final int pointerIndex, final View v,
                                          final float screenX, final float screenY) {
        v.getGlobalVisibleRect(mTempRect);
        event.getPointerProperties(pointerIndex, mPointerPropForReUse[0]);
        float rawOffsetX = event.getRawX() - event.getX();
        float rawOffsetY = event.getRawY() - event.getY();
        mPointerCoordForReuse[0].x = event.getX(pointerIndex) + rawOffsetX
                - (mTempRect.left + screenX);
        mPointerCoordForReuse[0].y = event.getY(pointerIndex) + rawOffsetY
                - (mTempRect.top + screenY);

        return MotionEvent.obtain(event.getDownTime(),
                event.getEventTime(),
                action,
                1,
                mPointerPropForReUse,
                mPointerCoordForReuse,
                event.getMetaState(),
                event.getButtonState(),
                event.getXPrecision(),
                event.getYPrecision(),
                event.getDeviceId(),
                event.getEdgeFlags(),
                event.getSource(),
                event.getFlags());
    }

    private boolean isBlockedTestJoystick(final Joystick joy,
                                          final Set<Class<Joystick>> blockedclasses) {
        Class<?> c = joy.getClass();
        while (c != Joystick.class) {
            if (blockedclasses.contains(c)) {
                return true;
            }
            c = Objects.requireNonNull(c).getSuperclass();
        }
        return false;
    }
    @SuppressLint("ClickableViewAccessibility")
    final void updateAllViews(final EmulationActivity ea,
                              final List<Joystick> joysticks,
                              @NonNull  final Runnable runOnHamburger,
                              final boolean withBottomMargin,
                              final EmulationConfiguration.PreferredJoystickType
                                      defaultJoystickType) {
        boolean showStick = false;
        boolean showWheel = false;
        boolean withMuxer = false;
        Set<Class<Joystick>> blocked = getBlockedTestJoysticks();
        EmulationConfiguration.PreferredJoystickType displaytype;
        if (!ea.isTv()) {
            for (Joystick j : joysticks) {
                if (!isBlockedTestJoystick(j, blocked)) {
                    if (j instanceof MultiplexerJoystick && j.getPortnumber() != PORT_NOT_CONNECTED
                            && j.getCurrentDeviceType() == Emulation.InputDeviceType.DSTICK) {
                        displaytype = ((MultiplexerJoystick) j).getPreferredVirtualType(ea);
                        if (displaytype == EmulationConfiguration.PreferredJoystickType.WHEEL) {
                            showWheel = true;
                        }
                        if (displaytype
                                == EmulationConfiguration.PreferredJoystickType.DIRECTIONAL) {
                            showStick = true;
                        }
                        withMuxer = true;
                    }
                }
            }
            int virtualport = PORT_NOT_CONNECTED;
            if (!withMuxer) {
                for (Joystick j : joysticks) {
                    if (!isBlockedTestJoystick(j, blocked)) {
                        if (j.getPortnumber() != PORT_NOT_CONNECTED) {
                            if (j instanceof TouchJoystick) {
                                virtualport = j.getPortnumber();
                                showStick = true;
                            }
                            if (j instanceof WheelJoystick) {
                                virtualport = j.getPortnumber();
                                showWheel = true;
                            }
                        }
                    }
                }
            }
            for (Joystick joy : joysticks) {
                if (!isBlockedTestJoystick(joy, blocked)) {
                    if (joy.getPortnumber() != PORT_NOT_CONNECTED || withMuxer) {
                        if (!(joy instanceof VirtualJoystick)
                                && !(joy instanceof KeyboardJoystick)
                                && !(joy instanceof MultiplexerJoystick)
                        && joy.getSupportedDeviceTypes()
                                .contains(Emulation.InputDeviceType.DSTICK)) {
                            if (joy.getPortnumber() == virtualport) {
                                showWheel = false;
                                showStick = false;
                            }
                        }
                    }
                }
            }
        }
        View newView = updateJoystickViews(ea, joysticks, showStick, showWheel);
        View hamburger = newView.findViewById(R.id.ib_hamburger_menu);
        if (hamburger != null) {
            hamburger.setVisibility(ea.isInMovieMode() || ea.isTv() ? View.GONE : View.VISIBLE);
            hamburger.setOnTouchListener((view, motionEvent) -> {
                if (motionEvent.getAction() == MotionEvent.ACTION_UP) {
                    runOnHamburger.run();
                    return true;
                }
                return false;
            });
        }
        setBottomMargin(withBottomMargin);
        post(() -> setBottomMargin(withBottomMargin));
    }

    private Set<Class<Joystick>> getBlockedTestJoysticks() {
        Class<DeviceSupportActivity.KeyboardUnitTest> clz;
        try {
            //noinspection unchecked
            clz = (Class<DeviceSupportActivity.KeyboardUnitTest>)
                    Objects.requireNonNull(EmulationActivity.class.getClassLoader())
                            .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("getDisabledJoystickClasses");
            //noinspection unchecked
            return (Set<Class<Joystick>>) method.invoke(clz.newInstance());
        } catch (ClassNotFoundException | NoSuchMethodException e) {
            return new HashSet<>();
        } catch (IllegalAccessException | InvocationTargetException | InstantiationException e) {
            throw new RuntimeException(e);
        }
    }
    final View updateJoystickViews(final EmulationActivity ea, final List<Joystick> joysticks,
                         final boolean showStick, final boolean showWheel) {
        int viewId;
        VirtualJoystick joystick = null;
        TouchJoystick touchJoystick = null;
        WheelJoystick wheelJoystick = null;
        for (Joystick j : joysticks) {
            if (j instanceof TouchJoystick && showStick) {
                joystick = (VirtualJoystick) j;
                touchJoystick = (TouchJoystick) j;
            }
            if (j instanceof WheelJoystick && showWheel) {
                joystick = (VirtualJoystick) j;
                wheelJoystick = (WheelJoystick) j;
            }
        }
        if (touchJoystick != null) {
            if (joystick.isRightHanded(ea)) {
                viewId = R.layout.view_controls_jl;
            } else {
                viewId = R.layout.view_controls_jr;
            }
        } else if (wheelJoystick != null) {
            if (joystick.isRightHanded(ea)) {
                viewId = R.layout.view_controls_wl;
            } else {
                viewId = R.layout.view_controls_wr;
            }

        } else {
            viewId = R.layout.view_controls;
        }
        boolean newViewRequired;
        View oldView = findViewById(R.id.fragment_dynamic_ui_elements);
        if (oldView != null && oldView.getTag() != Integer.valueOf(viewId)) {
            removeView(oldView);
            newViewRequired = true;
        } else {
            newViewRequired = oldView == null;
        }
        View newView;
        if (newViewRequired) {
            newView = ea.getLayoutInflater()
                    .inflate(viewId, this).findViewById(R.id.fragment_dynamic_ui_elements);

            if (newView != null) {
                newView.setTag(viewId);
                int intVal = ea.getCurrentUseropts().getIntegerValue(
                        ea.getString(R.string.key_joystick_opacity),
                        getResources().getInteger(R.integer.value_joystick_transparency));
                float alpha = MIN_OPACITY + ((float) intVal / PERCENT_DIVIDER);
                JoystickFireView jfw = newView.findViewById(R.id.jv_fire);
                JoystickStickView jsv = newView.findViewById(R.id.jv_directions);
                JoystickSecondaryView j2nd = newView.findViewById(R.id.jv_secondary);
                JoystickGearshiftView jgv = newView.findViewById(R.id.jv_flipswitch);
                JoystickWheelView jwv = newView.findViewById(R.id.jv_wheel);
                if (ea.isInMovieMode()) {
                    for (View v : Arrays.asList(jfw, jsv, j2nd, jgv, jwv)) {
                        if (v != null) {
                            v.setVisibility(View.GONE);
                        }
                    }
                } else {
                if (jfw != null && touchJoystick != null) {
                        jfw.setVisibility(View.VISIBLE);
                    jfw.setAlpha(alpha);
                    jfw.setJoystick(touchJoystick);
                    touchJoystick.setFireView(jfw);
                }
                if (jsv != null && touchJoystick != null) {
                        jsv.setVisibility(View.VISIBLE);
                    jsv.setJoystick(touchJoystick);
                    jsv.setAlpha(alpha);
                    touchJoystick.setDirectionView(jsv);
                }

                if (j2nd != null) {
                    j2nd.setVisibility(touchJoystick != null ? View.VISIBLE : View.GONE);
                    if (touchJoystick != null) {
                        j2nd.setAlpha(alpha);
                        j2nd.setJoystick(touchJoystick);
                    }
                }

                if (jgv != null && wheelJoystick != null) {
                        jgv.setVisibility(View.VISIBLE);
                    jgv.setJoystick(wheelJoystick);
                    jgv.setAlpha(alpha);
                    wheelJoystick.setGearshiftView(jgv, wheelJoystick.isRightHanded(ea));
                }
                if (jwv != null && wheelJoystick != null) {
                        jwv.setVisibility(View.VISIBLE);
                    jwv.setJoystick(wheelJoystick);
                    jwv.setAlpha(alpha);
                    wheelJoystick.setWheelview(jwv);
                    }
                }
                newView.addOnLayoutChangeListener((view, i, i1, i2, i3, i4, i5, i6, i7) -> {
                    mViews.clear();
                    for (int id: VIEW_IDS) {
                        View v = findViewById(id);
                        if (v != null  && v.getVisibility() == View.VISIBLE) {
                            mViews.add(v);
                        }
                    }
                });

            }
        } else {
             newView = oldView;
        }
        return newView;
    }

    final void setBottomMargin(final boolean withBottomMargin) {
        View root = findViewById(R.id.fragment_dynamic_ui_elements);
        if (root != null) {
            View spacer = root.findViewById(R.id.joystick_spacer);
            if (spacer != null) {
                ViewGroup.LayoutParams lp = spacer.getLayoutParams();
                lp.height = getResources().getDimensionPixelOffset(
                        withBottomMargin ? R.dimen.joystick_margin_bottom : R.dimen.minimal);
                spacer.setLayoutParams(lp);
            }
        }
    }
}
