package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.SeekBar;
import android.widget.Spinner;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.SwitchCompat;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.KeyException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

public class ShaderSettingsView extends RelativeLayout {
    private Bitmap mTestBitmap = null;

    /**
     * Constructor that is called when inflating a ShaderSettingsView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */

    public ShaderSettingsView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyleAttr An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public ShaderSettingsView(final Context context, final AttributeSet attrs,
                              final int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }
    private void init() {
        InputStream is = getContext().getResources().openRawResource(R.raw.screenshot);
        mTestBitmap =  BitmapFactory.decodeStream(is);
        try {
            is.close();
        } catch (IOException e) {
            if (BuildConfig.DEBUG) {
                throw new RuntimeException(e);
            }
        }
        LayoutInflater inflater = (LayoutInflater) getContext()
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        inflater.inflate(R.layout.view_glshaders_settings, this, true);
        addOnLayoutChangeListener((v, left, top, right,
                                   bottom, oldLeft, oldTop, oldRight, oldBottom) -> {
            if (left != oldLeft || right != oldRight || top != oldTop || bottom != oldBottom) {
                View parameters = findViewById(R.id.ll_parameters);
                int height = getRootView().findViewById(android.R.id.content).getHeight();
                for (int id : Arrays.asList(R.id.tv_parameters, R.id.cb_use_global_settings,
                        R.id.ll_parameters)) {
                    height -= findViewById(id).getHeight();
                }
                if (height < 2 * mTestBitmap.getHeight()) {
                    height = 2 * mTestBitmap.getHeight();
                }
                int width = height * mTestBitmap.getWidth() / mTestBitmap.getHeight();
                if (width > parameters.getWidth()) {
                    width = getWidth();
                    height = width * mTestBitmap.getHeight() / mTestBitmap.getWidth();
                }
                findViewById(R.id.gv_preview).setLayoutParams(
                        new LinearLayout.LayoutParams(width, height));
            }
        });

    }
    static class GlShader implements EmulationUi.SettingSpinnerElement {
        private final String mId;
        private final String mName;
        private final List<MonitorGlSurfaceView.OpenGlParameter> mParameters;
        private final String mAssetName;

        GlShader(final String id, final String name, final String assetName,
                        final List<MonitorGlSurfaceView.OpenGlParameter> parameters) {
            mId = id;
            mName = name;
            mParameters = parameters;
            mAssetName = assetName;
        }

        String getAssetName() {
            return mAssetName;
        }

        @SuppressLint("DiscouragedApi")
        static GlShader create(final Context ctx, final String assetName) throws IOException {
            InputStream is = ctx.getAssets().open("shaders/" + assetName);
            String name = null;
            String id = null;
            List<MonitorGlSurfaceView.OpenGlParameter> parameters = new LinkedList<>();
            byte[] data = new byte[is.available()];
            if (is.read(data) > 0) {
                String s = new String(data, StandardCharsets.UTF_8);
                is.close();
                for (String line : s.split("\r?\n")) {
                    if (line.startsWith("#pragma")) {
                        String[] parts = line.split(" ");
                        if (parts.length == 3 && "id".equals(parts[1])) {
                            int resId = ctx.getResources().getIdentifier(parts[2], "string",
                                    ctx.getPackageName());
                            if (resId > 0) {
                                id = parts[2];
                                name = ctx.getResources().getString(resId);
                            }
                        } else if (line.startsWith("#pragma parameter")) {
                            parameters.add(new MonitorGlSurfaceView.OpenGlParameter(line));
                        }
                    }
                }
                if (name != null && id != null) {
                    return new GlShader(id, name, assetName, parameters);
                }
            }
            return null;
        }

        @Override
        public String getId() {
            return mId;
        }

        @Override
        public String getText() {
            return mName;
        }

        @NonNull
        @Override
        public String toString() {
            return mName != null ? mName : super.toString();
        }

        List<MonitorGlSurfaceView.OpenGlParameter> getParameterList() {
            return mParameters;
        }
    }
    private final Map<String, Integer> mUseroptVieIds = new HashMap<>();
    private void updateShaderSpecificSetting(final SettingsController sc,  final GlShader shader) {
        LayoutInflater inflater = (LayoutInflater) getContext()
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        findViewById(R.id.filter_options).setVisibility(
                shader.getParameterList().isEmpty() ? View.GONE : View.VISIBLE);

        LinearLayout llParameters = findViewById(R.id.ll_parameters);
        LinearLayout llActions = findViewById(R.id.ll_actions);
        CheckBox cb = findViewById(R.id.cb_use_global_settings);
        final String useGlobalSettingsKey = "use_system_settings_" + shader.getId();
        cb.setChecked(sc.getCurrentValue(useGlobalSettingsKey, cb.isChecked()));
        final Map<String, SeekBar> seekbars = new HashMap<>();
        final Map<String, MonitorGlSurfaceView.OpenGlParameter> parameters = new HashMap<>();
        llParameters.removeAllViews();

        final Button bnReadFromGlobals = llActions.findViewById(R.id.bn_read_from_globals);
        final Button bnWriteToGlobals = llActions.findViewById(R.id.bn_write_to_globals);
        final Button bnReset = llActions.findViewById(R.id.bn_reset);
        final SwitchCompat swPreview = findViewById(R.id.sw_preview);
        final MonitorGlSurfaceView gvPreview = findViewById(R.id.gv_preview);
        swPreview.setChecked(sc.getCurrentValue("gl_show_preview", swPreview.isChecked()));
        swPreview.setOnCheckedChangeListener((buttonView, isChecked) -> {
            sc.storeValue(SettingsActivity.Scope.GLOBAL, "gl_show_preview", isChecked);
            gvPreview.setVisibility(isChecked ? View.VISIBLE : View.GONE);
        });

        final MonitorGlSurfaceView.SettingParameterList glParameters
                = new MonitorGlSurfaceView.SettingParameterList();
        gvPreview.setSettingsGlProgram(shader.getAssetName());
        for (MonitorGlSurfaceView.OpenGlParameter p : shader.getParameterList()) {
            @SuppressLint("DiscouragedApi") int resId = getContext().getResources()
                    .getIdentifier("gl_" + p.getId().toLowerCase(Locale.getDefault()), "string",
                    getContext().getPackageName());
            if (resId > 0) {
                if (!mUseroptVieIds.containsKey(shader.getId())) {
                    mUseroptVieIds.put(shader.getId(), View.generateViewId());
                }
                final String useroptId = shader.getId() + "_" + p.getId();
                View group = inflater.inflate(R.layout.view_glshader_parameter,
                        llParameters, true).findViewById(R.id.glshader_parameter);
                group.setId(View.NO_ID);
                ((TextView) group.findViewById(R.id.tv_label)).setText(getContext()
                        .getResources().getString(resId));
                SeekBar sb = group.findViewById(R.id.sp_value);
                seekbars.put(useroptId, sb);
                parameters.put(useroptId, p);
                sb.setMax(MonitorGlSurfaceView.OpenGlParameter.MAX_INT_REPR_VALUE);
                int std = p.intRepresentation(p.getDefault());
                int globalValue = sc.getCurrentValue("global_" + useroptId, std);
                int emuValue = sc.getCurrentValue(useroptId, std);
                int value = cb.isChecked() ? globalValue : emuValue;
                sb.setProgress(value);
                glParameters.put(p.getId(), p.floatRepresentation(value));
                if (cb.isChecked()) {
                    sb.setEnabled(false);
                } else {
                    if (globalValue != emuValue) {
                        bnReadFromGlobals.setEnabled(true);
                        bnWriteToGlobals.setEnabled(true);
                    }
                    if (p.intRepresentation(p.getDefault()) != emuValue) {
                        bnReset.setEnabled(true);
                    }
                }
                sb.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
                    @Override
                    public void onProgressChanged(final SeekBar seekBar, final int progress,
                                                  final boolean fromUser) {
                        if (!cb.isChecked()) {
                            sc.storeValue(SettingsActivity.Scope.UNIQUE, useroptId, progress);
                            if (progress != emuValue) {
                                bnReadFromGlobals.setEnabled(true);
                                bnWriteToGlobals.setEnabled(true);
                            }
                            if (progress != p.intRepresentation(p.getDefault())) {
                                bnReset.setEnabled(true);
                            }
                        }
                        if (fromUser) {
                            gvPreview.setSettingsGlParameter(
                                    p.getId(), p.floatRepresentation(progress));
                        }
                    }

                    @Override
                    public void onStartTrackingTouch(final SeekBar seekBar) {

                    }

                    @Override
                    public void onStopTrackingTouch(final SeekBar seekBar) {

                    }
                });
            }
        }
        gvPreview.useForSettings(shader, mTestBitmap);
        gvPreview.setSettingsGlParameters(glParameters);
        gvPreview.setVisibility(swPreview.isChecked() ? View.VISIBLE : View.GONE);
        gvPreview.requestRender();
        cb.setOnCheckedChangeListener((button, isChecked) -> {
            sc.storeValue(SettingsActivity.Scope.UNIQUE, useGlobalSettingsKey, isChecked);
            bnReadFromGlobals.setEnabled(false);
            bnWriteToGlobals.setEnabled(false);
            bnReset.setEnabled(false);
            MonitorGlSurfaceView.SettingParameterList params
                    = new MonitorGlSurfaceView.SettingParameterList();
            for (String useroptId : seekbars.keySet()) {
                MonitorGlSurfaceView.OpenGlParameter p = parameters.get(useroptId);
                SeekBar sb = seekbars.get(useroptId);
                if (sb != null && p != null) {
                    String optid;
                    if (isChecked) {
                        optid = "global_" + useroptId;
                    } else {
                        optid = useroptId;
                    }
                    int defaultValue = p.intRepresentation(p.getDefault());
                    sb.setProgress(sc.getCurrentValue(optid, defaultValue));
                    params.put(p.getId(), p.floatRepresentation(sb.getProgress()));
                    sb.setEnabled(!isChecked);
                    updateButtons(sc, isChecked, optid, defaultValue,
                            bnReadFromGlobals, bnWriteToGlobals, bnReset);
                } else {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(new KeyException());
                    }
                }

            }
            gvPreview.setSettingsGlParameters(params);
            gvPreview.requestRender();
        });
        bnReadFromGlobals.setOnClickListener(v
                -> updateValues(gvPreview, sc, shader, bnReadFromGlobals, bnWriteToGlobals, bnReset,
                        seekbars, cb, (sc2, key, defaultVal)
                        -> sc2.storeValue(SettingsActivity.Scope.UNIQUE, key,
                        sc2.getCurrentValue("global_" + key, defaultVal))));
        bnWriteToGlobals.setOnClickListener(v
                -> updateValues(gvPreview, sc, shader, bnReadFromGlobals, bnWriteToGlobals, bnReset,
                        seekbars, cb, (sc2, key, defaultVal)
                        -> sc2.storeValue(SettingsActivity.Scope.GLOBAL, "global_" + key,
                        sc2.getCurrentValue(key, defaultVal))));

        bnReset.setOnClickListener(v -> updateValues(gvPreview, sc, shader, bnReadFromGlobals,
                bnWriteToGlobals, bnReset, seekbars, cb,
                (sc2, key, defaultVal)
                        -> sc2.storeValue(SettingsActivity.Scope.UNIQUE, key, defaultVal)));

    }
    private interface UseroptsCopier {
        void copy(SettingsController sc, String key, int defaultVal);
    }
    // CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
    private static void updateValues(final MonitorGlSurfaceView glView,
                                     final SettingsController sc, final GlShader shader,
                                     final Button bnReadFromGlobals, final Button bnWriteToGlobals,
                                     final Button bnReset, final Map<String, SeekBar> seekbars,
                                     final CheckBox cb, final UseroptsCopier copier) {
        bnReadFromGlobals.setEnabled(false);
        bnWriteToGlobals.setEnabled(false);
        MonitorGlSurfaceView.SettingParameterList params
                = new MonitorGlSurfaceView.SettingParameterList();
        bnReset.setEnabled(false);
        for (MonitorGlSurfaceView.OpenGlParameter p : shader.getParameterList()) {
            String key = shader.getId() + "_" + p.getId();
            if (seekbars.containsKey(key)) {
                SeekBar sb = seekbars.get(key);
                if (sb != null) {
                    int defaultValue = p.intRepresentation(p.getDefault());
                    boolean globalValues = cb.isChecked();
                    copier.copy(sc, key, defaultValue);
                    updateButtons(sc, globalValues, key, defaultValue,
                            bnReadFromGlobals, bnWriteToGlobals, bnReset);

                    sb.setProgress(sc.getCurrentValue(key, defaultValue));
                    params.put(p.getId(), p.floatRepresentation(sb.getProgress()));
                } else {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(new KeyException());
                    }
                }
            }
        }
        glView.setSettingsGlParameters(params);
        glView.requestRender();
    }

    private static void updateButtons(final SettingsController sc, final boolean globalValues,
                                      final String key, final int defaultValue,
                                      final Button bnReadFromGlobals,
                                      final Button bnWriteToGlobals,
                                      final Button bnReset) {
        boolean valuesDiffer;
        boolean defaultDiffers;
        valuesDiffer = !globalValues && sc.getCurrentValue(key, defaultValue)
                != sc.getCurrentValue("global_" + key, defaultValue);
        defaultDiffers = !globalValues && sc.getCurrentValue(key, defaultValue)
                != defaultValue;
        if (!globalValues) {
            if (valuesDiffer) {
                bnReadFromGlobals.setEnabled(true);
                bnWriteToGlobals.setEnabled(true);
            }
            if (defaultDiffers) {
                bnReset.setEnabled(true);
            }
        }
    }

    final void populate(final SettingsController sc) {
        final Spinner sp = findViewById(R.id.sp_filter);
        String optsVal =  sc.getCurrentValue(sp.getTag().toString(), null);
        new Thread(() -> {
            ArrayAdapter<GlShader> elements
                    = new ArrayAdapter<>(getContext(), android.R.layout.simple_spinner_item);
            try {
                int selection = Spinner.NO_ID;
                for (String file : Objects.requireNonNull(
                        getContext().getAssets().list("shaders"))) {
                    if (file.matches("[0-9]{2}_fragment-.*\\.glsl")) {
                        GlShader s = GlShader.create(getContext(), file);
                        if (s != null) {
                            if (selection == View.NO_ID
                                    && (optsVal == null || s.getId().equals(optsVal))) {
                                selection = elements.getCount();
                            }
                            elements.add(s);
                        }
                    }
                }
                final int finalSelection = selection;
                post(() -> {
                    elements.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
                    sp.setAdapter(elements);
                    sp.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
                        @Override
                        public void onItemSelected(final AdapterView<?> parent, final View view,
                                                   final int position, final long id) {
                            int index = sp.getSelectedItemPosition();
                            if (index >= 0) {
                                GlShader shader = ((GlShader) sp.getItemAtPosition(index));
                                sc.storeValue(getId(), sp.getTag().toString(), shader.getId());
                                updateShaderSpecificSetting(sc, shader);
                            }
                        }

                        @Override
                        public void onNothingSelected(final AdapterView<?> parent) {

                        }
                    });
                    sp.setSelection(finalSelection);
                });

            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).start();
    }
}
