//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.util.Log;
import android.view.View;
import android.widget.AdapterView;
import android.widget.CheckBox;
import android.widget.RadioGroup;
import android.widget.SeekBar;
import android.widget.Spinner;

import androidx.annotation.NonNull;
import androidx.fragment.app.Fragment;

import de.rainerhock.eightbitwonders.SettingsActivity.ResourceDescription;
import de.rainerhock.eightbitwonders.SettingsActivity.Scope;
import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

class SettingsController implements Serializable {
    private final SettingsController mParentController;
    private final Map<Integer, List<Joystick>> mCurrentJoystick = new HashMap<>();

    SettingsController(final SettingsController source) {
        super();
        mCurrentJoystick.putAll(source.mCurrentJoystick);
        mStoredStrings.putAll(source.mStoredStrings);
        mStoredBooleans.putAll(source.mStoredBooleans);
        mStoredIntegers.putAll(source.mStoredIntegers);
        mUseropts = source.mUseropts;
        mActivity = source.mActivity;
        mParentController = source;
    }
    SettingsController() {
        super();
        mParentController = null;
    }
    private static final long serialVersionUID = -3582824211639573117L;
    private static final String TAG = SettingsController.class.getSimpleName();
    private Map<Integer, SettingsActivity.SettingsViewModifier> mModifiers;
    void onJoystickportChanged(final @NonNull  Joystick joy, final int port) {
        for (int p2 : mCurrentJoystick.keySet()) {
            List<Joystick> sticks = mCurrentJoystick.get(p2);
            if (sticks != null) {
                sticks.remove(joy);
            }
        }
        if (mCurrentJoystick.get(port) == null) {
            mCurrentJoystick.put(port, new LinkedList<>());
        }
        Objects.requireNonNull(mCurrentJoystick.get(port)).add(joy);

    }
    public void removeJoystick(final int port) {
        mCurrentJoystick.remove(port);
    }

    int getCurrentJoystickport(final @NonNull  Joystick joy) {
        for (int port : mCurrentJoystick.keySet()) {
            List<Joystick> sticks = mCurrentJoystick.get(port);
            if (sticks != null && sticks.contains(joy)) {
                return port;
            }
        }
        return getCurrentValue("JOYSTICK_" + joy.getId() + "_PORT",
                Joystick.PORT_NOT_CONNECTED);
    }
    private void onValueChanged(final String key, final String val) {
        for (int id : mViews.keySet()) {
            ResourceDescription rd = mActivity.getResourceDescription(id);
            String idKey = rd.getDependsOnKey();
            String idVal = rd.getDependsOnValue();
            if (idKey != null && idKey.equals(key) && idVal != null && val != null) {
                Objects.requireNonNull(mViews.get(id)).setVisibility(
                        idVal.equals(val) ? View.VISIBLE : View.GONE);
            }
        }
        for (Fragment f : mActivity.getSupportFragmentManager().getFragments()) {
            if (f instanceof SettingsFragment) {
                SettingsFragment sf = (SettingsFragment) f;
                sf.handleOption(key, val);
            }
        }
    }

    protected final Scope getScope(final int id) {
        return mActivity.getResourceDescription(id).getSettingsScope();
    }
    void updateJoystickMapping(final int optid, final String optKey, final String optModificator,
                               final String newKey, final String newModificator) {
        mValueStorers.put(-optid, () -> getUseropts().setValue(Useropts.Scope.CONFIGURATION,
                optKey, newKey));
        mValueStorers.put(optid, () -> getUseropts().setValue(Useropts.Scope.CONFIGURATION,
                optModificator, newModificator));

    }
    void storeValue(final int id, final String key, final boolean value) {
        mStoredBooleans.put(key, value);
        mValueStorers.put(id, () -> {
            switch (getScope(id)) {
                case GLOBAL:
                    getUseropts().setValue(Useropts.Scope.GLOBAL, key, value);
                    break;
                case SHARED:
                    getUseropts().setValue(Useropts.Scope.EMULATOR, key, value);
                    break;
                case UNIQUE:
                    getUseropts().setValue(Useropts.Scope.CONFIGURATION, key, value);
                    break;
                default:
                    break;
            }
        });
    }
    private final Map<String, Boolean> mStoredBooleans = new HashMap<>();

    boolean getCurrentValue(final String key, final boolean defaulvalue) {
        if (mStoredBooleans.containsKey(key) && mStoredBooleans.get(key) != null) {
            Boolean b = mStoredBooleans.get(key);
            if (b != null) {
                return b;
            }
        }
        return getUseropts().getBooleanValue(key, defaulvalue);
    }

    void storeValue(final int id, final String key, final int value) {
        mStoredIntegers.put(key, value);
        mValueStorers.put(id, () -> {
            switch (getScope(id)) {
                case GLOBAL:
                    getUseropts().setValue(Useropts.Scope.GLOBAL, key, value);
                    break;
                case SHARED:
                    getUseropts().setValue(Useropts.Scope.EMULATOR, key, value);
                    break;
                case UNIQUE:
                    getUseropts().setValue(Useropts.Scope.CONFIGURATION, key, value);
                    break;
                default:
                    break;
            }
        });
    }
    private final Map<String, Integer> mStoredIntegers = new HashMap<>();

    private final Map<Object, Runnable> mValueStorers = new LinkedHashMap<>();
    void storeValue(final Scope scope, final String key, final String value) {
        mStoredStrings.put(key, value);
        mValueStorers.put(key, () ->
                getUseropts().setValue(SCOPEMAPING.get(scope), key, value));
    }
    void storeValue(final Scope scope, final String key, final boolean value) {
        mStoredBooleans.put(key, value);
        mValueStorers.put(key, () ->
                getUseropts().setValue(SCOPEMAPING.get(scope), key, value));
    }
    void storeValue(final Scope scope, final String key, final int value) {
        mStoredIntegers.put(key, value);
        mValueStorers.put(key, () ->
                getUseropts().setValue(SCOPEMAPING.get(scope), key, value));
    }
    private static final Map<SettingsActivity.Scope, Useropts.Scope> SCOPEMAPING
            = new HashMap<SettingsActivity.Scope, Useropts.Scope>() {{
                put(SettingsActivity.Scope.GLOBAL, Useropts.Scope.GLOBAL);
                put(SettingsActivity.Scope.SHARED, Useropts.Scope.EMULATOR);
                put(SettingsActivity.Scope.UNIQUE, Useropts.Scope.CONFIGURATION);
    }};
    void storeValue(final int id, final String key, final String value) {
        mStoredStrings.put(key, value);
        mValueStorers.put(id, () ->
                getUseropts().setValue(SCOPEMAPING.get(getScope(id)), key, value));
    }
    private final Map<String, String> mStoredStrings = new HashMap<>();
    String getCurrentValue(final String key, final String defaulvalue) {
        if (mStoredStrings.containsKey(key)) {
            return mStoredStrings.get(key);
        }
        return getUseropts().getStringValue(key, defaulvalue);
    }
    int getCurrentValue(final String key, final int defaulvalue) {
        if (mStoredIntegers.containsKey(key)) {
            //noinspection DataFlowIssue
            return mStoredIntegers.get(key);
        }
        return getUseropts().getIntegerValue(key, defaulvalue);
    }

    void apply() {
        for (Joystick joy : mActivity.getAvailableJoysticks()) {
            int newPort = Joystick.PORT_NOT_CONNECTED;
            for (int port : mCurrentJoystick.keySet()) {
                List<Joystick> sticks = mCurrentJoystick.get(port);
                if (sticks != null && sticks.contains(joy)) {
                    newPort = port;
                }
            }
            final int n = newPort;
            mValueStorers.put(-newPort, () -> getUseropts().setValue(Useropts.Scope.CONFIGURATION,
                    "JOYSTICK_" + joy.getId() + "_PORT", n));
        }
        if (mParentController != null) {
            mParentController.mCurrentJoystick.putAll(mCurrentJoystick);
            for (Object id: mValueStorers.keySet()) {
                mParentController.mValueStorers.remove(id);
                mParentController.mValueStorers.put(id, mValueStorers.get(id));
            }
            for (String key: mStoredBooleans.keySet()) {
                mParentController.mStoredBooleans.remove(key);
                mParentController.mStoredBooleans.put(key, mStoredBooleans.get(key));
            }
            for (String key: mStoredIntegers.keySet()) {
                mParentController.mStoredIntegers.remove(key);
                mParentController.mStoredIntegers.put(key, mStoredIntegers.get(key));
            }
            for (String key: mStoredStrings.keySet()) {
                mParentController.mStoredStrings.remove(key);
                mParentController.mStoredStrings.put(key, mStoredStrings.get(key));
            }

        } else {
            for (int port: mCurrentJoystick.keySet()) {
                List<Joystick> sticks = mCurrentJoystick.get(port);
                if (sticks != null) {
                    for (Joystick joy : sticks) {
                        getUseropts().setValue(Useropts.Scope.CONFIGURATION,
                                "JOYSTICK_" + joy.getId() + "_PORT", port);
                    }
                }
            }
            for (Runnable r : mValueStorers.values()) {
                r.run();
            }
            mValueStorers.clear();
            mStoredBooleans.clear();
            mStoredStrings.clear();
            mStoredIntegers.clear();
        }
    }
    private Useropts mUseropts;

    protected final Useropts getUseropts() {
        return mUseropts;
    }

    private final Map<Integer, View> mViews = new LinkedHashMap<>();
    private SettingsActivity mActivity = null;
    SettingsActivity getActivity() {
        return mActivity;
    }
    final void setActivity(final SettingsActivity a) {
        mActivity = a;
        mUseropts = a.getCurrentUseropts();
    }

    public void setModifiers(final Map<Integer,
            SettingsActivity.SettingsViewModifier> modifier) {
        mModifiers = modifier;
    }

    final void addWidget(final int id, final View v) {
        mViews.put(id, v);
    }


    final void initWidget(final View v) {
        if (v instanceof Spinner && v.getTag() != null) {
            Spinner s = (Spinner) v;
            String defaultval = null;
            if (v instanceof MappedSpinner) {
                defaultval = ((MappedSpinner) s).getDefaultElementId();
            }
            String activeElementId = getCurrentValue((String) s.getTag(), defaultval);
            if (activeElementId != null) {
                if (v instanceof MappedSpinner) {
                    ((MappedSpinner) s).setSelection(activeElementId);
                } else {
                    int element = View.NO_ID;
                    for (int i = 0; i < s.getCount(); i++) {
                        if (((EmulationUi.SettingSpinnerElement) s.getItemAtPosition(i)).getId()
                                .equals(activeElementId)) {
                            element = i;
                        }
                    }
                    s.setSelection(element);
                }
            }
            if (s instanceof MappedSpinner) {
                if (s.getOnItemSelectedListener() == null) {
                    s.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
                        @Override
                        public void onItemSelected(final AdapterView<?> adapterView,
                                                   final View view,
                                                   final int i,
                                                   final long l) {
                            String value = ((MappedSpinner) s).getSelection();
                            onValueChanged((String) s.getTag(), value);
                            storeValue(s.getId(), (String) s.getTag(), value);
                        }

                        @Override
                        public void onNothingSelected(final AdapterView<?> adapterView) {

                        }
                    });
                }
            } else {
                s.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
                    @Override
                    public void onItemSelected(final AdapterView<?> parent,
                                               final View view,
                                               final int position,
                                               final long id) {
                        String value = ((EmulationUi.SettingSpinnerElement)
                                s.getItemAtPosition(position)).getId();
                        onValueChanged((String) s.getTag(), value);
                        storeValue(s.getId(), (String) s.getTag(), value);
                    }

                    @Override
                    public void onNothingSelected(final AdapterView<?> parent) {

                    }
                });
            }


        }
        if (v instanceof RadioGroup  && v.getTag() != null) {
            RadioGroup rg = (RadioGroup) v;
            rg.setOnCheckedChangeListener((group, checkedId) -> {
                onValueChanged((String) group.getTag(),
                        (String) group.findViewById(checkedId).getTag());
                storeValue(group.getId(), (String) group.getTag(),
                        (String) group.findViewById(checkedId).getTag());
            });
            String activeTag = getCurrentValue((String) rg.getTag(), null);
            if (activeTag != null) {
                for (int i = 0; i < rg.getChildCount(); i++) {
                    if (activeTag.equals(rg.getChildAt(i).getTag())) {
                        rg.check(rg.getChildAt(i).getId());
                    }
                }
            }
        }
        if (v instanceof CheckBox  && v.getTag() != null) {
            final CheckBox cb = (CheckBox) v;
            final String trueval =
                    cb instanceof MappedCheckBox ? ((MappedCheckBox) cb).getMappingValue(true)
                            : "true";
            final String falseval =
                    cb instanceof MappedCheckBox ? ((MappedCheckBox) cb).getMappingValue(false)
                            : "false";
            if (cb instanceof MappedCheckBox) {
                cb.setOnCheckedChangeListener((buttonView, isChecked) -> {

                    onValueChanged((String) cb.getTag(),
                            cb.isChecked() ? trueval : falseval);
                    storeValue(cb.getId(), (String) cb.getTag(),
                            cb.isChecked() ? trueval : falseval);
                    ((MappedCheckBox) cb).updateDependantVisibility();

                });
                String useroptval = getCurrentValue((String) cb.getTag(),
                        cb.isChecked() ? trueval : falseval);
                cb.setChecked(useroptval.equals(trueval));

            } else {
                cb.setOnCheckedChangeListener((buttonView, isChecked) -> {

                    onValueChanged((String) cb.getTag(), cb.isChecked() ? trueval : falseval);
                    storeValue(cb.getId(), (String) cb.getTag(), cb.isChecked());
                });
                cb.setChecked(getCurrentValue((String) cb.getTag(), cb.isChecked()));
            }
        }

        if (v instanceof SeekBar && v.getTag() != null) {
            Log.v(TAG, "entering SeekBar");
            final SeekBar sb = (SeekBar) v;
            sb.setProgress(getCurrentValue(sb.getTag().toString(), sb.getProgress()));
            sb.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
                @Override
                public void onProgressChanged(final SeekBar seekBar, final int i, final boolean b) {
                    onValueChanged((String) sb.getTag(), String.valueOf(sb.getProgress()));
                    storeValue(sb.getId(), (String) sb.getTag(), sb.getProgress());
                }

                @Override
                public void onStartTrackingTouch(final SeekBar seekBar) {

                }

                @Override
                public void onStopTrackingTouch(final SeekBar seekBar) {

                }
            });
            Log.v(TAG, "entering SeekBar");
        }
        if (v instanceof ShaderSettingsView) {
            ((ShaderSettingsView) v).populate(this);
        }
        if (v.getId() != View.NO_ID) {
            if (mModifiers.containsKey(v.getId())) {
                SettingsActivity.SettingsViewModifier modifier = mModifiers.get(v.getId());
                mModifiers.remove(v.getId());
                if (modifier != null) {
                    modifier.modify(v);
                }
            }
        }
    }
}
