//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import de.rainerhock.eightbitwonders.Emulation.AdditionalKey;
import de.rainerhock.eightbitwonders.vice.AndroidCompatibilityUtils;

import static de.rainerhock.eightbitwonders.R.styleable.useropts_scope;

import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.res.TypedArray;
import android.os.Bundle;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.Spinner;
import android.widget.TableLayout;
import android.widget.TableRow;

import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.contract.ActivityResultContract;
import androidx.annotation.IdRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.SwitchCompat;
import androidx.fragment.app.Fragment;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;

public final class SettingsActivity extends RemoteControlActivity {
    static final String CONFIGURATION = "CONFIGURATION";
    static final String OPENSECTIONS = "OPENSECTIONS";
    static final String ALLOWKEYBOARD = "ALLOWKEYBOARD";
    private static final String CURRENT_STATE = "CURRENT_STATE";
    private static final String MODEL_IS_FIXED = "MODEL_IS_FIXED";

    void onSoftkeyChanged(final SoftkeyConfigView.Softkey key) {
        SoftkeyConfigView v = findViewById(R.id.softkeyconfig);
        v.onSoftkeyChanged(key);
    }
    void onSoftkeyButtonsChanged(final SoftkeyConfigView.Softkey key, final NeutralButton oldbutton,
                                 final String fragmentTag,
                                 final NeutralButton newbutton) {
        Fragment f = getSupportFragmentManager().findFragmentByTag(fragmentTag);
        if (f instanceof SoftkeyConfigView.SoftkeyModificatorFragment) {
            key.getGamepadButtons().remove(oldbutton);
            if (newbutton != null) {
                key.getGamepadButtons().add(newbutton);
            }

            ((SoftkeyConfigView.SoftkeyModificatorFragment) f)
                    .updateButtons(f.getLayoutInflater(), (ViewGroup) f.requireView(), key);
        }
    }
    static class CurrentEmulationCallbacks implements Serializable,
            Emulation.JoystickFunctions,
            Emulation.HardwareKeyboardFunctions,
            Emulation.MachineSettingsFunctions {
        private final Map<Integer, String> mJoystickPorts = new HashMap<>();
        private final List<Integer> mUsedJoystickPorts;
        private final LinkedList<Integer> mExtendedJoysticks = new LinkedList<>();
        private final LinkedList<Integer> mSimulatingKeystrokes = new LinkedList<>();
        private final Map<Integer, List<Emulation.InputDeviceType>> mAvailableDevicetypes
                = new HashMap<>();
        private final List<KeyboardMapping> mKeyboardMapping = new LinkedList<>();
        private final Map<Integer, Integer>  mAutofireFrequency = new HashMap<>();
        private static final Map<String, SettingsFragment> ALL_SETTING_FRAGMENTS
                = new HashMap<>();
        private final Map<Boolean, List<Integer>> mLayoutIds = new HashMap<>();
        private final Map<Integer, List<AdditionalKey>> mMappedJoystickKeys = new HashMap<>();
        private final Map<Integer, List<AdditionalKey>> mMappedJoystickModificatorKeys
                = new HashMap<>();
        private final String mId;
        private final LinkedList<AdditionalKey> mSoftkeys = new LinkedList<>();
        private final LinkedList<AdditionalKey> mModificatorSoftkeys = new LinkedList<>();
        private final boolean mSoftwarekeyboardVariants;
        private final boolean mIsBareMachine;
        CurrentEmulationCallbacks(final Emulation source, final EmulationConfiguration conf,
                                  final EmulationViewModel viewmodel) {
            mId = UUID.randomUUID().toString();
            Emulation.JoystickFunctions jf = source.getJoystickFunctions();
            if (jf != null) {
                Map<Integer, String> ports = jf.getJoystickports();
                if (ports != null) {
                    mJoystickPorts.putAll(ports);
                    for (int i : mJoystickPorts.keySet()) {
                    //for (int i = 0; i < mJoystickPorts.size(); i++) {

                        if (jf.isExtentedJoystick(i)) {
                            mExtendedJoysticks.add(i);
                        }
                        if (jf.isSimulatingKeystrokes(i)) {
                            mSimulatingKeystrokes.add(i);
                        }
                        mAvailableDevicetypes.put(i,
                                new LinkedList<>(jf.getAvailableInputDevicetypes(i)));
                        mAutofireFrequency.put(i, jf.getAutofireFrequency(i));
                        if (jf.getJoystickToKeyboardMapper(i) != null) {
                            mMappedJoystickKeys.put(i, jf.getJoystickToKeyboardMapper(i).getKeys());
                            mMappedJoystickModificatorKeys.put(i,
                                    jf.getJoystickToKeyboardMapper(i).getModificatorKeys());
                        }
                    }
                    List<Integer> emuMaxPorts = new LinkedList<>(mJoystickPorts.keySet());
                    if (conf.getUsedJoystickports(emuMaxPorts) != null
                            && !conf.isBareMachine()) {
                        mUsedJoystickPorts = conf.getUsedJoystickports(emuMaxPorts);
                    } else {
                        mUsedJoystickPorts = new ArrayList<>(emuMaxPorts);
                    }
                } else {
                    mUsedJoystickPorts = null;
                }
            } else {
                mUsedJoystickPorts = null;
            }
            Emulation.HardwareKeyboardFunctions kbf = source.getHardwareKeyboardFunctions();
            if (kbf != null) {
                if (kbf.getMappings() != null) {
                    mKeyboardMapping.addAll(kbf.getMappings());
                }
                ALL_SETTING_FRAGMENTS.put(this.mId, kbf.getSettingsFragment());
            }
            Emulation.MachineSettingsFunctions msf = source.getMachineSettingsFunction();
            if (msf != null) {
                mLayoutIds.put(false, addDualMonitor(source,
                        new LinkedList<>(msf.getLayouts(false))));
                mLayoutIds.put(true, addDualMonitor(source,
                        new LinkedList<>(msf.getLayouts(true))));
            }
            Emulation.SoftkeyFunctions skf = source.getSoftkeyFunctions();
            if (skf != null) {
                mSoftkeys.addAll(skf.getKeys());
                mModificatorSoftkeys.addAll(skf.getModificatorKeys());
            }
            Emulation.SoftwareKeyboardFunctions skbf = source.getSoftwareKeyboardFunctions();
            if (skbf != null && viewmodel.isSoftwareKeyboardAvailable()) {
                mSoftwarekeyboardVariants = skbf.supportsDifferentLayouts();
            } else {
                mSoftwarekeyboardVariants = false;
            }
            mIsBareMachine = conf.isBareMachine();
        }
        private List<Integer> addDualMonitor(final Emulation emu,
                                             final List<Integer> val) {
            if (emu.getDualMonitorFunctions() != null) {
                val.add(R.id.multimonitor_options);
            }
            return val;

        }

        @Override
        protected void finalize() throws Throwable {
            ALL_SETTING_FRAGMENTS.remove(this.mId);
            super.finalize();
        }

        @Override
        public Map<Integer, String> getJoystickports() {
            return mJoystickPorts;
        }
        private void unhandledMethod() {
            if (BuildConfig.DEBUG) {
                throw new IllegalStateException("May not reach this code.");
            }
        }

        @Override
        public boolean isExtentedJoystick(final int port) {
            return mExtendedJoysticks.contains(port);
        }

        @Override
        public boolean isSimulatingKeystrokes(final int port) {
            return mSimulatingKeystrokes.contains(port);
        }

        @Override
        public List<Emulation.InputDeviceType> getAvailableInputDevicetypes(final int port) {
            return mAvailableDevicetypes.get(port);
        }
        List<AdditionalKey> getKeyboardMappedJoystickKeys(final int port) {
            return mMappedJoystickKeys.get(port);
        }
        List<AdditionalKey> getKeyboardMappedJoystickModifcatorKeys(final int port) {
            return mMappedJoystickModificatorKeys.get(port);
        }
        @Override
        public Emulation.SoftkeyFunctions getJoystickToKeyboardMapper(final Integer port) {
            unhandledMethod();
            return null;
        }

        @Override
        public void setActiveInputDeviceType(final int port,
                                             final @Nullable Emulation.InputDeviceType newType) {
            unhandledMethod();
        }

        @Override
        public int getAutofireFrequency(final int port) {
            Integer value = mAutofireFrequency.get(port);
            return value == null ? 0 : value;
        }

        @Override
        public void onJoystickChanged(final String joyId, final Emulation.InputDeviceType inputType,
                                      final int portnumber, final boolean absoluteValues,
                                      final float xvalue, final float yvalue,
                                      final Set<Emulation.InputButton> buttons) {
            unhandledMethod();

        }

        @Override
        public List<KeyboardMapping> getMappings() {
            return mKeyboardMapping;
        }

        @Override
        public boolean onKeyDown(final KeyEvent keycode) {
            unhandledMethod();
            return false;
        }

        @Override
        public boolean onKeyUp(final KeyEvent keycode) {
            unhandledMethod();
            return false;
        }

        @Override
        public SettingsFragment getSettingsFragment() {
            if (ALL_SETTING_FRAGMENTS.containsKey(this.mId)) {
                return ALL_SETTING_FRAGMENTS.get(this.mId);
            }
            return null;
        }

        @Override
        public void applyUseropts(final Useropts useropts, final Runnable runAfter) {
            unhandledMethod();
        }

        @Override
        public List<Integer> getLayouts(final boolean modelIsFix) {
            return mLayoutIds.get(modelIsFix);
        }
        LinkedList<AdditionalKey> getSoftkeys() {
            return mSoftkeys;
        }
        LinkedList<AdditionalKey> getModificatorSoftkeys() {
            return mModificatorSoftkeys;
        }
        List<Integer> getUsedJoystickPorts() {
            return mUsedJoystickPorts;
        }
        boolean isWithSoftwarekeyboardVariants() {
            return mSoftwarekeyboardVariants;
        }
        boolean isBareMachine() {
            return mIsBareMachine;
        }

    }
    static class SettingsConfiguration implements Serializable {
        private final EmulationViewModel mViewModel;
        private final CurrentEmulationCallbacks mCurrentEmulationCallbacks;
        private final boolean mModelFix;

        // CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
        SettingsConfiguration(final EmulationViewModel vm,
                              final boolean isModelFix,
                              final CurrentEmulationCallbacks currentEmulationCallbacks) {
            mModelFix = isModelFix;
            mViewModel = vm;
            mCurrentEmulationCallbacks = currentEmulationCallbacks;
        }
        boolean isModelFix() {
            return mModelFix;
        }
        EmulationViewModel getViewModel() {
            return mViewModel;
        }
        CurrentEmulationCallbacks getCurrentEmulationCallbacks() {
            return mCurrentEmulationCallbacks;
        }

    }
    static final class ShowSettingsContract
            extends ActivityResultContract<SettingsConfiguration, Intent> {
        @NonNull
        @Override
        public Intent createIntent(@NonNull final Context context, final SettingsConfiguration sc) {
            Intent i = new Intent(context, SettingsActivity.class);
            int orientation = ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED;
            switch (getEmulationScreenOrientation(context, sc.getViewModel().getConfiguration())) {
                case PORTRAIT:
                    orientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
                    break;
                case LANDSCAPE:
                    orientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
                    break;
                default:
                    break;
            }

            i.putExtra("screen_orientation", orientation);
            i.putExtra(OPENSECTIONS, sc.getViewModel().getOpenSettingsSections());
            i.putExtra(CONFIGURATION, sc.getViewModel().getConfiguration());
            i.putExtra(ALLOWKEYBOARD, true);
            i.putExtra(CURRENT_STATE, sc.getCurrentEmulationCallbacks());
            i.putExtra(MODEL_IS_FIXED, sc.isModelFix());
            return i;
        }
        @Override
        public Intent parseResult(final int resultCode, final @Nullable Intent intent) {
            return intent;
        }
    }
    private final LinkedHashMap<Integer, Joystick> mActiveJoystick = new LinkedHashMap<>();

    interface JoystickEntry extends MappedSpinner.MappedSpinnerElement {
        Joystick getJoystick();

    }

    @Override
    protected void onSaveInstanceState(final @NonNull Bundle outState) {
        Fragment f = getSupportFragmentManager().findFragmentByTag("FRAGMENT_JOYSTICKMAPPING");
        if (f != null) {
            getSupportFragmentManager().beginTransaction().remove(f).commitNow();
        }

        super.onSaveInstanceState(outState);
    }

    private void onJoystickChanged(final int port, final Joystick item) {
        if (mActiveJoystick.get(port) != item && mActiveJoystick.get(port) != null) {
            //noinspection ConstantConditions
            mActiveJoystick.get(port).setPortnumber(Joystick.PORT_NOT_CONNECTED);
            mSettingsController.onJoystickportChanged(
                    Objects.requireNonNull(mActiveJoystick.get(port)), Joystick.PORT_NOT_CONNECTED);
            mActiveJoystick.put(port, item);
        }
        if (item != null) {

            item.setPortnumber(port);
            mSettingsController.onJoystickportChanged(item, port);
        }
        if (port != Joystick.PORT_NOT_CONNECTED) {
            mActiveJoystick.put(port, item);
        }
        JoystickSettingsView jsv = mJoystickView.get(port);
        if (jsv != null) {
            addView(jsv);
        }

    }



    @Override
    protected void onAvailableKeyboardsChanged(final boolean hardware, final boolean software) {
        super.onAvailableKeyboardsChanged(hardware, software);
        boolean showHardware;
        try {
            getClassLoader().loadClass("de/rainerhock/eightbitwonders/HardwareKeyboardTest");
            showHardware = Objects.requireNonNull(getIntent().getExtras())
                    .getBoolean(ALLOWKEYBOARD);
        } catch (ClassNotFoundException e) {
            showHardware = hardware && Objects.requireNonNull(getIntent().getExtras())
                    .getBoolean(ALLOWKEYBOARD);
        }

        findViewById(R.id.hardwareKeyboardGroup).setVisibility(showHardware
                ? View.VISIBLE : View.GONE);
        boolean showSoftware = getCurrentEmulationCallbacks().isWithSoftwarekeyboardVariants();
        findViewById(R.id.softwareKeyboardGroup).setVisibility(showSoftware
                ? View.VISIBLE : View.GONE);
        findViewById(R.id.gh_keyboard).setVisibility(showHardware || showSoftware
                ? View.VISIBLE : View.GONE);

    }

    private List<Integer> mOpenSections = new LinkedList<>();
    private List<Integer> getOpenSection() {
        return mOpenSections;
    }
    @NonNull  List<Joystick> getAvailableJoysticks() {
        return mAvailableJoysticks;
    }
    @NonNull  CurrentEmulationCallbacks getCurrentEmulationCallbacks() {
        return (CurrentEmulationCallbacks)
                Objects.requireNonNull(
                        Objects.requireNonNull(getIntent().getExtras())
                        .getSerializable(CURRENT_STATE));
    }
    private final List<Joystick> mAvailableJoysticks = new LinkedList<>();
    @Override
    protected void onAvailableJoysticksChanged(@NonNull final List<Joystick> joysticks) {
        mAvailableJoysticks.clear();
        mAvailableJoysticks.addAll(joysticks);
        if (!joysticks.isEmpty()) {
            joysticks.add(0, new MultiplexerJoystick(this, joysticks));
        }
        super.onAvailableJoysticksChanged(joysticks);
        if (getIntent().getExtras() != null) {
            final List<Integer> joystickports;
            CurrentEmulationCallbacks cec = getCurrentEmulationCallbacks();
            if (cec.getJoystickports() != null) {
                List<Integer> emuMaxPorts = new ArrayList<>(cec.getJoystickports().keySet());
                if (cec.getUsedJoystickPorts() != null && !cec.isBareMachine()) {
                    joystickports = cec.getUsedJoystickPorts();
                } else {
                    joystickports = new ArrayList<>(emuMaxPorts);
                }
            } else {
                joystickports = new ArrayList<>();
            }
            if (joystickports.isEmpty()) {
                findViewById(R.id.gh_joysticks).setVisibility(View.GONE);
                findViewById(R.id.configuration_virtual_singlejoystick)
                        .setVisibility(View.GONE);
                findViewById(R.id.joysticks)
                        .setVisibility(View.GONE);
            } else {
                configureMultiPorts(joysticks, joystickports);
                findViewById(R.id.configuration_virtual_singlejoystick)
                        .setVisibility(View.GONE);
            }
        }
    }
    private static class EmptyJopystickEntry implements JoystickEntry {
        private final Context mContext;

        EmptyJopystickEntry(final Context context) {
            mContext = context;
        }
        @Override
        public String getId() {
            return "";
        }
        @Override
        public String getText() {
            return toString();
        }

        @Override
        public Joystick getJoystick() {
            return null;
        }

        @NonNull
        public String toString() {
            return mContext.getResources().getString(R.string.IDS_NONE);
        }
    }
    private void configureMultiPorts(final List<Joystick> joysticks,
                                     final List<Integer> joystickports) {

        mActiveJoystick.clear();
        JoystickEntry empty = new EmptyJopystickEntry(this);
        final List<JoystickEntry> allJoystickEntries = new LinkedList<>();
        allJoystickEntries.add(empty);
        final List<JoystickEntry> noKbJoystickEntries = new LinkedList<>();
        noKbJoystickEntries.add(empty);
        for (final Joystick joy : joysticks) {
            joy.readUseropts(this);
            joy.setPortnumber(mSettingsController.getCurrentJoystickport(joy));
            if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                mActiveJoystick.put(joy.getPortnumber(), joy);
            }
            JoystickEntry e = new JoystickEntry() {
                @Override
                public String getId() {
                    return joy.getId();
                }

                @Override
                public String getText() {
                    return toString();
                }

                @Override
                @NonNull
                public Joystick getJoystick() {
                    return joy;
                }

                @NonNull
                public String toString() {
                    return joy.toString();
                }
            };
            allJoystickEntries.add(e);
            if (!(joy instanceof KeyboardJoystick)) {
                noKbJoystickEntries.add(e);
            }
        }
        ArrayAdapter<JoystickEntry> allAdapter =
                new ArrayAdapter<>(this,
                        android.R.layout.simple_spinner_item, allJoystickEntries);
        ArrayAdapter<JoystickEntry> noKbAdapter =
                new ArrayAdapter<>(this,
                        android.R.layout.simple_spinner_item, noKbJoystickEntries);

        findViewById(R.id.gh_joysticks)
                .setVisibility(joysticks.isEmpty() ? View.GONE : View.VISIBLE);
        findViewById(R.id.configuration_virtual_singlejoystick).setVisibility(View.GONE);
        for (int port : mJoystickView.keySet()) {
            JoystickSettingsView jsv = mJoystickView.get(port);
            if (jsv != null) {
                boolean show = joystickports.size() > 1 || port == joystickports.get(0);
                if (show) {
                    jsv.setVisibility(View.VISIBLE);
                    allAdapter.setDropDownViewResource(
                            android.R.layout.simple_spinner_dropdown_item);
                    noKbAdapter.setDropDownViewResource(
                            android.R.layout.simple_spinner_dropdown_item);
                    jsv.setAdapter(jsv.isExcludingHardwarekeyboard() ? noKbAdapter : allAdapter);
                    int selection = 0;
                    jsv.setSelection(Spinner.NO_ID);
                    jsv.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
                        @Override
                        public void onItemSelected(final AdapterView<?> parent, final View view,
                                                   final int position, final long id) {
                            if (allAdapter.getItem(position) != null) {
                                //noinspection DataFlowIssue
                                Joystick joy = allAdapter.getItem(position).getJoystick();
                                //if (joy != null) {
                                for (JoystickSettingsView otherSpinner : mJoystickView.values()) {
                                    Joystick otherJoy = otherSpinner.getCurrentJoystick();
                                    if (otherSpinner.getId() != jsv.getId()
                                            && otherJoy != null && otherJoy.conflictsWith(joy)) {
                                        otherSpinner.setAdapter(otherSpinner.getAdapter());
                                        //noinspection ConstantConditions
                                        onJoystickChanged(mJoystickPort.get(otherSpinner), null);
                                        otherSpinner.getAdapter().notifyDataSetChanged();
                                        otherSpinner.setSelection(View.NO_ID);
                                    }
                                }
                                //}
                            }
                            //noinspection ConstantConditions
                            onJoystickChanged(mJoystickPort.get(jsv),
                                    allAdapter.getItem(position).getJoystick());
                            View assignKeysParent = jsv.findViewById(R.id.keyassignment_container);
                            assignKeysParent.setVisibility(position > 0 ? View.VISIBLE : View.GONE);

                        }

                        @Override
                        public void onNothingSelected(final AdapterView<?> parent) {
                            //noinspection ConstantConditions
                            onJoystickChanged(mJoystickPort.get(jsv), null);
                        }
                    });
                    CurrentEmulationCallbacks cec = getCurrentEmulationCallbacks();
                    if (joystickports.size() > 1) {

                        for (Joystick joy : joysticks) {
                            selection++;
                            //noinspection ConstantConditions
                            if (joy.getPortnumber() == mJoystickPort.get(jsv)) {
                                if (cec.isExtentedJoystick(joy.getPortnumber())) {
                                    ((SwitchCompat) findViewById(R.id.sw_enable_extended_joysticks))
                                            .setChecked(true);
                                    findViewById(R.id.extended_joysticks)
                                            .setVisibility(View.VISIBLE);
                                }
                                if (cec.isSimulatingKeystrokes(port)) {
                                    ((SwitchCompat) findViewById(
                                            R.id.sw_enable_keystroke_joysticks)).setChecked(true);
                                    findViewById(R.id.keystroke_joysticks)
                                            .setVisibility(View.VISIBLE);
                                }

                                jsv.setSelection(joy);
                                jsv.setSelection(selection);
                                Integer p = mJoystickPort.get(jsv);
                                if (p != null) {
                                    onJoystickChanged(p, joy);
                                }
                            }
                        }
                    } else {
                        jsv.setAsSingleview();
                        findViewById(R.id.keystroke_joysticks).setVisibility(View.VISIBLE);
                        findViewById(R.id.sw_enable_keystroke_joysticks).setVisibility(View.GONE);

                        findViewById(R.id.extended_joysticks).setVisibility(View.VISIBLE);
                        findViewById(R.id.sw_enable_extended_joysticks).setVisibility(View.GONE);
                    }
                } else {
                    jsv.setVisibility(View.GONE);
                }
            }
        }
    }

    private final Map<JoystickSettingsView, Integer> mJoystickPort = new HashMap<>();
    private final HashMap<Integer, JoystickSettingsView> mJoystickView = new HashMap<>();

    private void updateHeaderButtons(final View v, final List<Integer> opensections) {
        ViewGroup viewgroup = (ViewGroup) v;
        for (int i = 0; i < viewgroup.getChildCount(); i++) {
            View child = viewgroup.getChildAt(i);
            if (child instanceof ViewGroup) {
                updateHeaderButtons(child, opensections);
            }
            if (child instanceof GroupHeader) {
                GroupHeader b = (GroupHeader) child;
                b.setViewVisible(b.isViewVisible() || opensections.contains(b.getTargetId()));
                b.setStateChangedListener((btn, isOpen) -> {
                    if (isOpen) {
                        if (!getOpenSection().contains(btn.getTargetId())) {
                            getOpenSection().add(btn.getTargetId());
                        }
                    } else {
                        getOpenSection().remove(Integer.valueOf(btn.getTargetId()));
                    }

                });
            }
        }
    }

    private void addMachineSettings(final List<Integer> layouts) {
        for (int id : layouts) {
            if (findViewById(id) != null) {
                findViewById(id).setVisibility(View.VISIBLE);
            }
        }
    }

    interface ResourceDescription {
        Scope getSettingsScope();

        String getDependsOnKey();

        String getDependsOnValue();
    }

    ResourceDescription getResourceDescription(final Integer id) {
        return new ResourceDescription() {

            @Override
            public Scope getSettingsScope() {
                if (mViewScope.containsKey(id)) {
                    return mViewScope.get(id);
                } else {
                    return Scope.UNIQUE;
                }
            }

            @Override
            public String getDependsOnValue() {
                return Objects.requireNonNull(
                        mViewReferences.get(R.styleable.useropts_depends_on_val)).get(id);
            }

            @Override
            public String getDependsOnKey() {
                return Objects.requireNonNull(
                        mViewReferences.get(R.styleable.useropts_depends_on_key)).get(id);
            }
        };
    }

    enum Scope {
        GLOBAL, SHARED, UNIQUE
    }

    private final Map<Integer, Map<Integer, String>> mViewReferences =
            new LinkedHashMap<Integer, Map<Integer, String>>() {

                private static final long serialVersionUID = 7904857714035892071L;

                {
                    put(R.styleable.useropts_depends_on_key, new LinkedHashMap<>());
                    put(R.styleable.useropts_depends_on_val, new LinkedHashMap<>());
                    put(R.styleable.useropts_true_val, new LinkedHashMap<>());
                    put(R.styleable.useropts_false_val, new LinkedHashMap<>());
                    put(R.styleable.useropts_layout_id, new LinkedHashMap<>());
                    put(R.styleable.useropts_number_of_joysticks, new LinkedHashMap<>());
                    put(R.styleable.useropts_userdata, new LinkedHashMap<>());
                }
            };
    private final Map<Integer, Scope> mViewScope = new LinkedHashMap<Integer, Scope>() {
        {
            put(R.id.sb_joystick_transparency, Scope.GLOBAL);
        }
    };
    private SettingsController mSettingsController = null;
    SettingsController getSettingsController() {
        return mSettingsController;
    }
    void addView(final View view) {
        Set<View> views = addWidget(view, new HashSet<>());
        for (View v : views) {
            if (mSettingsController != null) {
                mSettingsController.addWidget(v.getId(), v);
                mSettingsController.initWidget(v);
            }
        }
        CurrentEmulationCallbacks cec = getCurrentEmulationCallbacks();
        boolean modelIsFixed = getIntent().getExtras().getBoolean(MODEL_IS_FIXED);

        List<Integer> layouts =
                cec.getLayouts(modelIsFixed);
        if (layouts != null) {
            for (View v : views) {
                if (layouts.contains(v.getId())) {
                    v.setVisibility(View.VISIBLE);
                }
            }
        }
    }

    static Set<View> addWidget(final View v, final Set<View> list) {
        if (v instanceof ViewGroup) {
            ViewGroup vg = (ViewGroup) v;
            for (int i = 0; i < vg.getChildCount(); i++) {
                addWidget(vg.getChildAt(i), list);
            }
        }
        if (v.getId() == View.NO_ID && v.getTag() != null) {
            v.setId(View.generateViewId());
        }
        if (v.getId() != View.NO_ID) {
            list.add(v);
        }
        return list;
    }
    private void updateOpenSections() {
        updateHeaderButtons(getWindow().getDecorView(), getOpenSection());

    }
    @Override
    protected void onResume() {
        super.onResume();
        updateHardware();
        updateOpenSections();
    }
    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        CurrentEmulationCallbacks cec = getCurrentEmulationCallbacks();
        boolean modelIsFix = getIntent().getExtras().getBoolean(MODEL_IS_FIXED);
        getOnBackPressedDispatcher().addCallback(this, new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                if (isEnabled()) {
                    Intent intent = new Intent();
                    intent.putExtra(OPENSECTIONS, new LinkedList<>(getOpenSection()));
                    setResult(RESULT_CANCELED, intent);
                    finish();
                }
            }
        });


        if (getIntent().getExtras() != null && getIntent().getExtras().getBoolean(ALLOWKEYBOARD)) {
            SettingsFragment f = cec.getSettingsFragment();
            if (f != null) {
                getSupportFragmentManager().beginTransaction().add(R.id.hardwareKeyboardGroup,
                        cec.getSettingsFragment(), "").commitNow();
            }
        }

        //noinspection unchecked
        mOpenSections = (List<Integer>) getIntent().getSerializableExtra(OPENSECTIONS);
        Map<Integer, SettingsViewModifier> modifiers = new HashMap<>();
        SettingsController controller = new SettingsController();
        setSettingsController(controller, modifiers);

        SoftkeyConfigView scv = findViewById(R.id.softkeyconfig);
        scv.setUseropts(mSettingsController.getUseropts());
        scv.loadConfiguration();
        scv.updateUi();
        List<Integer> joystickports = new LinkedList<>(cec.getJoystickports().keySet());
        Collections.sort(joystickports);

        updateJoystickViews(joystickports, cec);

        updateHeaderButtons(getWindow().getDecorView(), getOpenSection());
        List<Integer> layouts = cec.getLayouts(modelIsFix);
        if (layouts != null) {
            addMachineSettings(layouts);

        }
        if (getIntent() != null) {
            String title = getIntent().getStringExtra(
                    getApplication().getPackageName() + ".title");
            if (title != null) {
                setTitle(title);
            }
        }
        findViewById(R.id.bn_apply).setOnClickListener(view -> {
            mSettingsController.apply();
            scv.saveConfiguration();
            Intent intent = new Intent();
            intent.putExtra(OPENSECTIONS, new LinkedList<>(getOpenSection()));
            setResult(RESULT_OK, intent);

            finish();
        });
    }

    private void updateJoystickViews(
            final List<Integer> joystickports,
            final Emulation.JoystickFunctions jf) {

        if (joystickports.size() > 1) {
            mJoystickPort.clear();
            mJoystickView.clear();

            boolean hasExtendedPorts = false;
            boolean hasStandardPorts = false;
            boolean hasKeystrokePorts = false;
            List<JoystickSettingsView> standardJoysticks = new LinkedList<>();
            CurrentEmulationCallbacks cec = getCurrentEmulationCallbacks();
            if (jf != null) {
                for (int port : joystickports) {
                    ArrayList<Emulation.InputDeviceType> types = new ArrayList<>(
                            jf.getAvailableInputDevicetypes(port));

                    LinearLayout parent;
                    if (cec.isSimulatingKeystrokes(port)) {
                        parent = findViewById(R.id.keystroke_joysticks);
                        hasKeystrokePorts = true;
                    } else {
                        if (cec.isExtentedJoystick(port)) {
                            parent = findViewById(R.id.extended_joysticks);
                            hasExtendedPorts = true;
                        } else {
                            parent = findViewById(R.id.default_joysticks);
                            hasStandardPorts = true;
                        }
                    }
                    int maxFrequency = jf.getAutofireFrequency(port);
                    String portname = jf.getJoystickports().get(port);
                    if (portname != null) {
                        JoystickSettingsView jsv = createJoystickView(port, parent,
                                portname, types, maxFrequency);
                        if (parent.getId() == R.id.default_joysticks) {
                            standardJoysticks.add(jsv);
                        }
                    }
                }
                int usedJoysticks = 0;
                for (int port : cec.getUsedJoystickPorts()) {
                    if (!cec.isExtentedJoystick(port) && !cec.isSimulatingKeystrokes(port)) {
                        usedJoysticks++;
                    }
                }
                if (standardJoysticks.size() == 2 && usedJoysticks > 1) {
                    final JoystickSettingsView jsv1 = standardJoysticks.get(0);
                    final JoystickSettingsView jsv2 = standardJoysticks.get(1);

                    Button b = findViewById(R.id.bn_swap_standardports);
                    b.setVisibility(View.VISIBLE);
                    b.setOnClickListener(v -> {
                        if (jsv1.getAdapter() != null && jsv2.getAdapter() != null) {
                            Joystick j1 = jsv1.getCurrentJoystick();
                            Joystick j2 = jsv2.getCurrentJoystick();
                            jsv1.setSelection(j2);
                            jsv2.setSelection(j1);
                            Integer p1 = mJoystickPort.get(jsv1);
                            if (p1 != null) {
                                onJoystickChanged(p1, jsv1.getCurrentJoystick());
                            }
                            Integer p2 = mJoystickPort.get(jsv2);
                            if (p2 != null) {
                                onJoystickChanged(p2, jsv2.getCurrentJoystick());
                            }
                        }
                    });
                }
            }
            if (!hasStandardPorts && hasExtendedPorts) {
                findViewById(R.id.sw_enable_extended_joysticks).setVisibility(View.GONE);
                findViewById(R.id.extended_joysticks).setVisibility(View.VISIBLE);
                findViewById(R.id.default_joysticks).setVisibility(View.GONE);
            }
            if (hasExtendedPorts) {
                SwitchCompat sw = findViewById(R.id.sw_enable_extended_joysticks);
                sw.setOnCheckedChangeListener((compoundButton, b)
                        -> findViewById(R.id.extended_joysticks).setVisibility(b
                        ? View.VISIBLE : View.GONE));
            }
            if (hasKeystrokePorts) {
                SwitchCompat sw = findViewById(R.id.sw_enable_keystroke_joysticks);
                sw.setOnCheckedChangeListener((compoundButton, b)
                        -> findViewById(R.id.keystroke_joysticks).setVisibility(b
                        ? View.VISIBLE : View.GONE));

            }
        } else {
            findViewById(R.id.gh_joysticks).setVisibility(View.GONE);
            findViewById(R.id.joystickgroup).setVisibility(View.GONE);
        }
    }

    private void populateKeyMapping(final ViewGroup parent,
                                    final @IdRes int rowViewId,
                                    final int useroptIndex,
                                    final List<EmulationUi.SettingSpinnerElement> allKeys,
                                    final List<EmulationUi.SettingSpinnerElement>
                                            modificatorKeys,
                                    final int joystickport) {
        TableRow row = parent.findViewById(rowViewId);
        String keyUseropt = getResources().getResourceName(rowViewId)
                + "_" + joystickport + "_key";
        String keyValue = getCurrentUseropts()
                .getStringValue(keyUseropt, "");
        String modificatorUseropt = getResources().getResourceName(rowViewId)
                + "_" + joystickport + "_modificator";
        String modificatorValue = getCurrentUseropts()
                .getStringValue(modificatorUseropt, "");
        MappedSpinner spKey = row.findViewWithTag("key");
        spKey.populate(allKeys);
        spKey.setSelection(keyValue);

        MappedSpinner spModificator = row.findViewWithTag("modificator");
        spModificator.populate(modificatorKeys);
        spModificator.setSelection(modificatorValue);
        spModificator.setEnabled(!keyValue.isEmpty());

        spKey.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(final AdapterView<?> adapterView,
                                       final View view, final int pos, final long l) {
                String newKeyValue = ((EmulationUi.SettingSpinnerElement)
                        spKey.getItemAtPosition(pos)).getId();
                spModificator.setEnabled(!newKeyValue.isEmpty());
                if (newKeyValue.isEmpty()) {
                    spModificator.setSelection("");
                }
                String newModificatorValue = ((EmulationUi.SettingSpinnerElement) spModificator
                        .getItemAtPosition(spModificator.getSelectedItemPosition())).getId();
                //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
                mSettingsController.updateJoystickMapping(joystickport * 10 + useroptIndex,
                        keyUseropt, modificatorUseropt,
                        newKeyValue, newModificatorValue);
            }
            @Override
            public void onNothingSelected(final AdapterView<?> adapterView) {

            }
        });
        spModificator.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(final AdapterView<?> adapterView, final View view,
                                       final int pos, final long l) {
                String newModificatorValue = ((EmulationUi.SettingSpinnerElement)
                        spModificator.getItemAtPosition(pos)).getId();
                String newKeyValue = ((EmulationUi.SettingSpinnerElement)
                        spKey.getItemAtPosition(spKey.getSelectedItemPosition())).getId();
                //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
                mSettingsController.updateJoystickMapping(joystickport * 10 + useroptIndex,
                        keyUseropt, modificatorUseropt,
                        newKeyValue, newModificatorValue);

            }

            @Override
            public void onNothingSelected(final AdapterView<?> adapterView) {

            }
        });
    }
    private void populateKeymappingTable(final TableLayout view,
                                         final List<AdditionalKey> keys,
                                         final List<AdditionalKey> modificationKeys,
                                         final int port) {
        EmulationUi.SettingSpinnerElement emptyElement =
                new EmulationUi.SettingSpinnerElement() {
                    @Override
                    public String getId() {
                        return "";
                    }

                    @Override
                    public String getText() {
                        return view.getContext().getString(R.string.IDS_NONE);
                    }
                };

        List<EmulationUi.SettingSpinnerElement> modificatorKeys
                = new LinkedList<>();
        modificatorKeys.add(emptyElement);
        for (AdditionalKey k : modificationKeys) {
            modificatorKeys.add(new EmulationUi.SettingSpinnerElement() {
                @Override
                public String getId() {
                    return k.getId();
                }

                @Override
                public String getText() {
                    return k.getString();
                }
            });
        }
        List<EmulationUi.SettingSpinnerElement> allKeys = new LinkedList<>();
        allKeys.add(emptyElement);
        for (AdditionalKey k : keys) {
            allKeys.add(new EmulationUi.SettingSpinnerElement() {
                @Override
                public String getId() {
                    return k.getId();
                }

                @Override
                public String getText() {
                    return k.getString();
                }
            });
        }
        int index = 0;
        for (int id : Arrays.asList(R.id.tr_up_key, R.id.tr_down_key, R.id.tr_left_key,
                R.id.tr_right_key, R.id.tr_fire_key)) {
            populateKeyMapping(view, id, index, allKeys, modificatorKeys, port);
            index++;
        }
    }
    private JoystickSettingsView createJoystickView(final int port, final LinearLayout parent,
                                    final String portname,
                                    final List<Emulation.InputDeviceType> types,
                                    final int maxAutofireFrequency) {
        CurrentEmulationCallbacks cec = getCurrentEmulationCallbacks();

        JoystickSettingsView jsv = new JoystickSettingsView(this,
                getSupportFragmentManager(), mSettingsController, port, types, maxAutofireFrequency,
                parent.getId() == R.id.keystroke_joysticks);
        jsv.setEmulatedDevicetypes(types);
        parent.addView(jsv);
        jsv.setId(View.NO_ID);
        jsv.setPort(port, portname);
        jsv.setId(View.generateViewId());
        addView(jsv);
        //mJoystickDiagonalsView.put(port, joystick.findViewById(R.id.cb_lock_diagonals));
        mJoystickPort.put(jsv, port);
        mJoystickView.put(port, jsv);

        TableLayout tlMap = jsv.findViewById(R.id.table);

        if (cec.getKeyboardMappedJoystickModifcatorKeys(port) != null) {
            populateKeymappingTable(tlMap, cec.getKeyboardMappedJoystickKeys(port),
                    cec.getKeyboardMappedJoystickModifcatorKeys(port), port);
            jsv.findViewById(R.id.map_joystick_to_keys).setVisibility(View.VISIBLE);

        } else {
            jsv.findViewById(R.id.map_joystick_to_keys).setVisibility(View.GONE);
        }
        return jsv;
    }

    private static void setTypedAttributes(final Context context,
                                           final AttributeSet attrs,
                                           final Map<Integer, Map<Integer, String>>
                                                   viewReferences,
                                           final Map<Integer, SettingsActivity.Scope> viewScope) {
        Scope scope = null;
        //noinspection resource
        TypedArray typesUseropts = context.getResources()
                .obtainAttributes(attrs, R.styleable.useropts);
        //noinspection resource
        TypedArray typeId = context.obtainStyledAttributes(attrs,
                new int[]{android.R.attr.id});

        try {
            if (typesUseropts.hasValue(useropts_scope)) {
                switch (typesUseropts.getInt(useropts_scope, 0)) {
                    case 2:
                        scope = Scope.SHARED;
                        break;
                    case 3:
                        scope = Scope.UNIQUE;
                        break;
                    case 1:
                        scope = Scope.GLOBAL;
                        break;
                    default:
                        break;

                }
            }
            int id = typeId.getResourceId(0, View.NO_ID);
            if (id != View.NO_ID) {
                for (Integer i : viewReferences.keySet()) {
                    if (typesUseropts.hasValue(i)) {
                        if (viewReferences.get(i) != null) {
                            //noinspection ConstantConditions
                            viewReferences.get(i).put(id, typesUseropts.getString(i));
                        }
                    }
                }
                if (scope != null) {
                    viewScope.put(id, scope);
                }

            }
        } finally {
            typeId.recycle();
            typesUseropts.recycle();
        }
    }
    @Override
    public View onCreateView(final View parent,
                             @NonNull final String name,
                             @NonNull final Context context,
                             @NonNull final AttributeSet attrs) {
        setTypedAttributes(context, attrs, mViewReferences, mViewScope);
        return super.onCreateView(parent, name, context, attrs);
    }

    @Override
    public View onCreateView(@NonNull final String name,
                             @NonNull final Context context,
                             @NonNull final AttributeSet attrs) {
        setTypedAttributes(context, attrs, mViewReferences, mViewScope);
        return super.onCreateView(name, context, attrs);
    }
    interface SettingsViewModifier {
        void modify(View view);
    }
    private void setSettingsController(final SettingsController settingsController,
                                       final Map<Integer, SettingsViewModifier> modifier) {
        if (settingsController != null) {
            mSettingsController = settingsController;
            mSettingsController.setActivity(this);
            mSettingsController.setModifiers(modifier);
        }
        //getLayoutInflater().inflate(R.layout.activity_settings, null);
        setContentView(R.layout.activity_settings);
        final View root = getWindow().getDecorView().getRootView();
        addView(root);
        AndroidCompatibilityUtils.fixInsets(this);

        // Ensure insets are applied

        //root.post(root::requestFocus);
    }

}
