package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.res.Configuration;
import android.graphics.Point;
import android.os.Build;
import android.util.Log;
import android.view.Display;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.PointerIcon;
import android.view.View;

import androidx.annotation.NonNull;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

class MouseController extends AnalogController
        implements View.OnKeyListener, View.OnGenericMotionListener {
    private static final String TAG = MouseController.class.getSimpleName();
    private final InputDevice mDevice;
    private PointerIcon mPointerIconNull = null;
    private PointerIcon mPointerIconDefault = null;
    private EmulationActivity mActivity = null;

    MouseController(final InputDevice device) {
        mDevice = device;
    }
    @Override
    void connect(final @NonNull  EmulationActivity activity,
                 final JoystickListener listener,
                 final Emulation.GamepadFunctions gamepadListener,
                 final MultiplexerJoystick muxer) {
        mActivity = activity;
        Display display = mActivity.getWindowManager().getDefaultDisplay();
        Point size = new Point();
        display.getSize(size);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            mPointerIconDefault = PointerIcon.getSystemIcon(activity, PointerIcon.TYPE_DEFAULT);
            mPointerIconNull = PointerIcon.getSystemIcon(activity, PointerIcon.TYPE_NULL);
            View rootView = getRootView();
            if (mPointerIconNull.equals(rootView.getPointerIcon())) {
                if (getPortnumber() != Joystick.PORT_NOT_CONNECTED || isMuxerConnected()) {
                    catchMouse(rootView);
                } else {
                    rootView.releasePointerCapture();
                    rootView.setPointerIcon(mPointerIconDefault);

                }
            }
        }
        super.connect(activity, listener, gamepadListener, muxer);
    }
    @Override
    protected void onConfigurationChanged(final @NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
    }

    @Override
    String getId() {
        return mDevice.getDescriptor();
    }

    @Override
    String getHardwareDescriptor() {
        return mDevice.getDescriptor();
    }

    @Override
    int getHardwareId() {
        return mDevice.getId();
    }

    @Override
    boolean canLockDiagonals() {
        return false;
    }

    @Override
    boolean canForceCenterFirst() {
        return false;
    }

    @Override
    boolean hasSecondaryButton() {
        return true;
    }

    @Override
    InputDeviceConfigFragment getDstickConfigFragment() {
        return null;
    }

    @NonNull
    @Override
    List<Emulation.InputDeviceType> getSupportedDeviceTypes() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            return Arrays.asList(Emulation.InputDeviceType.MOUSE, Emulation.InputDeviceType.PADDLE);
        } else {
            return Collections.singletonList(Emulation.InputDeviceType.PADDLE);
        }
    }

    @Override
    boolean deviceHasButtonWithKeycode(final int keycode) {
        return false;
    }
    private boolean isMouseAttached() {
        if (mActivity != null) {
            View rootView = getRootView();
            return rootView != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.O
                    && mPointerIconNull.equals(rootView.getPointerIcon());
        } else {
            return false;
        }
    }
    private final Set<Integer> mPressedKeys = new HashSet<>();
    @SuppressLint("GestureBackNavigation")
    @Override
    public boolean onKey(final View v, final int keyCode, final KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            if (event.getDeviceId() == getHardwareId() && isMouseAttached()) {
                if (event.getAction() == KeyEvent.ACTION_DOWN) {
                    mPressedKeys.add(PSEUDO_KEYCODE_PADDLE_Y);
                    setRealButtonsPressed(mPressedKeys);
                    notifyListener();
                    return true;
                }
                if (event.getAction() == KeyEvent.ACTION_UP) {
                    mPressedKeys.remove(PSEUDO_KEYCODE_PADDLE_Y);
                    setRealButtonsPressed(mPressedKeys);
                    notifyListener();
                    return true;
                }
            }
        }
        return false;
    }
    protected boolean isDeviceMatch(final InputDevice dev) {
        if (dev == null && mDevice == null) {
            return true;
        }
        if (dev != null && mDevice != null) {
            return dev.getId() == mDevice.getId();
        }
        return false;
    }
    private View getRootView() {
        return mActivity.findViewById(R.id.screen);

    }
    private static final float BASE_SPEED_FACTOR = 1;

    @Override
    protected boolean showMouseUsageHint() {
        return true;
    }

    protected float getMovementFactor() {
        return BASE_SPEED_FACTOR * super.getMovementFactor();
    }

    private static final float MOUSE_MOVE_FACTOR = (float) Math.pow(2, 0.5);
    private static final  Set<Integer> NO_BUTTONS = new HashSet<>();
    @Override
    public boolean onGenericMotion(final View v, final MotionEvent event) {
        if (isDeviceMatch(event.getDevice()) && isListenerAttached()
                && (getPortnumber() != Joystick.PORT_NOT_CONNECTED || isMuxerConnected())
                && Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && mActivity != null) {
            if (event.getAction() == MotionEvent.ACTION_SCROLL) {
                View rootView = getRootView();
                int axis;
                if (BaseActivity.isInUnitTest()) {
                    axis = MotionEvent.AXIS_Y;
                } else {
                    axis = MotionEvent.AXIS_VSCROLL;
                }
                if (event.getAxisValue(axis) < 0) {
                    mActivity.getViewModel().addActiveMouse(this.getHardwareId());
                    rootView.setPointerIcon(mPointerIconNull);
                    setRealButtonsPressed(NO_BUTTONS);
                    notifyListener();
                    catchMouse(rootView);

                }

                if (event.getAxisValue(axis) > 0) {
                    mActivity.getViewModel().removeActiveMouse(this.getHardwareId());
                    rootView.releasePointerCapture();
                    rootView.setPointerIcon(mPointerIconDefault);
                    setRealButtonsPressed(NO_BUTTONS);
                    notifyListener();
                }
                return true;
            }
            if (isMouseAttached()) {
                final int action = event.getAction();
                switch (action) {
                    case MotionEvent.ACTION_MOVE:
                        final float f = getMovementFactor() / MOUSE_MOVE_FACTOR;
                        setXValue(event.getX() * f);
                        setYValue(event.getY() * f);
                        notifyListener();
                        return true;
                    case MotionEvent.ACTION_HOVER_ENTER:
                    case MotionEvent.ACTION_HOVER_EXIT:
                    case MotionEvent.ACTION_HOVER_MOVE:
                        mActivity.getViewModel().addActiveMouse(this.getHardwareId());
                        catchMouse(getRootView());
                        return true;
                    case MotionEvent.ACTION_BUTTON_PRESS:
                    case MotionEvent.ACTION_BUTTON_RELEASE:
                        boolean isPrimary;
                        if (DeviceSupportActivity.isInUnitTest()) {
                            isPrimary = event.getX() == MotionEvent.BUTTON_PRIMARY;
                        } else {
                            isPrimary = event.getActionButton() == MotionEvent.BUTTON_PRIMARY;
                        }
                        handleButtons(action == MotionEvent.ACTION_BUTTON_PRESS,
                                isPrimary, PSEUDO_KEYCODE_PADDLE_X);
                        boolean isSecondary;
                        if (DeviceSupportActivity.isInUnitTest()) {
                            isSecondary = event.getX() == MotionEvent.BUTTON_SECONDARY;
                        } else {
                            isSecondary = event.getActionButton() == MotionEvent.BUTTON_SECONDARY
                                    || event.getActionButton() == MotionEvent.BUTTON_BACK;
                        }
                        handleButtons(action == MotionEvent.ACTION_BUTTON_PRESS,
                                isSecondary, PSEUDO_KEYCODE_PADDLE_Y);
                        return true;
                    case MotionEvent.ACTION_DOWN:
                    case MotionEvent.ACTION_UP:
                        return true;
                    default:
                        Log.v(TAG, "Not handling " + event);
                        return false;

                }
            }
        }
        return false;
    }

    private void catchMouse(final View rootView) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {

            rootView.post(() -> {
            rootView.setOnFocusChangeListener((dummy, hasFocus) -> {
                if (hasFocus) {
                    rootView.requestPointerCapture();
                }
            });
            rootView.requestFocus();
            rootView.requestPointerCapture();
            Log.v(TAG, "hasPointerCapture: " + getRootView().hasPointerCapture());
            rootView.setOnCapturedPointerListener((view, event1)
                    -> onGenericMotion(null, event1));
            });
        }
    }

    private void handleButtons(final boolean isPressed, final boolean isPrimary,
                               final int keycode) {
        if (isPrimary && isMouseAttached()) {
            if (isPressed) {
                mPressedKeys.add(keycode);
            } else {
                mPressedKeys.remove(keycode);
            }
            setRealButtonsPressed(mPressedKeys);
            notifyListener();
        }
    }

    @NonNull
    @Override
    public String toString() {
        return mDevice.getName();
    }

}
