//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.ContextWrapper;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.graphics.Rect;
import android.opengl.GLES20;
import android.opengl.GLSurfaceView;
import android.opengl.GLUtils;
import android.os.Build;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.PointerIcon;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ScrollView;

import androidx.annotation.NonNull;
import androidx.core.view.GestureDetectorCompat;


import java.io.IOException;
import java.io.InputStream;
// CHECKSTYLE DISABLE UnusedImport FOR 1 LINES
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.microedition.khronos.egl.EGLConfig;
import javax.microedition.khronos.opengles.GL10;

/**
 * The View to be drawn on on. Emulations don't use it directly.
 */
public final class MonitorGlSurfaceView extends GLSurfaceView implements GLSurfaceView.Renderer,
        EmulationTestInterface {
    private static final String TAG = MonitorGlSurfaceView.class.getSimpleName();
    private int mCanvasWidth = 0;
    private float mCanvasAspectRatio = 0;
    private float mPixelAspectRatio = 0;
    private ScrollView mScrollview = null;
    private long mStartTime = 0;
    private Useropts mUseropts = null;
    private final Map<String, Float> mSettingsParameters = new HashMap<>();

    /**
     * Simple constructor to use when creating a MonitorGlSurfaceView from code.
     *
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public MonitorGlSurfaceView(final Context context) {
        super(context);
        init();
    }

    /**
     * Constructor that is called when inflating a MappedSpinner from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     *
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs   The attributes of the XML tag that is inflating the view.
     *                This value may be null.
     */
    public MonitorGlSurfaceView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }
    private boolean mScreenViewListenerAdded = false;
    private int mViewHeight = 0;
    /**
     * @noinspection unused
     */
    private int mViewWidth = 0;
    private final Rect mOldScrollbarRect = new Rect(0, 0, 0, 0);
    private final Rect mNewScrollbarRect = new Rect(0, 0, 0, 0);

    void setScrollView(final @NonNull ScrollView sv) {
        if (mScrollview != sv) {
            boolean cont = mScrollview == null;

            if (!cont) {
                sv.getGlobalVisibleRect(mNewScrollbarRect);
                mScrollview.getGlobalVisibleRect(mOldScrollbarRect);
                cont = !mNewScrollbarRect.equals(mOldScrollbarRect);
            }
            mScrollview = sv;
            if (cont) {
                Log.v("crashdetect", "scrollview update required");
                mScrollview.getGlobalVisibleRect(mOldScrollbarRect);
                // CHECKSTYLE DISABLE ParameterNumber FOR 2 LINES
                if (!mScreenViewListenerAdded) {
                    sv.addOnLayoutChangeListener((v, left, top,
                                                  right, bottom,
                                                  oldLeft, oldTop,
                                                  oldRight, oldBottom) -> {
                        Log.v("crashdetect", "scrollview relayouted");
                        if (right != oldRight || left != oldLeft
                                || top != oldTop || bottom != oldBottom) {
                            Log.v("LayoutChanged", "scrollview: " + left + ", " + top
                                    + ", " + right + ", " + bottom);
                            mViewWidth = right - left;
                            mViewHeight = bottom - top;
                            Log.v("crashdetect", "calcTransformation called from scrollview");
                            calcTransformation();
                        }
                    });
                    mScreenViewListenerAdded = true;
                }
            }
        }
    }

    private EmulationActivity getActivity() {
        Context context = getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof EmulationActivity) {
                return (EmulationActivity) context;
            }
            context = ((ContextWrapper) context).getBaseContext();
        }
        return null;
    }
    private void init() {
        setEGLContextClientVersion(2);
        // setRenderer(this);
        setRenderer(this);
        setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
        GestureDetectorCompat mGestureDetector = new GestureDetectorCompat(getContext(),
                new GestureDetector.SimpleOnGestureListener() {
                    @Override
                    public boolean onDown(@NonNull final MotionEvent e) {
                        return true;
                    }

                    @Override
                    public boolean onDoubleTap(@NonNull final MotionEvent e) {
                        if ((e.getToolType(0) & MotionEvent.TOOL_TYPE_MOUSE) != 0) {
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N
                                    && PointerIcon.getSystemIcon(
                                            getContext(), PointerIcon.TYPE_NULL)
                                    .equals(Objects.requireNonNull(getActivity())
                                            .findViewById(R.id.screen).getPointerIcon())) {
                                return false;
                            }
                        }
                        if (mCanZoom) {
                            EmulationViewModel vm = Objects.requireNonNull(getActivity())
                                    .getViewModel();
                            boolean landscape = getActivity().isInLandscape();
                            vm.setZoomValue(landscape, !vm.getZoomValue(landscape));
                        }
                        changeToNextFitting();
                        return true;
                    }
                });
        setOnTouchListener((view, event) -> {
            if (!mGestureDetector.onTouchEvent(event)) {
                return performClick();
            }
            return true;
        });


    }

    private void changeToNextFitting() {
        if (getActivity() != null) {
            getActivity().restoreCanvasSize();
            getActivity().getCurrentUseropts().setValue(
                    Useropts.Scope.GLOBAL, "show_double_tap_hint", false);
        }
    }

    static final int FULLVISIBLE_OR_SCROLL = 1;
    static final int ZOOM_OUT = 2;
    private int mTvMode = FULLVISIBLE_OR_SCROLL;

    void setTvMode(final int mode) {
        if (mode != mTvMode) {
            mTvMode = mode;
            calcTransformation();
        }
    }
    private static final int PROGRAM_RELOAD_REQUIRED = -1;
    private int mProgram = 0;
    private int mTextureId = 0;

    private int loadShader(final int type, final String source) {
        int shader = GLES20.glCreateShader(type);
        try {
            InputStream is = getContext().getAssets().open(source);
            byte[] data = new byte[is.available()];
            if (is.read(data) >= 0) {
                String s = new String(data, StandardCharsets.UTF_8);
                is.close();
                GLES20.glShaderSource(shader, s);
                GLES20.glCompileShader(shader);
                int[] compiled = new int[1];
                GLES20.glGetShaderiv(shader, GLES20.GL_COMPILE_STATUS, compiled, 0);
                if (compiled[0] == 0) {
                    Log.e("ShaderUtils", "Shader compilation failed: "
                            + GLES20.glGetShaderInfoLog(shader));
                    GLES20.glDeleteShader(shader);
                    return 0;
                }

                return shader;

            } else {
                throw new RuntimeException("shader code is empty.");
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

    }
    private Runnable mGlCleanupRunnable = () -> { };
    private int createProgram(final String fragmentSource) {
        final int fragmentShader = loadShader(GLES20.GL_FRAGMENT_SHADER, fragmentSource);
        final int vertexShader = loadShader(GLES20.GL_VERTEX_SHADER, "shaders/vertex-shader.glsl");
        if (vertexShader == 0 || fragmentShader == 0) {
            Log.e("ShaderUtils", "Shader creation failed");
            return 0;
        }

        int program = GLES20.glCreateProgram();
        GLES20.glAttachShader(program, vertexShader);
        GLES20.glAttachShader(program, fragmentShader);
        GLES20.glLinkProgram(program);

        int[] linkStatus = new int[1];
        GLES20.glGetProgramiv(program, GLES20.GL_LINK_STATUS, linkStatus, 0);
        if (linkStatus[0] == 0) {
            Log.e("ShaderUtils", "Program linking failed: "
                    + GLES20.glGetProgramInfoLog(program));
            GLES20.glDeleteProgram(program);
            return 0;
        }
        try {
            InputStream is = getContext().getAssets().open(fragmentSource);
            byte[] data = new byte[is.available()];
            if (is.read(data) > 0) {
                String s = new String(data, StandardCharsets.UTF_8);
                mOpenGlParameters.clear();
                is.close();
                for (String line : s.split("\r?\n")) {
                    if (line.startsWith("#pragma parameter")) {
                        mOpenGlParameters.add(new OpenGlParameter(line));
                    }
                }
            }

        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        mGlCleanupRunnable = () -> {
            GLES20.glDetachShader(program, fragmentShader);
            GLES20.glDetachShader(program, vertexShader);
            GLES20.glDeleteShader(fragmentShader);
            GLES20.glDeleteShader(vertexShader);
            GLES20.glDeleteProgram(program);
        };
        return program;
    }
    private String findProgram(final String id) {
        try {
            String fallback = null;
            for (String file : Objects.requireNonNull(
                    getContext().getAssets().list("shaders"))) {
                if (file.matches("[0-9]{2}_fragment-.*\\.glsl")) {
                    if (fallback == null) {
                        fallback = "shaders/" + file;
                    }
                    InputStream is = getContext().getAssets().open("shaders/" + file);
                    byte[] data = new byte[is.available()];
                    if (is.read(data) > 0) {
                        String s = new String(data, StandardCharsets.UTF_8);
                        for (String line : s.split("\r?\n")) {
                            if (line.equals("#pragma id " + id)) {
                                return "shaders/" + file;
                            }
                        }
                    }
                    is.close();
                }
            }
            return fallback;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

    }


    @Override
    public void onSurfaceCreated(final GL10 gl, final EGLConfig eglConfig) {
        Log.v("crashdebug", "onSurfaceCreated");
        mStartTime = System.currentTimeMillis();
        loadProgram();
        int[] textureIds = new int[1];
        GLES20.glGenTextures(1, textureIds, 0);
        int textureId = textureIds[0];

        GLES20.glBindTexture(GLES20.GL_TEXTURE_2D, textureId);
        GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_MIN_FILTER,
                GLES20.GL_LINEAR);
        GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_MAG_FILTER,
                GLES20.GL_LINEAR);
        GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_WRAP_S,
                GLES20.GL_CLAMP_TO_EDGE);
        GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_WRAP_T,
                GLES20.GL_CLAMP_TO_EDGE);

        mTextureId = textureIds[0];

    }
    private int mLayoutBlockers = 0;
    void increaseLayoutBlockers() {
        mLayoutBlockers++;
        Log.v(TAG, "Redraw blockers increased to " + mLayoutBlockers + " from "
                + new Exception().getStackTrace()[1].toString());
    }
    void decreaseLayoutBlockers() {

        mLayoutBlockers--;
        Log.v(TAG, "Redraw blockers decreased to " + mLayoutBlockers + " from "
                + new Exception().getStackTrace()[1].toString());
        if (mLayoutBlockers == 0) {

            if (getActivity() != null && getActivity().getViewModel() != null
                    && getActivity().getViewModel().getEmulation() != null
            && getActivity().getViewModel().getEmulation().isPaused()) {
                Log.v(TAG, "Redraw required, emulation is paused");
                setCurrentBitmap(getActivity().getCurrentBitmap());
            } else {
                Log.v(TAG, "No redraw required, emulation will do it");
            }
        }
    }
    boolean hasLayoutBlockers() {
        return getHolder() == null || !getHolder().getSurface().isValid()
                && mLayoutBlockers != 0;
    }
    private String mSettingsAssetName = null;
    void useForSettings(final ShaderSettingsView.GlShader shader, final Bitmap bmp) {
        mLayoutBlockers = 0;
        mSettingsAssetName = shader.getAssetName();
        mApplyGlParameters = false;
        mProgram = PROGRAM_RELOAD_REQUIRED;
        setCurrentBitmap(bmp);
    }
    private void loadProgram() {
        if (mSettingsAssetName != null) {
            mProgram = createProgram("shaders/" + mSettingsAssetName);
        } else {
            String useroptsKey = getResources().getString(R.string.shader_key);
            String shaderId = Objects.requireNonNull(getActivity())
                    .getCurrentUseropts().getStringValue(useroptsKey, "");
            GLES20.glClearColor(0f, 0f, 0f, 1f);
            mProgram = createProgram(
                    findProgram(shaderId));
        }
    }

    @Override
    public void onSurfaceChanged(final GL10 gl, final int width, final int height) {
        increaseLayoutBlockers();
        Log.v("crashdebug", "onSurfaceChanged");
        GLES20.glViewport(0, 0, width, height);
        Log.v("crashdebug", "onSurfaceChanged done");
        post(this::decreaseLayoutBlockers);
    }
    private static final float[] VERTICES = {
            -1f, 1f, 0f, 0f, 0f,
            1f, 1f, 0f, 1f, 0f,
            -1f, -1f, 0f, 0f, 1f,
            1f, -1f, 0f, 1f, 1f
    };

    private final Point mBitmapSize = new Point(-1, -1);
    void drawQuad(final int program) {

        FloatBuffer buffer = ByteBuffer.allocateDirect(VERTICES.length * 4)
                .order(ByteOrder.nativeOrder())
                .asFloatBuffer();
        buffer.put(VERTICES).position(0);

        int aPosition = GLES20.glGetAttribLocation(program, "a_Position");
        int aTexCoord = GLES20.glGetAttribLocation(program, "a_TexCoord");

        buffer.position(0);
        // CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
        GLES20.glVertexAttribPointer(aPosition, 3, GLES20.GL_FLOAT, false, 5 * 4, buffer);
        GLES20.glEnableVertexAttribArray(aPosition);

        // CHECKSTYLE DISABLE MagicNumber FOR 2 LINES
        buffer.position(3);
        GLES20.glVertexAttribPointer(aTexCoord, 2, GLES20.GL_FLOAT, false, 5 * 4, buffer);
        GLES20.glEnableVertexAttribArray(aTexCoord);

        GLES20.glDrawArrays(GLES20.GL_TRIANGLE_STRIP, 0, 4);
    }
    static final class SettingParameterList extends HashMap<String, Float> {

    }
    void setSettingsGlProgram(final String assetname) {
        mSettingsAssetName = assetname;
    }
    void setSettingsGlParameters(final SettingParameterList glParameters) {
        mSettingsParameters.clear();
        mSettingsParameters.putAll(glParameters);


    }

    @Override
    public void requestRender() {
        if (mLayoutBlockers == 0) {
            super.requestRender();
        }
    }

    void setSettingsGlParameter(final String id, final float v) {
        mSettingsParameters.put(id, v);
        requestRender();
    }

    static class OpenGlParameter {
        private final Float mDefault;
        private final String mId;
        private final Float mMin;
        private final Float mMax;
        static final int MAX_INT_REPR_VALUE = 50;
        int intRepresentation(final float f) {
            return (int) ((MAX_INT_REPR_VALUE * f) / (mMax - mMin));
        }
        float floatRepresentation(final int x) {
            return mMin + ((mMax - mMin) * ((float) x / MAX_INT_REPR_VALUE));
        }
        String getId() {
            return mId;
        }
        float getDefault() {
            return mDefault;
        }
        private static final int POS_ID = 2;
        private static final int POS_DEFAULT = 3;
        private static final int POS_MIN = 4;
        private static final int POS_MAX = 5;
        OpenGlParameter(final String pragma) {
            String[] parts = pragma.split(" ");
            mId = parts[POS_ID];
            mDefault = Float.valueOf(parts[POS_DEFAULT]);
            mMin = Float.valueOf(parts[POS_MIN]);
            mMax = Float.valueOf(parts[POS_MAX]);

        }

        void apply(final int program, final int intRepresantion) {
            int loc = GLES20.glGetUniformLocation(program, mId);
            if (loc >= 0) {

                GLES20.glUniform1f(loc, floatRepresentation(intRepresantion));
            }
        }
    }
    private final List<OpenGlParameter> mOpenGlParameters = new LinkedList<>();
    private boolean mApplyGlParameters = true;


    @Override
    public void onDrawFrame(final GL10 gl) {
        GLES20.glClear(GLES20.GL_COLOR_BUFFER_BIT);
        if (mProgram == PROGRAM_RELOAD_REQUIRED) {
            mGlCleanupRunnable.run();
            loadProgram();
            mApplyGlParameters = true;
        }
        GLES20.glUseProgram(mProgram);
        if (mApplyGlParameters) {
            mApplyGlParameters = false;
            String useroptsKey = getResources().getString(R.string.shader_key);
            String optsPrefix;
            if (getActivity() != null) {
                String shaderId = Objects.requireNonNull(getActivity())
                        .getCurrentUseropts().getStringValue(useroptsKey, null);
                if (mUseropts.getBooleanValue(
                        "use_system_settings_" + shaderId, true)) {
                    optsPrefix = "global_";
                } else {
                    optsPrefix = "";
                }
                if (shaderId != null) {
                    for (OpenGlParameter p : mOpenGlParameters) {
                        String key = optsPrefix + shaderId + "_" + p.getId();
                        int val = mUseropts.getIntegerValue(key,
                                p.intRepresentation(p.getDefault()));
                        p.apply(mProgram, val);
                    }
                }
            }
        }
        if (!mSettingsParameters.isEmpty()) {
            for (String key : mSettingsParameters.keySet()) {
                int loc = GLES20.glGetUniformLocation(mProgram, key);
                Float value = mSettingsParameters.get(key);
                if (loc >= 0 && value != null) {
                    GLES20.glUniform1f(loc, value);
                }
            }
        }
        int timeHandle = GLES20.glGetUniformLocation(mProgram, "time");
        GLES20.glUniform1f(timeHandle, System.currentTimeMillis() - mStartTime);

        if (mVideoBitmap != null) {
            if (mBitmapSize.x != mVideoBitmap.getWidth()
                    || mBitmapSize.y != mVideoBitmap.getHeight()) {
                mBitmapSize.x = mVideoBitmap.getWidth();
                mBitmapSize.y = mVideoBitmap.getHeight();
                GLUtils.texImage2D(GL10.GL_TEXTURE_2D, 0, mVideoBitmap, 0);
            }
            int handle = GLES20.glGetUniformLocation(mProgram, "u_BitmapSize");
            if (handle > 0) {
                GLES20.glUniform2f(handle, mBitmapSize.x, mBitmapSize.y);
            }
            handle = GLES20.glGetUniformLocation(mProgram, "uTexelSize");
            if (handle > 0) {
                GLES20.glUniform2f(handle, 1f / mBitmapSize.x, 1f / mBitmapSize.y);
            }



            GLES20.glBindTexture(GLES20.GL_TEXTURE_2D, mTextureId);
            GLUtils.texSubImage2D(GLES20.GL_TEXTURE_2D, 0, 0, 0, mVideoBitmap);
        }
        drawQuad(mProgram);
    }

    private Bitmap mVideoBitmap = null;

    byte[] getLastRecordedBitmap() {
        if (mVideoBitmap != null) {
            Bitmap bmp = mVideoBitmap;
            int bytes = bmp.getAllocationByteCount();
            ByteBuffer buffer = ByteBuffer.allocate(bytes); //Create a new buffer
            bmp.copyPixelsToBuffer(buffer);
            return buffer.array();
        }
        return null;
    }

    static class StateCondition implements Condition {
        private final Condition mCondition;
        private boolean mWasSignalled = false;

        StateCondition(final Lock lock) {
            mCondition = lock.newCondition();
        }

        @Override
        public void await() throws InterruptedException {
            if (mWasSignalled) {
                return;
            }
            mCondition.await();
        }

        @Override
        public void awaitUninterruptibly() {
            if (mWasSignalled) {
                return;
            }
            mCondition.awaitUninterruptibly();
        }

        @Override
        public long awaitNanos(final long l) throws InterruptedException {
            if (mWasSignalled) {
                return l / 2;
            }
            return mCondition.awaitNanos(l);
        }

        @Override
        public boolean await(final long l, final TimeUnit timeUnit) throws InterruptedException {
            if (mWasSignalled) {
                return true;
            }
            return mCondition.await(l, timeUnit);
        }

        @Override
        public boolean awaitUntil(final Date date) throws InterruptedException {
            if (mWasSignalled) {
                return true;
            }
            return mCondition.awaitUntil(date);
        }

        @Override
        public void signal() {
            mWasSignalled = true;
            mCondition.signal();
        }

        @Override
        public void signalAll() {
            mWasSignalled = true;
            mCondition.signalAll();
        }
    }

    private static final int MINIMAL_WIDTH = 32;
    private ByteBuffer mWorkBuffer = ByteBuffer.allocate(1);

    void setCurrentBitmap(final Bitmap value) {
        if (value != null) {

            if (!hasLayoutBlockers()) {
                boolean waitingForScreenUpdate;
                synchronized (mScreenTestCondition) {
                    waitingForScreenUpdate = !mScreenTestCondition.isEmpty();
                }

                if (waitingForScreenUpdate) {
                    Log.v(TAG, "waitingForScreenUpdate");
                    mScreenTestLock.lock();
                    List<Condition> conditions;
                    synchronized (mScreenTestCondition) {
                        conditions = new LinkedList<>(mScreenTestCondition);
                    }
                    for (Condition c : conditions) {
                        c.signalAll();
                    }
                    mScreenTestCondition.clear();
                    mScreenTestLock.unlock();
                }

                if (mWaitingForBitmap != null) {
                    Log.v(TAG, "mWaitingForBitmap");
                    int bytes = value.getAllocationByteCount();
                    mWorkBuffer.clear();
                    if (mWorkBuffer.remaining() != bytes) {
                        mWorkBuffer = ByteBuffer.allocate(bytes); //Create a new buffer
                    }
                    value.copyPixelsToBuffer(mWorkBuffer);
                    if (getWidth() > MINIMAL_WIDTH) {
                        if (Arrays.equals(mWorkBuffer.array(), mWaitingForBitmap)) {
                            mBitmapTestLock.lock();
                            mWaitingForBitmap = null;
                            mBitmapCondition.signalAll();
                            mBitmapTestLock.unlock();
                        }
                    }
                }
                requestRender();
                if (mScrollview != null) {
                    boolean waitingForScrollUpdate;
                    synchronized (mScrollTestCondition) {
                        waitingForScrollUpdate = !mScrollTestCondition.isEmpty();
                    }

                    if (waitingForScrollUpdate) {
                        Log.v(TAG, "waitingForScrollUpdate");
                        List<ScrollTest> toDelete = new LinkedList<>();
                        int mTop = mScrollview.getScrollY();
                        int mBottom = getBottom()
                                - (mScrollview.getHeight() + mScrollview.getScrollY());
                        mScrollTestLock.lock();
                        for (ScrollTest test : mScrollTestCondition.keySet()) {
                            if (test.match(mTop, mBottom)) {
                                Condition c = Objects.requireNonNull(
                                        mScrollTestCondition.get(test));
                                c.signalAll();
                                toDelete.add(test);
                            }
                        }
                        mScrollTestLock.unlock();
                        synchronized (mScrollTestCondition) {
                            for (ScrollTest test : toDelete) {
                                mScrollTestCondition.remove(test);
                            }
                        }
                    }
                }
            }
            mVideoBitmap = value;
        }
    }

    void onGeometryChanged() {
        if (getActivity() != null) {
            getActivity().restoreCanvasSize();
        }
        requestRender();
    }

    void setCanvasGeometry(final int canvasWidth,
                           final int canvasHeight,
                           final float pixeAspectRatio,
                           final float canvasAspectRatio) {
        mCanvasWidth = canvasWidth;
        mCanvasAspectRatio = canvasAspectRatio;
        mPixelAspectRatio = pixeAspectRatio;
        Log.v(getClass().getSimpleName(), String.format("setCanvasGeometry (%d, %d, %f, %f)",
                canvasWidth, canvasHeight, pixeAspectRatio, canvasAspectRatio));
        calcTransformation();

    }

    void setBitmap(final Bitmap bitmap) {
        setCurrentBitmap(bitmap);
    }

    Bitmap getCurrentBitmap() {
        return mVideoBitmap;
    }

    void setUseropts(final Useropts useropts) {
        //mZoomInitialized = viewmodel.getZoomInitialized().get(orientation);
        mProgram = PROGRAM_RELOAD_REQUIRED;
        mUseropts = useropts;



    }


    private boolean mCanZoom = false;


    private final DisplayMetrics mDisplayMetrics = new DisplayMetrics();

    private float getDeviceSpecificScreenAspectRatio(final float screenAspectRatio) {
        @SuppressWarnings("ConstantConditions")
        Display display = getActivity().getWindowManager().getDefaultDisplay();

        display.getMetrics(mDisplayMetrics);
        return screenAspectRatio * (mDisplayMetrics.ydpi / mDisplayMetrics.xdpi);

    }
    void calcTransformation() {
        //noinspection ConstantConditions
        if (getActivity() != null && getActivity().getViewModel() != null) {
            final int displayId = getActivity().getViewModel().getCurrentDisplayId();
            Objects.requireNonNull(getActivity())
                    .getViewModel().setPixelAspectRatio(
                            displayId, mPixelAspectRatio);
            getActivity().getViewModel().setScreenAspectRatio(displayId, mCanvasAspectRatio);
            if (mCanvasWidth != 0 && mCanvasAspectRatio != 0 && mPixelAspectRatio != 0) {
                Display display = getActivity().getWindowManager().getDefaultDisplay();
                Point displaySize = new Point();
                display.getSize(displaySize);
                float width = displaySize.x;
                float scaleX;
                float scaleY;
                float ratio = getDeviceSpecificScreenAspectRatio(mCanvasAspectRatio);
                int addheight;
                boolean zoomValue = getActivity().getViewModel()
                        .getZoomValue(getActivity().isInLandscape());
                if (zoomValue && mTvMode != ZOOM_OUT) {
                    addheight = 0;
                    scaleX = displaySize.x / width;
                    scaleY = displaySize.x / getDeviceSpecificScreenAspectRatio(mCanvasAspectRatio)
                            / mViewHeight;
                } else {
                    int height = mViewHeight;
                    View v = getActivity().findViewById(R.id.tvkeyboardspacer);
                    if (v.getVisibility() == VISIBLE && mTvMode != ZOOM_OUT) {
                        addheight = v.getHeight();
                    } else {
                        addheight = 0;
                    }
                    height += addheight;
                    if (width > height * ratio) {
                        scaleY = 1.0f;
                        scaleX = height * ratio / width;
                    } else {
                        scaleX = displaySize.x / width;
                        scaleY = displaySize.x / (ratio * height);

                    }
                }
                mCanZoom = width > mViewHeight * ratio;
                int newwidth = (int) (scaleX * width);
                int newheight = (int) (scaleY * mViewHeight);
                ViewGroup.LayoutParams lp = getLayoutParams();
                if (lp.width != newwidth || lp.height != newheight + addheight) {
                    //firstRun = false;
                    lp.width = newwidth;
                    lp.height = newheight + addheight;
                    Log.v(TAG, "width = " + width);
                    Log.v(TAG, "scalex = " + scaleX);
                    Log.v(TAG, "height = " + mViewHeight);
                    Log.v(TAG, "scaley = " + scaleY);
                    Log.v(TAG, "layout.width = " + lp.width);
                    Log.v(TAG, "layout.height = " + lp.height);
                    increaseLayoutBlockers();
                    setLayoutParams(lp);

                    Log.v("crashdetect", "glview update required");
                    addOnLayoutChangeListener(new OnLayoutChangeListener() {
                        //CHECKSTYLE DISABLE ParameterNumber FOR 2 LINES
                        @Override
                        public void onLayoutChange(final View view, final int l, final int t,
                                                   final int r, final int b,
                                                   final int ot, final int ol,
                                                   final int or, final int ob) {
                            Log.v("crashdetect", "scrollview relayouted");
                            removeOnLayoutChangeListener(this);
                            Log.v("crashdetect", "mInLayout reset to false from "
                                    + new Exception().getStackTrace()[1].toString());

                            decreaseLayoutBlockers();
                        }
                    });
                    requestLayout();
                }
            }
        }
        Log.v(TAG, "calcTransformation finished");
    }

    private final Lock mScreenTestLock = new ReentrantLock();
    /**
     * @noinspection MismatchedQueryAndUpdateOfCollection
     */
    private final List<Condition> mScreenTestCondition = new LinkedList<>();
    private final Lock mAudioTestLock = new ReentrantLock();
    private final Condition mAudioTestCondition = mAudioTestLock.newCondition();

    /**
     * Testing: Wait for the screen to be updated.
     *
     * @param t wait for how many time units given in u.
     * @param u unit of time to wait for.
     * @return true if the screen has been redrawn within the given time.
     */
    @SuppressWarnings("SameParameterValue")
    public boolean waitForScreenRedraw(final long t, final TimeUnit u) {
        boolean ret = false;
        try {
            mScreenTestLock.lock();
            Log.v(getClass().getSimpleName() + "waitForScreenRedraw", "await called");
            Condition c = new StateCondition(mScreenTestLock);
            synchronized (mScreenTestCondition) {
                mScreenTestCondition.add(c);
            }
            ret = c.await(t, u);
            Log.v(getClass().getSimpleName() + "waitForScreenRedraw",
                    "await returned " + (ret ? "true" : "false"));

        } catch (InterruptedException e) {
            // nothing
        } finally {
            mScreenTestLock.unlock();
        }
        if (!ret) {
            Log.v("x", "y");
        }
        return ret;


    }

    private interface ScrollTest {
        boolean match(int top, int bottom);
    }

    private final Lock mScrollTestLock = new ReentrantLock();
    private final Map<ScrollTest, Condition> mScrollTestCondition = new HashMap<>();

    boolean waitForScroll(final int position, final long t, final TimeUnit u) {

        if (mScrollview != null) {
            int mTop = mScrollview.getScrollY();
            int mBottom = getHeight() - mScrollview.getHeight() + mScrollview.getScrollY();

            ScrollTest test;

            switch (position) {
                case EmulationUi.SCREENPART_TOP:
                    test = (top, bottom) -> top == 0;
                    break;
                case EmulationUi.SCREENPART_BOTTOM:
                    test = (top, bottom) -> bottom == 0;
                    break;
                case EmulationUi.SCREENPART_CENTER:
                    test = (top, bottom) -> Math.abs(top - bottom) < 2;
                    break;
                default:
                    return false;
            }
            if (test.match(mTop, mBottom)) {
                return true;
            } else {
                Condition condition = mScrollTestLock.newCondition();
                mScrollTestCondition.put(test, condition);
                try {
                    mScrollTestLock.lock();
                    return condition.await(t, u);

                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                } finally {
                    mScrollTestLock.unlock();

                }
            }
        }
        return false;
    }

    /**
     * Testing: Wait for the screen to be first drawn.
     *
     * @param t wait for how many time units given in u.
     * @param u unit of time to wait for.
     * @return true if the screen has been drawn within the given time.
     */
    @SuppressWarnings("SameParameterValue")
    public boolean waitForScreenInit(final long t, final TimeUnit u) {
        if (mVideoBitmap != null) {
            return true;
        } else {
            return waitForScreenRedraw(t, u);
        }
    }

    private final Lock mBitmapTestLock = new ReentrantLock();
    private final Condition mBitmapCondition = mBitmapTestLock.newCondition();
    private byte[] mWaitingForBitmap = null;

    /**
     * Testing: Wait for the screen to match a specific value.
     *
     * @param t    wait for how many time units given in u.
     * @param u    unit of time to wait for.
     * @param data data to match, to be determined with {@link Bitmap#copyPixelsToBuffer(Buffer)}
     * @return true if the screen's pixels match the given data within the given time.
     */
    @SuppressWarnings("SameParameterValue")
    public boolean waitForBitmap(final byte[] data, final long t, final TimeUnit u) {
        if (mWaitingForBitmap != null) {
            return false;
        }

        try {
            mBitmapTestLock.lock();
            mWaitingForBitmap = data;
            boolean ret = mBitmapCondition.await(t, u);
            mWaitingForBitmap = null;
            return ret;
        } catch (InterruptedException e) {
            // nothing
        } finally {
            mBitmapTestLock.unlock();
        }
        return false;
    }

    @Override
    public boolean waitForAudio(final long t, final TimeUnit u) {
        //noinspection ConstantConditions

        Emulation.SoundFunctions sf = getActivity().getViewModel()
                .getEmulation().getSoundFunctions();
        if (sf != null) {
            boolean ret;
            mAudioTestLock.lock();
            sf.callOnAudioPlaying(() -> {
                mAudioTestLock.lock();
                mAudioTestCondition.signalAll();
                mAudioTestLock.unlock();
            });

            try {
                ret = mAudioTestCondition.await(t, u);
            } catch (InterruptedException e) {
                ret = false;
            } finally {
                mAudioTestLock.unlock();
            }
            return ret;
        }
        return false;
    }
}
