//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.View;
import android.widget.ArrayAdapter;
import android.widget.PopupWindow;
import android.widget.Spinner;

import androidx.annotation.NonNull;

import java.util.LinkedList;
import java.util.List;

/**
 * A Spinner that can interact with other Views in an {@link SettingsActivity}.
 */
public final class MappedSpinner extends androidx.appcompat.widget.AppCompatSpinner {
    /**
     * Simple constructor to use when creating a MappedSpinner from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public MappedSpinner(final Context context) {
        super(context);
        init(null, 0);
    }
    /**
     * Constructor that is called when inflating a MappedSpinner from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public MappedSpinner(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init(attrs, 0);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public MappedSpinner(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
        init(attrs, defStyle);
    }

    static void showGamepadDropdown(final Spinner spinner) {
        List<Emulation.MenuFeature> data = new LinkedList<>();
        for (int i = 0; i < spinner.getAdapter().getCount(); i++) {
            final int pos = i;
            data.add(new BaseActivity.PopupCallback(spinner.getItemAtPosition(pos).toString(),
                    NO_ID, () -> spinner.setSelection(pos), false, false));
        }
        PopupWindow pw = BaseActivity.createPopup(spinner, data);
        spinner.measure(MeasureSpec.UNSPECIFIED, MeasureSpec.UNSPECIFIED);
        pw.setWidth(spinner.getWidth());
        BaseActivity.showPopup(pw, spinner);

    }

    interface MappedSpinnerElement extends Emulation.DataEntry {
        String getId();
    }

    String getSelection() {
        int position = getSelectedItemPosition();
        if (position >= 0) {
            return ((MappedSpinnerElement) getItemAtPosition(position)).getId();
        }
        return null;
    }

    @Override
    public void setSelection(final int position) {
        super.setSelection(position);
    }

    /**
     * Set selection to the element with the given id, see {@link MappedSpinnerElement#getId()}.
     * @param id id of the element to be set as active.
     */
    public void setSelection(final String id) {
        int foundId = NO_ID;
        int foundText = NO_ID;
        for (int i = 0; i < getCount(); i++) {
            if (((MappedSpinnerElement) getItemAtPosition(i)).getId().equals(id)) {
                foundId = i;
            }
            if (((MappedSpinnerElement) getItemAtPosition(i)).getText().equals(id)) {
                foundText = i;
            }

        }
        if (foundId != NO_ID) {
            setSelection(foundId);
        } else {
            setSelection(foundText);
        }

    }

    private String mDefaultkey = null;

    /**
     * ID of the element that is visible if there is no saved user option.
     * @return id
     */
    public String getDefaultElementId() {
        return mDefaultkey;
    }

    /**
     * Populate view with given elements.
     * @param elements List of
     *                 {@link de.rainerhock.eightbitwonders.EmulationUi.SettingSpinnerElement}
     *                 to be made available.
     */
    public void populate(final List<EmulationUi.SettingSpinnerElement> elements) {
        List<EmulationUi.SettingSpinnerElement> values = new LinkedList<>();
        for (EmulationUi.SettingSpinnerElement entry : elements) {
            values.add(new EmulationUi.SettingSpinnerElement() {
                @Override
                public String getId() {
                    return entry.getId();
                }

                @Override
                public String getText() {
                    return entry.getText();
                }

                @NonNull
                @Override
                public String toString() {
                    return getText();
                }
            });
        }
        ArrayAdapter<EmulationUi.SettingSpinnerElement> adapter =
                new ArrayAdapter<>(getContext(),
                        android.R.layout.simple_spinner_item, values);
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        setAdapter(adapter);
    }
    private void init(final AttributeSet attrs, final int defStyle) {
        // Load attributes
        //noinspection resource
        final TypedArray a = getContext().obtainStyledAttributes(
                attrs, R.styleable.MappedSpinner, defStyle, 0);
        int keyId = a.getResourceId(R.styleable.MappedSpinner_keyarray, NO_ID);
        int textId = a.getResourceId(R.styleable.MappedSpinner_textarray, NO_ID);
        String defaultkey = a.getString(R.styleable.MappedSpinner_defaultkey);
        List<MappedSpinnerElement> values = new LinkedList<>();
        if (keyId != View.NO_ID && textId != View.NO_ID) {
            String[] texts = getResources().getStringArray(textId);
            String[] keys = getResources().getStringArray(keyId);
            if (texts.length == keys.length) {
                for (int pos = 0; pos < texts.length; pos++) {
                    int finalPos = pos;
                    values.add(new MappedSpinnerElement() {
                        @Override
                        public String getText() {
                            return toString();
                        }

                        @Override
                        public String getId() {
                            return keys[finalPos];
                        }

                        @NonNull
                        @Override
                        public String toString() {
                            return texts[finalPos];
                        }
                    });
                    if (defaultkey != null) {
                        if (keys[finalPos].equals(defaultkey)) {
                            mDefaultkey = keys[finalPos];
                        }
                    }
                }
            }
        }
        String assetfolder = a.getString(R.styleable.MappedSpinner_assetfolder);
        String assetpattern = a.getString(R.styleable.MappedSpinner_assetpattern);
        if (assetfolder != null && assetpattern != null) {
            //noinspection RegExpRedundantEscape
            for (String f : assetfolder.split("\\:")) {
                for (String s : BaseActivity.getAssetsFromFolder(this, f)) {
                    if (s.matches(assetpattern)) {
                        values.add(new MappedSpinnerElement() {

                            @Override
                            public String getText() {
                                return toString();
                            }

                            @Override
                            public String getId() {
                                return s;

                            }

                            @NonNull
                            @Override
                            public String toString() {
                                //noinspection RegExpRedundantEscape
                                return s.split("\\.(?=[^\\.]+$)")[0];
                            }
                        });
                    }
                }
            }
        }
        ArrayAdapter<MappedSpinnerElement> adapter =
                new ArrayAdapter<>(getContext(), android.R.layout.simple_spinner_item, values);
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        setAdapter(adapter);
        setOnKeyListener((view, keycode, keyEvent) -> {
            if (keyEvent.getDevice().hasKeys(keyEvent.getKeyCode())[0]) {
                return false;
            }
            if (EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keycode)
                    && keyEvent.getAction() == KeyEvent.ACTION_UP) {
                showGamepadDropdown(this);
                return true;
            }
            return false;
        });
        a.recycle();
    }
}
