package de.rainerhock.eightbitwonders;

import static android.graphics.Bitmap.CompressFormat.PNG;
import static de.rainerhock.eightbitwonders.BaseActivity.remapKeyEvent;
import static de.rainerhock.eightbitwonders.ConfigurationFactory.FULLQUALITY;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.ADD_TO_START;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.PAUSE;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.TIMEMACHINE;
import static de.rainerhock.eightbitwonders.EmulationUi.JOYSTICK_BACK_BUTTONS;
import static de.rainerhock.eightbitwonders.EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.widget.Button;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.Toast;

import androidx.annotation.IdRes;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.core.content.res.ResourcesCompat;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.Serializable;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Set;

public final class MainMenuView extends RelativeLayout {
    private InputDevice mLastInputDevice = null;
    private void connectButton(final @IdRes int id,
                               final Runnable callback) {
        final View bn = findViewById(id);
        if (callback != null) {
            bn.setOnKeyListener((v, keyCode, event) -> {
                if (event.getAction() == KeyEvent.ACTION_UP
                        && JOYSTICK_BACK_BUTTONS.contains(keyCode)) {
                    post(this::dismiss);
                }
                mLastInputDevice = event.getDevice();
                if (event.getAction() == KeyEvent.ACTION_UP) {
                    return handleKeyUp(bn, remapKeyEvent(event)
                            .getKeyCode());
                }
                if (event.getAction() == KeyEvent.ACTION_DOWN) {
                    return handleKeyDown(bn, remapKeyEvent(event)
                            .getKeyCode());
                }

                return false;
            });
            bn.setOnClickListener(v -> onClick(v, callback));
            setButtonsDialogtop(bn);
            bn.setVisibility(View.VISIBLE);

        }
    }

    void setButtonsDialogtop(final View button) {
        button.addOnLayoutChangeListener(new OnLayoutChangeListener() {
            @Override
            // CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
            public void onLayoutChange(final View v, final int left, final int top,
                                       final int right, final int bottom,
                                       final int oldLeft, final int oldTop,
                                       final int oldRight, final int oldBottom) {
                int[] pos = new int[2];
                v.getLocationOnScreen(pos);
                DisplayMetrics dm = new DisplayMetrics();
                getActivity().getWindowManager().getDefaultDisplay().getMetrics(dm);
                button.removeOnLayoutChangeListener(this);
            }
        });
    }
    private void connectButton(final int bnId, final int containerId,
                               final List<BaseActivity.PopupCallback> functions,
                               final MenuState state) {
        LayoutInflater inflater = (LayoutInflater) getContext()
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        ViewGroup container = findViewById(containerId);
        createButtons(inflater, container, functions);
        View button = findViewById(bnId);
        button.setVisibility(View.VISIBLE);
        setButtonStyleToParent(state, button, container);
        setButtonsDialogtop(button);
        container.setVisibility(state.isOpen(bnId) ? View.VISIBLE : View.GONE);

    }

    private void setButtonStyleToParent(final MenuState state, final View button,
                                        final ViewGroup container) {
        Drawable closed =  ResourcesCompat.getDrawable(getResources(),
                R.drawable.ic_chevron_down, getContext().getTheme());

        Drawable open = ResourcesCompat.getDrawable(getResources(),
                R.drawable.ic_chevron_up, getContext().getTheme());
        if (button instanceof DialogsView.MenuButton) {
            final DialogsView.MenuButton bn = (DialogsView.MenuButton) button;
            Drawable[] d = bn.getCompoundDrawables();
            bn.setCompoundDrawablesWithIntrinsicBounds(d[0], d[1], closed, d[3]);
            button.setOnClickListener(v -> {
                if (container.getVisibility() == View.GONE) {
                    container.setVisibility(View.VISIBLE);
                    bn.setCompoundDrawablesWithIntrinsicBounds(d[0], d[1], open, d[3]);
                } else {
                    container.setVisibility(View.GONE);
                    bn.setCompoundDrawablesWithIntrinsicBounds(d[0], d[1], closed, d[3]);
                }
            });
        } else {

            ImageButton ib;
            if (button instanceof ImageButton) {
                ib = (ImageButton) button;
                ib.setImageDrawable(closed);
            } else {
                ib = null;
            }
            button.setOnClickListener(v -> {
                if (container.getVisibility() == View.GONE) {
                    container.setVisibility(View.VISIBLE);
                    state.setOpen(button.getId(), true);
                    if (ib != null) {
                        ib.setImageDrawable(open);
                    }

                } else {
                    container.setVisibility(View.GONE);
                    state.setOpen(button.getId(), false);
                    if (ib != null) {
                        ib.setImageDrawable(closed);
                    }

                }
            });


        }
    }

    private void createButtons(final LayoutInflater inflater,
                               final ViewGroup container,
                               final List<BaseActivity.PopupCallback> functions) {
        for (BaseActivity.PopupCallback f : functions) {
            DialogsView.MenuButton bn = inflater.inflate(
                            R.layout.view_emulation_popup_menubutton, container)
                    .findViewById(R.id.bn_text);
            bn.setText(f.getName());
            bn.setId(f.getId());
            if (f.isRemoteOnly()) {
                if (bn.getTag() != null) {
                    bn.setTag(bn.getTag().toString() + "|NO_GAMEPAD");
                } else {
                    bn.setTag("NO_GAMEPAD");
                }
            }
            if (f.getIconResource() > 0) {
                Drawable[] d = bn.getCompoundDrawables();
                bn.setCompoundDrawablesWithIntrinsicBounds(
                        ResourcesCompat.getDrawable(bn.getResources(), f.getIconResource(),
                                bn.getContext().getTheme()),
                        d[1], d[2], d[3]);

            }
            bn.setOnKeyListener((v, keyCode, event) -> {
                mLastInputDevice = event.getDevice();
                return false;
            });
            setButtonsDialogtop(bn);
            if (f.getRunnable() != null) {
                bn.setOnClickListener(v -> onClick(v, f.getRunnable()));
            } else {
                bn.setEnabled(false);
                bn.setFocusable(false);
            }
        }
    }

    boolean handleKeyUp(final View view, final int keyCode) {
        if (keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                || keyCode == KeyEvent.KEYCODE_SPACE
                || keyCode == KeyEvent.KEYCODE_ENTER
                || JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keyCode)) {
            return view.callOnClick();
        }
        return false; // keyCode != KeyEvent.KEYCODE_BACK && keyCode != KeyEvent.KEYCODE_ESCAPE;
    }
    private boolean requestNavigation(final View v) {
        if (v != null && !(v instanceof ViewGroup)) {
            v.requestFocus();
        }
        return true;
    }
    boolean handleKeyDown(final View view, final int keyCode) {
        if (keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
            return requestNavigation(view.focusSearch(View.FOCUS_DOWN));
        } else if (keyCode == KeyEvent.KEYCODE_DPAD_UP) {
            return requestNavigation(view.focusSearch(View.FOCUS_UP));
        } else if (keyCode == KeyEvent.KEYCODE_DPAD_RIGHT) {
            return requestNavigation(view.focusSearch(View.FOCUS_RIGHT));
        } else if (keyCode == KeyEvent.KEYCODE_DPAD_LEFT) {
            return requestNavigation(view.focusSearch(View.FOCUS_LEFT));
        }
        return keyCode != KeyEvent.KEYCODE_BACK && keyCode != KeyEvent.KEYCODE_ESCAPE;
    }

    private static final int[] DPAD_KEYS = new int[]{
            KeyEvent.KEYCODE_DPAD_CENTER, KeyEvent.KEYCODE_DPAD_DOWN,
            KeyEvent.KEYCODE_DPAD_LEFT, KeyEvent.KEYCODE_DPAD_UP,
            KeyEvent.KEYCODE_DPAD_RIGHT, KeyEvent.KEYCODE_BACK};

    private boolean remoteControlisCompatible(final View v, final InputDevice dev) {
        final String tag;
        if (v.getTag() != null) {
            tag = String.format("|%s|", v.getTag().toString());
        } else {
            tag = "";
        }
        if (tag.contains("|NO_GAMEPAD|")) {
            return dev.hasKeys(DPAD_KEYS).length == DPAD_KEYS.length;
        } else {
            return true;
        }
    }
    private void runAfterClick() {
        dismiss();
    }
    private void onClick(final View v, final Runnable action) {
        if (mLastInputDevice == null || remoteControlisCompatible(v, mLastInputDevice)) {
            action.run();
            runAfterClick();
            mLastInputDevice = null;
        } else {
            Toast.makeText(v.getContext(), R.string.use_remote_control, Toast.LENGTH_LONG)
                    .show();

        }
    }

    EmulationActivity getActivity() {
        Activity activity;
        try {
            activity = FragmentManager.findFragment(this).getActivity();
            if (activity instanceof EmulationActivity) {
                return (EmulationActivity) activity;
            }
        } catch (IllegalStateException e) {
            Context context = getContext();
            while (context instanceof ContextWrapper) {
                if (context instanceof EmulationActivity) {
                    return (EmulationActivity) context;
                }
                context = ((ContextWrapper) context).getBaseContext();
            }
        }
        throw new IllegalStateException(MainMenuView.class.getSimpleName()
                + " must be in an " + EmulationActivity.class.getSimpleName());

    }

    private void setKeyboardVisible(final boolean value) {
        EmulationActivity ea = getActivity();
        int visibility = value ? View.VISIBLE : View.GONE;
        ea.getViewModel().setKeyboardVisible(ea.isInLandscape(), visibility);
        ea.onKeyboardVisibilityChanged(visibility, value && ea.isTv() ? mLastInputDevice : null);
    }
    private void sendToBackground() {
        EmulationActivity ea = getActivity();
        ea.getEmulationUi().setScreenRefresh(false);
        ea.getViewModel().setRunningInBackground(true);
        ea.getDialogsController().showUiElement(DialogsView.UiElement.BACKGROUND_MODE);
        ea.getViewModel().getEmulation().setPaused(false);
    }
    private void share(final Emulation emu, final EmulationConfiguration conf) {
        final Bitmap bmp = getActivity().getCurrentBitmap();
        Runnable r = emu.getPackCurrentStateFunctions().getDumpContentsRunnable(0, conf);
        Intent i = ConfigurationFactory.prepareCreatePackageIntent(
                getActivity(), emu, conf, r, bmp);
        getActivity().shareContent(i, false);

    }
    private static final int ID_BN_DETACH = View.generateViewId();
    private void addToStartScreen() {
        Bundle b = new Bundle();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        final Bitmap bmp = getActivity().getCurrentBitmap();
        if (bmp != null) {
            bmp.compress(PNG, FULLQUALITY, baos);
            b.putByteArray("pixeldata", baos.toByteArray());
        }
        getActivity().getDialogsController().showUiElement(ADD_TO_START);


    }
    private void showSubMenu(final LayoutInflater inflater, final DialogsView.MenuButton b,
                             final MenuState state,
                             final List<BaseActivity.PopupCallback> functions) {
        if (functions.isEmpty()) {
            b.setEnabled(false);
            b.setFocusable(false);
        } else {
            if (b != null) {
                ViewParent p = b.getParent();
                if (p instanceof ViewGroup) {
                    View child = ((ViewGroup) p).findViewById(R.id.ll_sublist);
                    if (child instanceof ViewGroup) {
                        createButtons(inflater, (ViewGroup) child, functions);
                        setButtonStyleToParent(state, b, (ViewGroup) child);
                    }
                }
            }
        }
    }
    // CHECKSTYLE DISABLE MethodLength FOR 1 LINES
    void populate() {
        removeAllViews();
        LayoutInflater inflater = (LayoutInflater) getContext().getSystemService(
                Context.LAYOUT_INFLATER_SERVICE);
        inflater.inflate(R.layout.viewgroup_emulation_popupmenu, this);
        EmulationActivity activity = getActivity();
        final MenuState state = activity.getViewModel().getMenuState() != null
                ? activity.getViewModel().getMenuState() : new MenuState();
        activity.getViewModel().setMenuState(state);
        DialogsView containerview = getActivity().getDialogsController();
        Emulation emu = activity.getViewModel().getEmulation();
        EmulationConfiguration conf = activity.getViewModel().getConfiguration();
        Set<String> softkeyIds = activity.getViewModel().getSoftkeyIds();
        SoftkeyConfigView.SoftkeyList uiSoftkeys = activity.getViewModel().getUiSoftkeys();

        if (emu.getTimeMachineFunctions() != null) {
            connectButton(R.id.bn_timemachine,
                    () -> containerview.showUiElement(TIMEMACHINE, false));
        }
        if (emu.getPackCurrentStateFunctions() != null) {
            connectButton(R.id.bn_savestate, activity::showSnapshotsDialog);
        }
        List<BaseActivity.PopupCallback> shareFunctions = new LinkedList<>();
        if (conf.isBareMachine()) {
            shareFunctions.add(
                    new BaseActivity.PopupCallback(getContext(), R.string.add_to_main_activity,
                            0, this::addToStartScreen, true, false));
        }
        if (conf.getLocalPath() != null) {
            shareFunctions.add(
                new BaseActivity.PopupCallback(getContext(), R.string.use_on_main_activity,
                    0, () -> ConfigurationFactory.updateBitmap(conf,
                        getActivity().getCurrentBitmap()), true, false));
        }
        if (conf.getLocalPath() != null && conf.isStream()
                && emu.getInitialSnapshotStorer() != null) {
            shareFunctions.add(
                    new BaseActivity.PopupCallback(getContext(), R.string.start_from_here,
                            0, emu.getInitialSnapshotStorer(),  true, false));
        }

        if (!conf.isStream()  && emu.getPackCurrentStateFunctions() != null) {
            shareFunctions.add(
                    new BaseActivity.PopupCallback(getContext(), R.string.share_and_save,
                            0, () -> share(emu, conf),  true, false));
        }
        shareFunctions.add(
                new BaseActivity.PopupCallback(getContext(), R.string.share_screenshot,
                        0, () -> {
                    try {
                        BaseActivity.exportScreenshot(getActivity(),
                                getActivity().getCurrentBitmap(), false);
                    } catch (FileNotFoundException e) {
                        // should not happen;
                    }
                },  true, false));

        connectButton(R.id.bn_share_current_state, R.id.ll_share_current_state,
                shareFunctions,
                state);
        if (emu.getFileFunctions() != null) {
            connectButton(R.id.bn_storage_access, activity::launchOpenFileActivity);
        }
        connectButton(R.id.bn_settings, activity::openSettingsDialog);
        setOnClickListener(v -> dismiss());

        if (emu.getResetFunctions() != null && emu.getResetFunctions().getFunctions() != null) {
            List<BaseActivity.PopupCallback> resetFunctions = new LinkedList<>();
            for (Emulation.MenuFeature f : emu.getResetFunctions().getFunctions()) {
                resetFunctions.add(new BaseActivity.PopupCallback(f.getName(), 0,
                        f.getRunnable(), false, false));
            }
            if (!resetFunctions.isEmpty()) {
                connectButton(R.id.bn_reset, R.id.ll_reset,
                        resetFunctions,
                        state);
            }
        }
        List<BaseActivity.PopupCallback> extendedFileFunctions = new LinkedList<>();
        Runnable runCreate;
        if (emu.getFileCreationFunction() != null) {
            runCreate = activity::openCreateFilesDialog;
        } else {
            runCreate = null;
        }
        extendedFileFunctions.add(new BaseActivity.PopupCallback(getContext(),
                R.string.create_and_attach_image, R.drawable.ic_menu_add,
                runCreate, true, true));
        List<BaseActivity.PopupCallback> detach = new LinkedList<>();
        if (emu.getFileFunctions() != null
                && emu.getFileFunctions().getDetachFunctions() != null
                && !emu.getFileFunctions().getDetachFunctions().getFunctions().isEmpty()) {
            for (Emulation.MenuFeature f : emu.getFileFunctions().getDetachFunctions()
                    .getFunctions()) {
                detach.add(new BaseActivity.PopupCallback(f.getName(), 0, f.getRunnable(),
                        false, false));
            }
        }
        BaseActivity.PopupCallback cb = new BaseActivity.PopupCallback(getContext(),
                R.string.detach_file, R.drawable.ic_menu_clear_playlist, () -> { },
                true, false);
        cb.setId(ID_BN_DETACH);
        extendedFileFunctions.add(cb);
        Runnable runTapeFunctions;
        if (emu.getTapeDeviceFunctions() != null
                && emu.getTapeDeviceFunctions().isAvailableNow()) {
            runTapeFunctions =  () -> containerview.showUiElement(
                    DialogsView.UiElement.TAPE_CONTROL, false);
        } else {
            runTapeFunctions = null;
        }
        extendedFileFunctions.add(new BaseActivity.PopupCallback(getContext(),
                R.string.IDS_MP_DATASETTE_CONTROL, R.drawable.ic_cassette,
               runTapeFunctions,
                true, false));
        Runnable runFliplist = null;
        if (conf.getFliplist() != null) {
            Set<Uri> fl = conf.getFliplist();
            Emulation.FliplistFunctions ff = emu.getFliplistFunctions(fl);
            if (ff != null) {
                if (fl.isEmpty()) {
                    fl = ff.getFilesInList();
                }
                if (!fl.isEmpty() && !ff.getTargetDevices().isEmpty()) {
                    runFliplist = activity::showFliplistDialog;
                }
            }
        }
        extendedFileFunctions.add(new BaseActivity.PopupCallback(
                getContext(), R.string.flip_file, R.drawable.ic_diskstack,
                runFliplist, true, false));

        connectButton(R.id.ib_handle_opened_files, R.id.ll_handle_opened_files,
                extendedFileFunctions,
                state);
        showSubMenu(inflater, findViewById(ID_BN_DETACH), state, detach);
        List<BaseActivity.PopupCallback> backgroundFunctions = new LinkedList<>();
        if (emu.getSoundFunctions() != null) {
            if (emu.getSoundFunctions() != null) {
                Runnable r = this::sendToBackground;
                boolean ask = (ContextCompat.checkSelfPermission(
                        getActivity(),
                        android.Manifest.permission.READ_EXTERNAL_STORAGE)
                        == PackageManager.PERMISSION_DENIED)
                        && (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU);
                Runnable denied = () -> Toast.makeText(getActivity(),
                        R.string.notifications_denied, Toast.LENGTH_SHORT).show();
                backgroundFunctions.add(new BaseActivity.PopupCallback(
                        getContext(), R.string.audio_background, R.drawable.ic_headset,
                        () -> getActivity().executeWithAccess(ask,
                                new String[]{Manifest.permission.POST_NOTIFICATIONS},
                                null, r, denied, r),
                        false, false));
            }

        }
        if (!getActivity().isTv()) {
            backgroundFunctions.add(new BaseActivity.PopupCallback(getContext(),
                    R.string.moviemode, R.drawable.ic_moviemode,
                    () -> getActivity().playInMovieMode(), false, false));
        }
        connectButton(R.id.bn_pause, () -> containerview.showUiElement(PAUSE, false));
        connectButton(R.id.ib_background, R.id.ll_background,
                backgroundFunctions,
                state);

        if (conf.isKeyboardUsed()) {
            if (activity.getKeyboardVisibility() == View.VISIBLE) {
                connectButton(R.id.bn_hide_keyboard, () -> setKeyboardVisible(false));
            } else {
                connectButton(R.id.bn_show_keyboard, () -> setKeyboardVisible(true));
            }
        }
        if (activity.isTv() && !(softkeyIds.isEmpty() && uiSoftkeys.isEmpty())) {
            connectButton(R.id.bn_show_softkeys, activity::showSoftkeyDialog);
        }
        Emulation.SpeedUpFunctions suf = emu.getSpeedUpFunctions();
        if (suf != null) {
            if (suf.hasMaximumSpeedMode()) {
                if (suf.getMaximumSpeedMode()) {
                    connectButton(R.id.bn_normal_speed,
                            () -> suf.setMaximumSpeedMode(false));
                } else {
                    connectButton(R.id.bn_warp, () -> suf.setMaximumSpeedMode(true));
                }
            }
            List<Integer> availableSpeeds = suf.getAvailableNormalSpeeds();
            if (availableSpeeds != null && !availableSpeeds.isEmpty()
                    && !(availableSpeeds.size() == 1
                    && availableSpeeds.get(0) == Emulation.SpeedUpFunctions.NORMAL_SPEED)) {
                ((Button) findViewById(R.id.bn_speed_settings)).setText(
                        String.format(Locale.getDefault(), "%d %%", suf.getNormalSpeed()));
                connectButton(R.id.bn_speed_settings, () -> containerview.showUiElement(
                        DialogsView.UiElement.SPEED_SELECTION));

            }
        }
        if (emu.getDualMonitorFunctions() != null) {
            connectButton(R.id.bn_switch_monitor, activity::switchMonitor);
        }
        if (findViewById(R.id.bn_quit) != null) {
            connectButton(R.id.bn_quit, activity::quitEmulation);
        }
        findViewById(R.id.sv_content).measure(View.MeasureSpec.UNSPECIFIED,
                View.MeasureSpec.UNSPECIFIED);
        int height = findViewById(R.id.sv_content).getMeasuredHeight();
        DisplayMetrics dm = new DisplayMetrics();
        getActivity().getWindowManager().getDefaultDisplay().getMetrics(dm);
        if (height > dm.heightPixels) {
            height = ViewGroup.LayoutParams.WRAP_CONTENT;
        }
        LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams) getLayoutParams();
        setLayoutParams(lp);
        getActivity().getWindowManager().getDefaultDisplay().getMetrics(dm);
        int bottomMargin = getContext().getResources()
                .getDimensionPixelSize(R.dimen.touchable_minimal_height);
        int maxheight = getActivity().isInLandscape()
                ? Math.min(dm.widthPixels, dm.heightPixels)
                : Math.max(dm.widthPixels, dm.heightPixels);
        if ((height < 0) || (height + bottomMargin > maxheight)) {
            bottomMargin = 0;
        }
        lp.height = height + bottomMargin;
        setLayoutParams(lp);
    }
    void dismiss() {
        MenuState state = getActivity().getViewModel().getMenuState();
        if (state != null) {
            Serializable sr = state.getPauseReason();
            if (sr != null) {
                getActivity().removePauseReason(sr);
                state.setPauseReason(null);
            }
        }
        EmulationActivity a = getActivity();
        a.getViewModel().setMenuState(null);
        setVisibility(View.GONE);
        for (Fragment f: a.getSupportFragmentManager().getFragments()) {
            if (f instanceof DialogsView.BaseFragment) {
                return;
            }
        }
        a.findViewById(R.id.dialogsview).setVisibility(View.GONE);
    }

    private void init() {
        if (getVisibility() == View.VISIBLE) {
            populate();
        }
    }
    // CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
    public MainMenuView(final Context context,
                        final @Nullable AttributeSet attrs) {
        super(context, attrs);
        init();
    }
    // CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
    public MainMenuView(final Context context,
                        final @Nullable AttributeSet attrs,
                        final int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    @Override
    public void setVisibility(final int visibility) {
        if (visibility == View.VISIBLE) {
            populate();
            MenuState state = getActivity().getViewModel().getMenuState();
            if (state != null && state.getPauseReason() == null) {
                state.setPauseReason(getActivity().addPauseReason());
            }

        }
        super.setVisibility(visibility);
    }

    static class MenuState implements Serializable {
        private Serializable mPauseReason = null;
        private final HashSet<Integer> mOpenSections = new HashSet<>();
        void setOpen(final int id, final boolean value) {
            if (value) {
                mOpenSections.add(id);
            } else {
                mOpenSections.remove(id);
            }
        }
        boolean isOpen(final int id) {
            return mOpenSections.contains(id);
        }
        void setPauseReason(final Serializable reason) {
            mPauseReason = reason;
        }
        Serializable getPauseReason() {
            return mPauseReason;
        }
    }
}
